/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.data;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.feature.*;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.track.TrackType;
import org.broad.igv.util.AsciiLineReader;

//~--- JDK imports ------------------------------------------------------------

import java.io.*;

import java.util.*;

/**
 * Class description
 *
 *
 * @version    Enter version here..., 09/01/09
 * @author     Enter your name here...
 */
public class SegmentFileParser {

    int sampleColumn = 0;
    int chrColumn = 1;
    int startColumn = 2;
    int endColumn = 3;
    int snpCountColumn = 4;    // Default value
    int dataColumn = 5;        // Default value
    ResourceLocator locator;

    /**
     * Constructs ...
     *
     *
     * @param locator
     */
    public SegmentFileParser(ResourceLocator locator) {
        this.locator = locator;
    }

    /**
     * Method description
     *
     *
     * @param dataset
     */
    public void loadSegmentTracks(SegmentedAsciiDataSet dataset) {
        loadSegments(dataset);

    }

    /**
     * Return a map of trackId -> segment datasource
     * @param file
     * @return
     */
    private void loadSegments(SegmentedAsciiDataSet dataset) {
        AsciiLineReader reader = null;
        try
        {
            reader = ParsingUtils.openAsciiReader(locator);

            // Parse comments, if any
            String nextLine = reader.readLine();
            while (nextLine.startsWith("#") || (nextLine.trim().length() == 0))
            {
                if (nextLine.length() > 0)
                {
                    parseComment(nextLine, dataset);
                }
                nextLine = reader.readLine();
            }

            // Read column headings
            String[] headings = reader.readLine().split("\t");

            // The data value is always the last column
            dataColumn = headings.length - 1;

            // We assume the snp count is next to last, but not 0-3
            snpCountColumn = dataColumn - 1;
            if (snpCountColumn < 4)
            {
                snpCountColumn = -1;
            }

            String[] tokens = new String[headings.length];

            while ((nextLine = reader.readLine()) != null && (nextLine.trim().length() > 0))
            {

                int nTokens = ParsingUtils.split(nextLine, tokens, '\t');
                if (nTokens > 4)
                {
                    String chr = ParsingUtils.convertChrString(tokens[chrColumn].trim());
                    int start = Integer.parseInt(tokens[startColumn].trim());
                    int end = Integer.parseInt(tokens[endColumn].trim());
                    String trackId = new String(tokens[sampleColumn].trim());

                    int snpCount = 0;
                    if (snpCountColumn > 0)
                    {
                        try
                        {
                            snpCount = Integer.parseInt(tokens[snpCountColumn]);
                        } catch (NumberFormatException numberFormatException)
                        {

                        // This is an expected condition, nothing needs done.
                        }
                    }


                    float value = Float.NaN;
                    try
                    {
                        value = Float.parseFloat(tokens[dataColumn]);

                    } catch (NumberFormatException numberFormatException)
                    {

                    // This is an expected condition, nothing needs done.
                    }

                    dataset.addSegment(trackId, chr, start, end, value, snpCount);
                }
            }

            reader.close();
        } catch (IOException e)
        {

            // TODO -- log properly
            e.printStackTrace();
        } finally
        {
            if (reader == null)
            {
                reader.close();
            }
        }
    }


    /**
     * Note:  This is an exact copy of the method in GCTDatasetParser.  Refactor to merge these
     * two parsers, or share a common base class.
     * @param comment
     * @param dataset
     */
    private void parseComment(String comment, SegmentedAsciiDataSet dataset) {

        String tmp = comment.substring(1, comment.length());
        String[] tokens = tmp.split("=");
        if (tokens.length != 2)
        {
            return;
        }
        String key = tokens[0].trim().toLowerCase();
        if (key.equals("type"))
        {

            try
            {
                dataset.setTrackType(TrackType.valueOf(tokens[1].trim().toUpperCase()));
            } catch (Exception exception)
            {

            // Ignore

            }
        }
    }
}
