/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.data;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.preprocess.old.BinaryReader;
import cern.colt.list.DoubleArrayList;

import cern.jet.stat.quantile.DoubleQuantileFinder;
import cern.jet.stat.quantile.QuantileFinderFactory;

import org.broad.igv.feature.LocusScore;

import org.broad.igv.preprocess.*;
import org.broad.igv.track.WindowFunction;

//~--- JDK imports ------------------------------------------------------------

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 *
 * @author jrobinso
 */
public class ProcessingUtils {

    protected static int[] findBoundaries(List<LocusScore> scores) {
        // Create list of all boundaries
        int[] boundaries = new int[2 * scores.size() + 1];
        for (int i = 0; i < scores.size(); i++) {
            LocusScore score = scores.get(i);
            boundaries[2 * i] = score.getStart();
            boundaries[2 * i + 1] = score.getEnd();
        }
        Arrays.sort(boundaries);
        // Remove duplicates
        IntArrayList boundaryList = new IntArrayList(boundaries.length);
        int lastPos = -1;
        for (int i = 0; i < boundaries.length; i++) {
            if (boundaries[i] != lastPos) {
                lastPos = boundaries[i];
                boundaryList.add(lastPos);
            }
        }
        boundaries = boundaryList.toArray();
        return boundaries;
    }

    private static boolean nullDataCheck(float[] data) {

        // Check for null or missing data.  Float.NaN is interpreted
        // as a placeholder for "no data.
        boolean noData = true;
        if ((data != null) && (data.length > 0)) {
            for (int i = 0; i < data.length; i++) {
                if (!Float.isNaN(data[i])) {
                    noData = false;
                    break;
                }
            }
        }


        return noData;
    }

    /**
     * Constructs ...
     *
     */
    public ProcessingUtils() {
    }

    private static float computeQuantile(float[] data, double quantile) {
        DoubleArrayList al = new DoubleArrayList(2);
        al.add(quantile);
        DoubleQuantileFinder qf = QuantileFinderFactory.newDoubleQuantileFinder(true, data.length,
                0.001, 0.001, al.size(), null);
        for (int i = 0; i < data.length; i++) {
            if (isValidData(data[i])) {
                qf.add(data[i]);
            }
        }
        if (qf.size() > 0) {

            return (float) qf.quantileElements(al).get(0);
        } else {
            return Float.NaN;
        }
    }

    private static float computeMin(float[] data) {
        float min = Float.MAX_VALUE;
        for (int i = 0; i < data.length; i++) {
            if (!Float.isNaN(data[i])) {
                min = Math.min(data[i], min);
            }
        }
        return min;
    }

    private static float computeMax(float[] data) {
        float max = -Float.MAX_VALUE;
        for (int i = 0; i < data.length; i++) {
            if (!Float.isNaN(data[i])) {
                max = Math.max(data[i], max);
            }
        }
        return max;
    }

    private static float computeMean(float[] data) {
        float sum = 0.0f;
        int nPts = 0;
        for (int i = 0; i < data.length; i++) {
            if (!Float.isNaN(data[i])) {
                sum += data[i];
                nPts++;
            }
        }
        return (nPts == 0 ? Float.NaN : sum / nPts);
    }

    /**
     * Method description
     *
     *
     * @param data
     * @param function
     *
     * @return
     */
    public static float computeStat(float[] data, WindowFunction function) {

        if (nullDataCheck(data)) {
            return Float.NaN;
        } else {
            switch (function) {
                case mean:
                    return computeMean(data);
                case median:
                    return computeQuantile(data, 0.5);
                case min:
                    return computeMin(data);
                case max:
                    return computeMax(data);
                case percentile10:
                    return computeQuantile(data, 0.1);
                case percentile90:
                    return computeQuantile(data, 0.9);
                case percentile98:
                    return computeQuantile(data, 0.98);
                case count:
                    return data.length;

            }
            return Float.NaN;
        }

    }

    /**
     * Compute some statistics for the input data
     *   TODO move to a utility class
     *   TODO take statis desired as input
     * @param data
     * @return
     */
    public static DataStatistics computeStats(float[] data) {

        if (nullDataCheck(data)) {
            return DataStatistics.nullDataStat;
        } else {

            DataStatistics stats = new DataStatistics();

            // Check for array of NaNs
            DoubleArrayList al = new DoubleArrayList(2);
            al.add(0.1);
            al.add(0.5);
            al.add(0.9);
            al.add(0.98);
            DoubleQuantileFinder qf = QuantileFinderFactory.newDoubleQuantileFinder(true,
                    data.length, 0.001, 0.001, al.size(), null);


            double minStat = Double.MAX_VALUE;
            double maxStat = -Double.MAX_VALUE;
            double meanStat = 0;
            for (int i = 0; i < data.length; i++) {
                if (isValidData(data[i])) {
                    qf.add(data[i]);
                    minStat = Math.min(minStat, data[i]);
                    maxStat = Math.max(maxStat, data[i]);
                    meanStat += data[i];
                }
            }

            if (qf.size() > 0) {

                meanStat /= qf.size();
                double err2 = 0;
                for (int i = 0; i < data.length; i++) {
                    if (isValidData(data[i])) {
                        err2 += (data[i] - meanStat) * (data[i] - meanStat);
                    }
                }
                double stdDev = Math.sqrt(err2 / qf.size());

                DoubleArrayList quantiles2 = qf.quantileElements(al);
                stats.setMin(minStat);
                stats.setMax(maxStat);
                stats.setMean(meanStat);
                stats.setPercentile10(quantiles2.get(0));
                stats.setMedian(quantiles2.get(1));
                stats.setPercentile90(quantiles2.get(2));
                stats.setPercentile98(quantiles2.get(3));
                stats.setStdDev(stdDev);
            }

            return stats;

        }
    }

    /**
     *
     * @param data
     * @return
     */
    public double computeMedian(float[] data) {

        DataStatistics stats = new DataStatistics();


        if (data.length > 0) {
            DoubleArrayList al = new DoubleArrayList();
            al.add(0.5);
            DoubleQuantileFinder qf = QuantileFinderFactory.newDoubleQuantileFinder(true,
                    data.length, 0.001, 0.001, al.size(), null);


            for (int i = 0; i < data.length; i++) {
                if (isValidData(data[i])) {
                    qf.add(data[i]);

                }
            }

            if (qf.size() > 0) {

                DoubleArrayList quantiles2 = qf.quantileElements(al);
                return quantiles2.get(0);
            }
        }
        return 0;
    }

    private static boolean isValidData(float data) {

        // if (chipFormat) {
        // return !Float.isNaN(data) && (data >= 0);
        // } else {
        return !Float.isNaN(data);

    // }
    }

    /**
     * Dump the first N data values from the given file as a java formatted
     *  array.  Used for debugging
     *  and testing.
     *
     * @param file
     * @param type
     * @param nPoints
     */
    public static void dump(File file, Class type, int nPoints) {

        BinaryReader reader = new BinaryReader();

        System.out.print("{");
        if (type == Long.class) {
            long[] values = reader.readLongs(file);
            for (int i = 0; i < nPoints; i++) {
                System.out.print(values[i]);
                if (i < nPoints - 1) {
                    System.out.print(", ");
                }
            }
            System.out.println("}");
        } else if (type == Float.class) {
            float[] values = reader.readFloats(file);
            for (int i = 0; i < nPoints; i++) {
                System.out.print(values[i]);
                if (i < nPoints - 1) {
                    System.out.print(", ");
                }
            }
            System.out.println("}");
        }
    }

    /**
     * Count the number of points between start and end location.  Used
     * for debugging.
     *  and testing.
     *
     * @param file
     * @param startLocation
     * @param endLocation
     */
    public static void countPoints(File file, long startLocation, long endLocation) {

        BinaryReader reader = new BinaryReader();
        long[] values = reader.readLongs(file);
        int count = 0;
        for (int i = 0; i < values.length; i++) {
            if (values[i] >= endLocation) {
                break;
            }
            if (values[i] >= startLocation) {
                count++;
            }
        }
        System.out.println("Count= " + count);
    }

    /**
     * Method description
     *
     *
     * @param file
     * @param startLocation
     * @param endLocation
     */
    public static void dumpPoints(File file, long startLocation, long endLocation) {

        PrintWriter pw = null;
        try {
            pw = new PrintWriter(new FileWriter("feature_dump_gt.txt"));
            BinaryReader reader = new BinaryReader();
            long[] values = reader.readLongs(file);
            for (int i = 0; i < values.length; i++) {
                if (values[i] >= endLocation) {
                    break;
                }
                if (values[i] >= startLocation) {
                    pw.println(values[i]);

                }
            }
        } catch (IOException ex) {
            ex.printStackTrace();
        } finally {
            pw.close();

        }
    }

    /**
     * Method description
     *
     *
     * @param args
     *
     * @return
     */
    public static LinkedHashMap<String, String> parseArgs(String[] args) {
        LinkedHashMap<String, String> argMap = new LinkedHashMap();

        for (int i = 0; i < args.length; i++) {
            System.out.println(args[i]);

            String key = args[i];

            if (args[i].startsWith("-")) {
                if (i < args.length - 1) {
                    i++;

                    if (args[i].startsWith("-")) {
                        argMap.put(key, "");
                        i--;
                    } else {
                        argMap.put(key, args[i]);
                    }
                } else {
                    argMap.put(key, "");
                }
            } else {
                argMap.put(key, key);
            }
        }

        return argMap;
    }

    static class Interval {

        int start;
        int end;
        List<LocusScore> scores = new ArrayList(3);

        /**
         * Constructs ...
         *
         *
         * @param start
         * @param end
         */
        public Interval(int start, int end) {
            this.start = start;
            this.end = end;
        }
    }

    /**
     * 
     *
     * @param scores
     * @param wf
     *
     * @return
     */

    public static List<LocusScore> segregateScores(List<LocusScore> scores, WindowFunction wf) {


        int[] boundaries = findBoundaries(scores);


        LinkedHashMap<Integer, Interval> intervals = new LinkedHashMap(2 * scores.size());
        for (int i = 1; i < boundaries.length; i++) {
            int start = boundaries[i - 1];
            int end = boundaries[i];
            if (end > start) {
                intervals.put(start, new Interval(start, end));
            }
        }

        Map<Integer, Integer> reverseMap = new HashMap();
        for (int i = 0; i < boundaries.length; i++) {
            reverseMap.put(boundaries[i], i);
        }

        // Assign scores to intervals.  A score can be split across muliple intervals
        for (LocusScore score : scores) {
            int start = score.getStart();
            int end = score.getEnd();
            int idxStart = reverseMap.get(start);
            int idxEnd = reverseMap.containsKey(end) ? reverseMap.get(end) : boundaries.length - 1;

            int pos = -1;
            for (int i = idxStart; i < idxEnd; i++) {
                // test for duplicated boundary.  Why is this neccessary?
                if (boundaries[i] != pos) {
                    pos = boundaries[i];
                    Interval interval = intervals.get(pos);
                    if (interval.end <= score.getEnd()) {
                        LocusScore tmp = score.copy();
                        tmp.setStart(interval.start);
                        tmp.setEnd(interval.end);
                        interval.scores.add(tmp);
                    }
                }
            }
        }

        // Create list of segregated scores. 
        List<LocusScore> segScores = new ArrayList(scores.size());
        for (Interval interval : intervals.values()) {
            if (interval.scores.size() > 0) {
                if (interval.scores.size() == 1) {
                    segScores.add(interval.scores.get(0));
                } else {
                    segScores.add(new CompositeScore(interval.scores, wf));
                }
            }
        }
        return segScores;

    }

    /**
     * Method description
     *
     *
     * @param args
     */
    public static void main(String[] args) {
        long startLocation = 4568550;
        long endLocation = 4611950;
        dumpPoints(new File("test/data/es/chrX.snplocation.bin"), startLocation, endLocation);

    // dump(new File("test/data/es/chrX.snplocation.bin"), Long.class, 20);
    // dump(new File("test/data/es/chrX.ES.K27.bin"), Float.class, 20);
    }
}
