/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.data;

import java.io.File;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.broad.igv.feature.Genome;
import org.broad.igv.feature.GenomeManager;
import org.broad.igv.util.ObjectCache;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.track.TrackProperties;
import org.broad.igv.track.TrackType;

/**
 *
 * @author jrobinso
 */
public class IGVDataset implements Dataset {

    private String name;

    private TrackType type = TrackType.OTHER;

    private boolean logNormalized;

    /**
     * Genome id for the location information in this dataset.  This parameter
     * is needed when the dataset is preprocessed
     */
    private String genomeId;

    private String[] dataHeadings;

    private Map<String, ChromosomeSummary> chromosomeSummaries = new LinkedHashMap();

    private GenomeSummaryData genomeSummary;

    private IGVDatasetParser parser;

    private ObjectCache<String, ChromosomeData> chromsomeDataCache = new ObjectCache(30);
    
    private float dataMin;
    
    private float dataMax;
    
    TrackProperties trackProperties = new TrackProperties();

    public IGVDataset(String genomeId, ResourceLocator locator) {
        this.genomeId = genomeId;

        //GenomeManager.getInstance().findGenomeAndLoad(genomeId);
        Genome genome = GenomeManager.getInstance().getGenome(genomeId);
        if (genome == null) {
            throw new RuntimeException("Unknown genome: " + genomeId);
        }
        genomeSummary = new GenomeSummaryData(genome);

        // Turn off sampling in the summary object, sampling will be done as
        // the file is parsed.
        genomeSummary.setSamplingFreq(1);

        parser = new IGVDatasetParser(locator, genomeId);

        List<ChromosomeSummary> summaries = parser.scan(this);

        for (ChromosomeSummary summary : summaries) {
            chromosomeSummaries.put(summary.getName(), summary);
        }
    }

    public IGVDataset(String genomeId, File file) {
        this(genomeId, new ResourceLocator(file.getAbsolutePath()));
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getName() {
        return name;
    }

    public void setTrackType(TrackType type) {
        this.type = type;
    }

    public TrackType getType() {
        return type;
    }

    public void setGenome(String genomeId) {
        this.genomeId = genomeId;
    }

    public String getGenome() {
        return genomeId;
    }

    public String[] getChromosomes() {
        return chromosomeSummaries.keySet().toArray(new String[]{});
    }

    public void setDataHeadings(String[] dataHeadings) {
        this.dataHeadings = dataHeadings;
    }

    public String[] getDataHeadings() {
        return dataHeadings;
    }

    public int[] getStartLocations(String chr) {
        ChromosomeData cd = getChromosomeData(chr);
        return cd == null ? null : cd.getStartLocations();
    }

    public int[] getEndLocations(String chr) {
        ChromosomeData cd = getChromosomeData(chr);
        return cd == null ? null : cd.getEndLocations();
    }

    public String[] getFeatureNames(String chr) {
        return null;
    }

    public int getWindowSpan() {
        return 1;
    }

    public float[] getData(String heading, String chr) {
        ChromosomeData cd = getChromosomeData(chr);
        return cd == null ? null : cd.getData(heading);
    }

    /**
     * Get the data for all samples (tracks) for the given chromosome.
     * 
     * This method is synchronized to insure that the data for a chromosome
     * is only loaded once.
     * 
     * @param chr
     * @return
     */
    private synchronized ChromosomeData getChromosomeData(String chr) {
        ChromosomeData cd = chromsomeDataCache.get(chr);
        if (cd == null) {
            ChromosomeSummary sum = chromosomeSummaries.get(chr);
            if (sum == null) {
                //todo -- throw exception
                return null;
            }
            //synchronized (sum) {

            cd = parser.loadChromosomeData(sum, dataHeadings);
            chromsomeDataCache.put(chr, cd);
        //}

        }
        return cd;
    }

    public GenomeSummaryData getGenomeSummary() {
        return genomeSummary;
    }

    public void setLogNormalized(boolean logNormalized) {
        this.logNormalized = logNormalized;
    }

    public boolean isLogNormalized() {
        return logNormalized;
    }

    public

    float getDataMin() {
        return dataMin;
    }

    public

    float getDataMax() {
        return dataMax;
    }

    public void setDataMin(float dataMin) {
        this.dataMin = dataMin;
    }

    public void setDataMax(float dataMax) {
        this.dataMax = dataMax;
    }

    public TrackProperties getTrackProperties() {
        return trackProperties;
    }
}
