/*
* The Broad Institute
* SOFTWARE COPYRIGHT NOTICE AGREEMENT
* This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
* of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
* (the "License"); you may not use this file except in compliance with
*  the License.  You may obtain a copy of the License at
*
*    http://www.opensource.org/licenses/gpl-2.0.php
*
* This software is supplied without any warranty or guaranteed support
* whatsoever. Neither the Broad Institute nor MIT can be responsible for its
* use, misuse, or functionality.
*/


package org.broad.igv.data;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.renderer.BarChartRenderer;
import org.broad.igv.renderer.HeatmapRenderer;
import org.broad.igv.renderer.ScatterplotRenderer;
import org.broad.igv.track.TrackProperties;
import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.IGVMainFrame;

import org.broad.igv.util.ColorUtilities;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;

import javax.swing.JOptionPane;
import org.apache.log4j.Logger;
import org.broad.igv.renderer.LineplotRenderer;


/**
 *
 * @author jrobinso
 */
public class DatasetParserUtils {

    private static Logger log = Logger.getLogger(DatasetParserUtils.class);
    /**
     * graphType         bar|points           # default is bar
     * yLineMark         real-value           # default is 0.0
     * yLineOnOff        on|off               # default is off
     * windowingFunction maximum|mean|minimum # default is maximum
     * smoothingWindow   off|[2-16]           # default is off
     *
     * @param nextLine
     * @param trackProperties
     *
     * @throws NumberFormatException
     */
    public static boolean parseTrackLine(String nextLine, TrackProperties trackProperties)
            throws NumberFormatException {

        boolean foundProperties = false;
        try
        {
            // track type=wiggle_0 name="CSF +" description="CSF +" visibility=full autoScale=off viewLimits=-50:50
            String[] tokens = nextLine.split(" ");
            for (String pair : tokens)
            {
                String[] kv = pair.split("=");
                if (kv.length == 2)
                {
                    foundProperties = true;
                    String key = kv[0].toLowerCase().trim();
                    String value = kv[1].toLowerCase().trim();

                    if (key.equals("name"))
                    {
                        trackProperties.setName(value);
                    }
                    else if (key.equals("description"))
                    {
                        trackProperties.setDescription(value);
                    }
                    else if (key.equals("color"))
                    {
                        Color color = ColorUtilities.convertRGBStringToColor(value);
                        trackProperties.setColor(color);
                    }
                    else if (key.equals("altcolor"))
                    {
                        Color color = ColorUtilities.convertRGBStringToColor(value);
                        trackProperties.setAltColor(color);
                    }
                    else if (key.equals("midcolor"))
                    {
                        Color color = ColorUtilities.convertRGBStringToColor(value);
                        trackProperties.setMidColor(color);
                    }
                    else if (key.equals("autoscale"))
                    {
                        boolean autoscale = value.equals("on");
                        trackProperties.setAutoScale(autoscale);
                    }
                    else if (key.equals("maxheightpixels"))
                    {

                        // Ignore the min and max
                        String[] maxDefMin = value.split(":");
                        trackProperties.setHeight(Integer.parseInt(maxDefMin[1].trim()));
                        trackProperties.setMinHeight(Integer.parseInt(maxDefMin[2].trim()));

                    }
                    else if (key.equals("graphtype"))
                    {
                        if (value.equals("bar"))
                        {
                            trackProperties.setRendererClass(BarChartRenderer.class);
                        }
                        else if (value.equals("points"))
                        {
                            trackProperties.setRendererClass(ScatterplotRenderer.class);
                        }
                        else if (value.equals("line"))
                        {
                            trackProperties.setRendererClass(LineplotRenderer.class);
                        }
                        else if (value.equals("heatmap"))
                        {
                            trackProperties.setRendererClass(HeatmapRenderer.class);
                        }
                    }
                    else if (key.toLowerCase().equals("viewlimits"))
                    {
                        String[] limits = value.split(":");
                        if (limits.length == 2)
                        {
                            float min = Float.parseFloat(limits[0].trim());
                            float max = Float.parseFloat(limits[1].trim());
                            trackProperties.setMinValue(min);
                            trackProperties.setMaxValue(max);
                            float mid = min;
                            if (min * max < 0)
                            {
                                mid = 0;
                            }
                            trackProperties.setMidValue(mid);

                        }
                    }
                    else if (key.equals("ylinemark"))
                    {
                        try {
                            float midValue = Float.parseFloat(value);
                            trackProperties.setMidValue(midValue);
                        } catch(NumberFormatException e) {
                            log.error("Number format exception in track line (ylinemark): " + nextLine );
                        }
                    }
                    else if (key.equals("ylineonoff"))
                    {
                        trackProperties.setDrawMidValue(value.equals("on"));
                    }
                    else if (key.equals("windowingfunction"))
                    {
                        if (value.equals("maximum"))
                        {
                            trackProperties.setWindowingFunction(WindowFunction.max);
                        }
                        else if (value.equals("minimum"))
                        {
                            trackProperties.setWindowingFunction(WindowFunction.min);

                        }
                        else if (value.equals("mean"))
                        {
                            trackProperties.setWindowingFunction(WindowFunction.mean);

                        }
                        else if (value.equals("median"))
                        {
                            trackProperties.setWindowingFunction(WindowFunction.median);

                        }
                        else if (value.equals("percentile10"))
                        {
                            trackProperties.setWindowingFunction(WindowFunction.percentile10);

                        }
                        else if (value.equals("percentile90"))
                        {
                            trackProperties.setWindowingFunction(WindowFunction.percentile90);

                        }
                    }

                }
            }

        }
        catch (Exception exception)
        {
            JOptionPane.showMessageDialog(IGVMainFrame.getInstance(),
                                          "Error parsing track line: " + nextLine + " ("
                                          + exception.getMessage() + ")");
        }
        return foundProperties;

    }


}
