/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/



/*
* To change this template, choose Tools | Templates
* and open the template in the editor.
 */
package org.broad.igv.data;

//~--- non-JDK imports --------------------------------------------------------

import org.apache.log4j.Logger;

import org.broad.igv.IGVConstants;
import org.broad.igv.feature.Chromosome;
import org.broad.igv.feature.Genome;
import org.broad.igv.track.TrackType;
import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.IGVModel;

//~--- JDK imports ------------------------------------------------------------

import java.util.List;

/**
 *
 * @author jrobinso
 */
public class DatasetDataSource extends AbstractDataSource {

    private Logger log = Logger.getLogger(DatasetDataSource.class);

    String trackId;
    Dataset dataset;
    GenomeSummaryData genomeSummaryData;

    /**
     * Constructs ...
     *
     *
     * @param genome
     * @param trackId
     * @param dataset
     */
    public DatasetDataSource(Genome genome, String trackId, Dataset dataset) {
        this.trackId = trackId;
        this.dataset = dataset;

        // TODO -- remove this "instanceof" hack
        if (dataset instanceof IGVDataset)
        {
            genomeSummaryData = ((IGVDataset) dataset).getGenomeSummary();
        }
        else
        {
            genomeSummaryData = new GenomeSummaryData(genome);
            for (Chromosome chr : genome.getChromosomes())
            {
                int[] startLocations = dataset.getStartLocations(chr.getName());
                {
                    if (!chr.getName().equals(IGVConstants.CHR_ALL) && (startLocations != null)
                            && (startLocations.length > 0))
                    {
                        genomeSummaryData.addLocations(chr.getName(),
                                                       dataset.getStartLocations(chr.getName()));
                        genomeSummaryData.addData(trackId, chr.getName(),
                                                  dataset.getData(trackId, chr.getName()));
                    }
                }
            }
        }
    }

    @Override
    protected int getNumZoomLevels(String chr) {
        return 0;
    }

    @Override
    protected DataTile getRawData(String chr, int startLocation, int endLocation) {
        int[] startLocs = null;
        int[] endLocs = null;
        float[] data = null;
        String [] features = null;

        if (chr.equals(IGVConstants.CHR_ALL))
        {
            startLocs = genomeSummaryData.getLocations();
            endLocs = startLocs;
            data = genomeSummaryData.getData(trackId);
        }
        else
        {
            startLocs = dataset.getStartLocations(chr);
            endLocs = dataset.getEndLocations(chr);
            data = dataset.getData(trackId, chr);
            features = dataset.getFeatureNames(chr);
        }
        if (startLocs == null)
        {
            return null;
        }

        assert(startLocs.length == endLocs.length);
        assert(startLocs.length == data.length);

        int maxIndex = startLocs.length - 1;

        // Until we can guarantee sorted values just use the index limits
        int startIdx = 0;    // Math.min(maxIndex,  DataUtils.getIndexBefore(startLocs, startLocation));
        int endIdx = maxIndex;    // Math.min(maxIndex, DataUtils.getIndexBefore(startLocs, endLocation) + 1);
        int[] tmpStart = startLocs;
        int[] tmpEnd = endLocs;
        float[] tmpData = data;
        String [] tmpFeatures = features;
        int nPts = endIdx - startIdx + 1;

        if ((tmpStart == null) || (tmpData == null) || (nPts <= 0))
        {
            return null;
        }

        startLocs = new int[nPts];
        endLocs = ((tmpEnd == null) ? null : new int[nPts]);
        data = new float[nPts];
        features = (tmpFeatures == null ? null : new String[nPts]);

        try
        {
            System.arraycopy(tmpStart, startIdx, startLocs, 0, nPts);
            System.arraycopy(tmpData, startIdx, data, 0, nPts);
            if (endLocs != null)
            {
                System.arraycopy(tmpEnd, startIdx, endLocs, 0, nPts);
            }
            if(features != null) {
                System.arraycopy(tmpFeatures, startIdx, features, 0, nPts);
            }

        }
        catch (Exception exception)
        {
            log.error("Exception getting raw data tile", exception);
            return null;
        }



        return new DataTile(startLocs, endLocs, data, features);
    }


    /**
     * Method description
     *
     *
     * @return
     */
    public TrackType getTrackType() {
        try
        {
            return dataset.getType();
        }
        catch (Exception exception)
        {
            return TrackType.OTHER;
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public boolean isLogNormalized() {
        return dataset.isLogNormalized();
    }

    /**
     * Method description
     *
     *
     * @param chr
     *
     * @return
     */
    public double getDataMax(String chr) {
        return dataset.getDataMax();
    }

    /**
     * Method description
     *
     *
     * @param chr
     *
     * @return
     */
    public double getDataMin(String chr) {
        return dataset.getDataMin();
    }

    /**
     * Method description
     *
     *
     * @param zoom
     * @param chr
     *
     * @return
     */
    public double getMedian(int zoom, String chr) {
        return 1.0;
    }



}
