/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/



/*
* To change this template, choose Tools | Templates
* and open the template in the editor.
 */
package org.broad.igv.data;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;

/**
 *
 * @author jrobinso
 */
public class DataUtils {

    public static int getIndexBefore(int [] values, int x) {
        return getIndexBefore(values, x, 0, values.length);
    }
    
    public static int getIndexBefore(int[] values, int x, int leftBound, int rightBound) {
        int idx = (leftBound + rightBound) / 2;

        if ((idx == 0) || (idx == values.length - 1))
        {
            return idx;
        }
        if (values[idx] == x)
        {
            return idx;
        }

        if (values[idx] < x)
        {
            if (values[idx + 1] >= x)
            {
                return idx;
            }
            else
            {
                leftBound = idx;
                return getIndexBefore(values, x, leftBound, rightBound);
            }
        }
        else
        {    // values[idx] > x
            if (values[idx - 1] <= x)
            {
                return idx - 1;
            }
            else
            {
                rightBound = idx;
                return getIndexBefore(values, x, leftBound, rightBound);
            }
        }
    }



    /**
     * Estimate the number of rows in an ascii data file.  Estimate is based on
     * the first 100 lines, and assumes the line length is approximately
     * constant.
     *
     * @param textFile
     * @return
     */
    public static AsciiFileMetrics estimateFileMetrics(String textFile) {
        int estRowCount = 0;
        try
        {

            BufferedReader reader = null;

            File file = new File(textFile);

            reader = new BufferedReader(new FileReader(file));
            String nextLine = reader.readLine();

            double lineCount = 0;
            double nChars = 0;
            while ((nextLine = reader.readLine()) != null && (lineCount < 100))
            {
                nChars += nextLine.length();
                lineCount++;
            }

            int columnCount = nextLine.split("\t").length;

            double charsPerLine = ((lineCount > 0) ? nChars / lineCount : 0);
            estRowCount = (int) (file.length() / charsPerLine);
            return new AsciiFileMetrics(estRowCount, columnCount, charsPerLine);

        }
        catch (FileNotFoundException ex)
        {
            ex.printStackTrace();

        }
        catch (IOException ex)
        {
            ex.printStackTrace();

        }
        return null;
    }

    /**
     * Method description
     *
     *
     * @param fileMetrics
     *
     * @return
     */
    public static int estimatePreprocessingTime(AsciiFileMetrics fileMetrics) {

        // =(H25/100)*0.0036*G25 +8

        return 8 + (int) ((0.0036 * fileMetrics.getEstRowCount() * fileMetrics.getColumnCount())
                          / 100);
    }

    /**
     * This class has some useful metrics for optimizing reading of large ascii files
     * @author jrobinso
     */
    public static class AsciiFileMetrics {

        private int estRowCount;

        private int columnCount;

        private double estBytesPerLine;

        /**
         * Constructs ...
         *
         *
         * @param estRowCount
         * @param columnCount
         * @param estBytesPerLine
         */
        public AsciiFileMetrics(int estRowCount, int columnCount, double estBytesPerLine) {
            this.estRowCount = estRowCount;
            this.columnCount = columnCount;
            this.estBytesPerLine = estBytesPerLine;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public double getEstBytesPerLine() {
            return estBytesPerLine;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public int getEstRowCount() {
            return estRowCount;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public int getColumnCount() {
            return columnCount;
        }
    }
}
