/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.data;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.feature.FeatureUtils;
import org.broad.igv.feature.GeneManager;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.util.ObjectCache;
import org.broad.igv.preprocess.old.FeatureBin;
import org.broad.igv.preprocess.old.FeatureBinCalculator;
import org.broad.igv.track.TrackType;
import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.IGVModel;

//~--- JDK imports ------------------------------------------------------------

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 *
 * @author jrobinso
 */
public abstract class AbstractDataSource implements DataSource {

    private static Logger log = Logger.getLogger(AbstractDataSource.class);

    // DataManager dataManager;
    boolean cacheSummaryTiles = true;
    WindowFunction windowFunction = WindowFunction.mean;
    ObjectCache<String, SummaryTile> summaryTileCache = new ObjectCache();

    // abstract DataManager getDataManager();
    /**
     * Return the number of precomputed zoom levels for the given chromosome.
     * @param chr
     * @return
     */
    abstract protected int getNumZoomLevels(String chr);

    // abstract protected TrackType getTrackType();
    /**
     * Return "raw" (i.e. not summarized) data for the specified interval.
     * @param chr
     * @param startLocation
     * @param endLocation
     * @return
     */
    abstract protected DataTile getRawData(String chr, int startLocation, int endLocation);

    /**
     * Return the precomputed summary tiles for the given locus and zoom level.  If
     * there are non return null.
     *
     * @param chr
     * @param startLocation
     * @param endLocation
     * @param zoom
     * @return
     */
    List<SummaryTile> getPrecomputedSummaryTiles(String chr, int startLocation, int endLocation,
            int zoom) {
        return null;
    }

    /**
     * Return a maximum value for the chromosome.  This is used to autoscale XY
     * plots.
     *
     * @param zoom
     * @param chr
     * @param stat
     * @return
     */

    // abstract double getDataMax(String chr, WindowFunction windowFunction);
    /**
     * Return a minimum value for the chromosome.  This is used to autoscale XY
     * plots.
     * @param chr
     * @return
     */
    // abstract double getDataMin(String chr, WindowFunction windowFunction);
    /**
     * Return the median value for the chromosome, zoom level, and window function.
     * This is used to nromalize some types of data in heatmaps.
     * @param zoom
     * @param chr
     * @param stat
     * @return
     */
    abstract public double getMedian(int zoom, String chr);

    /**
     * Method description
     *
     *
     * @param chr
     *
     * @return
     */
    public int getChrLength(String chr) {
        return IGVModel.getInstance().getViewContext().getChromosomeLength();
    }

    /**
     * Refresh the underlying data. Default implementation does nothing, subclasses
     * can override
     *
     * @param timestamp
     */
    public void refreshData(long timestamp) {

        // ignore --
    }

    /**
     * Return the longest feature in the dataset for the given chromosome.  This
     * is needed when computing summary data for a region.
     *
     * TODO - This default implementaiton is crude and should be overriden by subclasses.
     *
     * @param chr
     * @return
     */
    public int getLongestFeature(String chr) {
        return (getTrackType() == TrackType.GENE_EXPRESSION) ? 2000000 : 1000;
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param startLocation
     * @param endLocation
     * @param zoom
     * @param windowFunction
     *
     * @return
     */
    public List<LocusScore> getSummaryScoresForRange(String chr, int startLocation,
            int endLocation, int zoom) {


        List<SummaryTile> tiles = getSummaryTilesForRange(chr, startLocation, endLocation, zoom);


        // If copy # optionally join segments with identical values

        List<LocusScore> summaryScores = new ArrayList(tiles.size() * 700);

        // TODO -- refactor this for efficiency.  Features can be returned from multiple
        // tiles if they overlap the tiles.  However we only want to count them once.
        // HashSet<String> locusKeys = new HashSet();
        for (SummaryTile tile : tiles) {
            summaryScores.addAll(tile.getScores());

        // for (LocusScore score : tile.getScores()) {
        // String key = score.getStart() + "_" + score.getEnd();
        // if (!locusKeys.contains(key)) {
        // summaryScores.add(score);
        // locusKeys.add(key);
        // }
        // }
        }
        FeatureUtils.sortFeatureList(summaryScores);
        return summaryScores;
    }

    private List<SummaryTile> getSummaryTilesForRange(String chr, int startLocation,
            int endLocation, int zoom) {
        assert endLocation >= startLocation;

        int startTile = 0;
        int endTile = 0;
        if (zoom < getNumZoomLevels(chr)) {
            return getPrecomputedSummaryTiles(chr, startLocation, endLocation, zoom);
        } else {
            int chrLength = getChrLength(chr);
            endLocation = Math.min(endLocation, chrLength);

            // Get some extra tiles to be sure we get long features that start
            // before startLocaion
            int longestFeature = getLongestFeature(chr);

            int adjustedStart = Math.max(0, startLocation - longestFeature);
            int adjustedEnd = Math.min(chrLength, endLocation);

            int z = Math.min(8, zoom);
            int nTiles = (int) Math.pow(2, z);
            double tileWidth = ((double) chrLength) / nTiles;


            startTile = (int) (adjustedStart / tileWidth);
            endTile = (int) (Math.min(chrLength, adjustedEnd) / tileWidth) + 1;
            List<SummaryTile> tiles = new ArrayList(nTiles);
            for (int t = startTile; t <= endTile; t++) {
                int tileStart = (int) (t * tileWidth);
                int tileEnd = Math.min(chrLength, (int) ((t + 1) * tileWidth));
                SummaryTile summaryTile = computeSummaryTile(chr, t, tileStart, tileEnd, zoom);
                if (summaryTile != null) {
                    tiles.add(summaryTile);
                }
            }

            return tiles;
        }
    }

    private SummaryTile computeSummaryTile(String chr, int tileNumber, int startLocation,
            int endLocation, int zoom) {
        String key = chr + "_" + zoom + "_" + tileNumber + getWindowFunction();
        SummaryTile tile = summaryTileCache.get(key);

        if (tile == null) {

            // TODO -- binWidth should be passed in
            double binWidth = IGVModel.getInstance().getViewContext().getScale();

            // Get a window   prior to start to be sure we get all features.
            String genomeId = IGVModel.getInstance().getViewContext().getGenomeId();

            GeneManager gm = GeneManager.getGeneManager(genomeId);
            int longestGene = (gm == null) ? 1000000 : gm.getLongestGeneLength(chr);
            int adjustedStart = Math.max(startLocation - longestGene, 0);
            DataTile rawTile = getRawData(chr, adjustedStart, endLocation);

            if ((rawTile != null) && !rawTile.isEmpty()) {

                tile = new SummaryTile(tileNumber, startLocation);

                int[] starts = rawTile.getStartLocations();
                int[] ends = rawTile.getEndLocations();
                float[] values = rawTile.getValues();
                String[] features = rawTile.getFeatureNames();

                List<LocusScore> scoresToSegregate = new ArrayList(starts.length);
                List<LocusScore> scoresToBin = new ArrayList(starts.length);
                for (int i = 0; i < starts.length; i++) {
                    int start = starts[i];
                    int end = (ends == null) ? start + 1 : ends[i];
                    String featureName = features == null ? null : features[i];

                    SummaryScore ss = featureName == null ? new SummaryScore(start, end, values[i],
                            1.0f) : new NamedSummaryScore(start, end, values[i], featureName);
                    double pixelWidth = (end - start) / binWidth;
                    if (pixelWidth < 1) {
                        scoresToBin.add(ss);
                    } else {
                        scoresToSegregate.add(ss);
                    }
                }

                List<LocusScore> locusScores = ProcessingUtils.segregateScores(scoresToSegregate,
                        windowFunction);

                List<FeatureBin> bins = computeBins(startLocation, endLocation, scoresToBin);
                for (FeatureBin fBin : bins) {

                    int binStart = fBin.getStart();
                    int binEnd = (int) (binStart + binWidth);
                    int counts = fBin.getFeatureCount();
                    float binValue = 0;

                    if ((windowFunction == null) || (windowFunction == WindowFunction.count)) {
                        binValue = counts;
                    } else {
                        float[] tmp = fBin.getFeatureScores();

                        if (tmp == null) {
                            binValue = Float.NaN;
                        } else {
                            binValue = ProcessingUtils.computeStat(tmp, windowFunction);
                        }
                    }

                    locusScores.add(new SummaryScore(binStart, binEnd, binValue, 1.0f));
                }
                tile.addAllScores(locusScores);

                if (this.cacheSummaryTiles) {
                    synchronized (summaryTileCache) {
                        summaryTileCache.put(key, tile);
                    }
                }
            }
        }
        return tile;
    }

    private static boolean hasOverlappingScores(int[] starts, int[] ends) {

        // assert starts.length = ends.length;
        if (starts.length == 1) {
            return false;
        }
        for (int i = 0; i < starts.length - 1; i++) {
            if (ends[i] > starts[i + 1]) {
                return true;
            }
        }
        return false;
    }

    private static List<FeatureBin> computeBins(int startLocation, int endLocation,
            List<LocusScore> features) {
        double binSize = IGVModel.getInstance().getViewContext().getScale();
        int nBins = (int) Math.ceil((endLocation - startLocation) / binSize);
        double correctedBinSize = ((double) (endLocation - startLocation)) / nBins;
        return (new FeatureBinCalculator()).computeFeatureBins(features, nBins, correctedBinSize,
                startLocation, endLocation);
    }

    private List<LocusScore> aggregateScores(List<SummaryTile> tiles, int leftBoundary,
            int rightBoundary) {
        List<LocusScore> joinedScores = new ArrayList(tiles.size() * 700);
        LocusScore previousScore = null;

        // Extend the first and last

        for (SummaryTile tile : tiles) {
            for (LocusScore score : tile.getScores()) {
                if (!Float.isNaN(score.getScore())) {

                    if (previousScore == null) {

                        // This first score.  Extend its start to the left boundary.
                        previousScore = new SummaryScore(score);
                        score.setStart(leftBoundary);
                        joinedScores.add(previousScore);
                    } else {
                        if (score.getScore() == previousScore.getScore()) {

                            // score values are identical, stretch the current score
                            // to encompass the entire interval
                            previousScore.setEnd(score.getEnd());
                            previousScore.setConfidence(1);

                        } else {

                            SummaryScore newScore = new SummaryScore(score);

                            // score value has changed. Adjust end of previous
                            // score (if any), and start of this score to meet 1/2
                            // way
                            int delta = newScore.getStart() - previousScore.getEnd();
                            previousScore.setEnd(previousScore.getEnd() + delta / 2);
                            newScore.setStart(previousScore.getEnd());

                            joinedScores.add(newScore);

                            previousScore = newScore;
                        }

                    }
                }

            }
        }

        // Finally extend the last score the the right boundary
        if (!joinedScores.isEmpty()) {
            joinedScores.get(joinedScores.size() - 1).setEnd(rightBoundary);
        }


        return joinedScores;
    }

    /**
     * Return true if the data has been log normalized.
     * @return
     */
    public boolean isLogNormalized() {
        return true;
    }

    /**
     * Method description
     *
     *
     * @param statType
     */
    public void setWindowFunction(WindowFunction statType) {
        this.windowFunction = statType;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public WindowFunction getWindowFunction() {
        return windowFunction;
    }
    // TODO -- get window functions dynamically from data
    static List<WindowFunction> wfs = new ArrayList();
    

    static {
        wfs.add(WindowFunction.percentile10);
        wfs.add(WindowFunction.median);
        wfs.add(WindowFunction.mean);
        wfs.add(WindowFunction.percentile90);
        wfs.add(WindowFunction.max);

    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Collection<WindowFunction> getAvailableWindowFunctions() {
        return wfs;
    }
}
