/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv;

import org.broad.igv.IGVConstants;
import java.io.File;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.RollingFileAppender;

/**
 *
 * @author eflakes
 */
public class Utilities {

    static StringBuffer buffer = new StringBuffer();

    static private RollingFileAppender appender;

    final private static String separator =
            System.getProperties().getProperty("file.separator");

    public static void addRollingAppenderToRootLogger() {
        addRollingAppenderToLogger(Logger.getRootLogger());
    }

    public static void addRollingAppenderToLogger(Logger logger) {

        if (appender == null) {

            try {

                PatternLayout layout = new PatternLayout();
                layout.setConversionPattern("%p [%d{ISO8601}] [%F:%L]  %m%n");

                // Create a log file that is ready to have text appended to it
                appender = new RollingFileAppender();
                appender.setName("IGV_ROLLING_APPENDER");
                appender.setFile(getLogFilePath());
                appender.setThreshold(Level.ALL);
                appender.setMaxFileSize("1000KB");
                appender.setMaxBackupIndex(1);
                appender.setLayout(layout);
                appender.setAppend(true);
                appender.activateOptions();
                logger.addAppender(appender);

            // Show what logging is turned on
                /*
            logger.info(" ");
            logger.info(" ");
            logger.debug("DEBUG logging enabled!");
            logger.warn("WARNING logging enabled!");
            logger.error("ERROR logging enabled!");
            logger.info("INFORMATION logging enabled!");
            logger.fatal("FATAL logging enabled!");
             */
            } catch (Exception e) {
                e.printStackTrace();
                throw new RuntimeException(e);
            }
        }
    }

    static public String getLogFilePath() {

        // Build the log file path
        StringBuffer logFilePath = new StringBuffer();
        logFilePath.append(IGVConstants.DEFAULT_IGV_DIRECTORY);
        logFilePath.append(separator);
        logFilePath.append("igv.log");
        
        // Added for Linux which does notr automatically create the log file
        File logFile = null;
        try {
            logFile = new File(logFilePath.toString().trim());
            if(!logFile.getParentFile().exists()) {
                logFile.getParentFile().mkdir();
            }
            if(!logFile.exists()) {
                logFile.createNewFile();
            }
        } 
        catch(Exception e) {
            System.out.println("Error creating log file: "+logFile.getAbsolutePath());
            e.printStackTrace();
        }

        return logFilePath.toString().trim();
    }

    static public String getRelativePath(File baseFile, String targetFile) {
        return getRelativePath(baseFile, new File(targetFile));
    }

    static public String getRelativePath(File baseFile, File targetFile) {

        if (!baseFile.isDirectory()) {
            throw new RuntimeException(baseFile.getAbsolutePath() +
                    " is not a directory!");
        }

        boolean isTargetAFile = targetFile.isFile();
        File target = targetFile;
        if (isTargetAFile) {
            target = targetFile.getParentFile();
        } else {
            if (!targetFile.isDirectory()) {
                throw new RuntimeException(targetFile.getAbsolutePath() +
                        " is not a directory or file!");
            }
        }

        String path = "???";
        boolean isUsingRelativePath = true;
        boolean isMoreTargetPath = true;
        String delimeter = separator.equals("\\") ? "\\\\" : separator;
        String path1 = baseFile.getAbsolutePath();
        String path2 = target.getAbsolutePath();

        if (!path1.equals(path2)) {

            String[] basePath = path1.split(delimeter);
            String[] targetPath = path2.split(delimeter);

            // Compare paths
            int i = 0;
            for (; i < basePath.length; i++) {

                if (i == basePath.length) { // No more path
                    break; // No more path string to process
                }

                if (i == targetPath.length) { // No more path
                    int levelsBack = basePath.length - i;
                    path = completeTheRelativePath(levelsBack, targetPath, i);
                    break; // No more path string to process
                }

                // Roots must match or we use the absolute path
                if (i == 0) {
                    if (!basePath[i].equals(targetPath[i])) {
                        path = target.getAbsolutePath();
                        isUsingRelativePath = false;
                        break;// use default absolute path
                    } else {
                        path = "." + separator; // set a default path for same drive
                    }
                    continue;
                }

                // Nodes in path still match so keep looking
                if (basePath[i].equals(targetPath[i])) {
                    continue;
                }

                // Finally reached a point where path nodes don't match
                int levelsBack = basePath.length - i;
                path = completeTheRelativePath(levelsBack, targetPath, i);
                isMoreTargetPath = false; // Target now completely processed
                break;
            }

            if (isUsingRelativePath) {
                if (isMoreTargetPath) {
                    path += completeTheRelativePath(0, targetPath, i);
                }
            }
        } else {
            path = "." + separator;
        }

        if (isTargetAFile) {
            if (isUsingRelativePath) {
                path += targetFile.getName();
            } else {
                path = new File(path, targetFile.getName()).getAbsolutePath();
            }
        }
        return path;
    }

    /**
     * Return the canonical path of the file.  If the canonical path cannot
     * be computed return the absolute path.
     * 
     * @param arg
     */
    static public String getCanonicalPath(File file) {
        try {
            return file.getCanonicalPath();
        } catch (Exception e) {
            return file.getAbsolutePath();
        }
    }

    static public String getPlatformIndependentPath(String path) {
        return path.replace('\\', '/');
    }

    static private String completeTheRelativePath(int levelsBack,
            String[] targetPath, int startingIndexForTarget) {

        // Complete the relative path
        buffer.delete(0, buffer.length());
        for (int j = 0; j < levelsBack; j++) {
            buffer.append("..");
            buffer.append(separator);
        }

        for (int k = startingIndexForTarget; k < targetPath.length; k++) {
            buffer.append(targetPath[k]);
            buffer.append(separator);
        }
        return buffer.toString();
    }

    /**
     * Delete a directory and all subdirectories
     * @param dir
     * @return
     */
    public static boolean deleteDir(File dir) {
        if (dir.isDirectory()) {
            String[] children = dir.list();
            for (int i = 0; i < children.length; i++) {
                boolean success = deleteDir(new File(dir, children[i]));
                if (!success) {
                    return false;
                }
            }
        }

        // The directory is now empty so delete it
        return dir.delete();
    }

    static public void main(String arg[]) {

        File baseFile = new File("C:\\");
        File targetFile = new File("C:\\");
        String path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + ".\\");

        baseFile = new File("C:\\");
        targetFile = new File("C:\\bdata\\foo\\last");
        path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + ".\\bdata\\foo\\last");

        baseFile = new File("C:\\BaseDirectory\\base");
        targetFile = new File("C:\\");
        path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + "..\\..\\");

        baseFile = new File("C:\\BaseDirectory\\base");
        targetFile = new File("C:\\BaseDirectory\\base");
        path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + ".\\");

        baseFile = new File("C:\\BaseDirectory\\base\\node\\last");
        targetFile = new File("C:\\bdata\\foo\\last");
        path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + "..\\..\\..\\..\\bdata\\foo\\last");

        baseFile = new File("C:\\BaseDirectory\\base");
        targetFile = new File("C:\\bdata\\foo\\last");
        path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + "..\\..\\bdata\\foo\\last");

        baseFile = new File("C:\\BaseDirectory\\base");
        targetFile = new File("C:\\BaseDirectory\\base\\node\\last");
        path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + ".\\node\\last");

        baseFile = new File("C:\\BaseDirectory\\base\\node\\last");
        targetFile = new File("C:\\bdata\\foo\\last\\test_file.txt");
        path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + "..\\..\\..\\..\\bdata\\foo\\last\\test_file.txt");

        baseFile = new File("C:\\bdata\\foo\\last");
        targetFile = new File("C:\\bdata\\foo\\last\\test_file.txt");
        path = Utilities.getRelativePath(baseFile, targetFile);
        System.out.println("\nBase Path: " + baseFile.getAbsolutePath());
        System.out.println("Target Path: " + targetFile.getAbsolutePath());
        System.out.println("\tCalculated new target Path: " + path);
        System.out.println("\tExpected new target Path: " + ".\\test_file.txt");
    }
}
