/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0ƒ
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.feature.Mutation;
import org.broad.igv.ui.util.PropertyManager;
import org.broad.igv.renderer.ColorScale;
import org.broad.igv.renderer.ColorScaleFactory;
import org.broad.igv.renderer.ContinuousColorScale;
import org.broad.igv.track.TrackType;
import org.broad.igv.ui.AboutDialog;

import static org.broad.igv.IGVConstants.*;

//static
import static org.broad.igv.ui.util.UIUtilities.getcommaSeparatedRGBString;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.Rectangle;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.util.*;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import org.broad.igv.maf.MAFManager;

/**
 *
 * @author eflakes
 */
public class PreferenceManager implements PropertyManager {

    /** Field description */
    public static final String DEFAULT_CHART_TRACK_HEIGHT = "40";
    private static Logger log = Logger.getLogger(PreferenceManager.class);
    /** Field description */
    public static final String TRACK_PROPERTIES_FILE = "TRACK_PROPERTIES_FILE";
    private static final String DEFAULT_DATA_SERVER_URL;

    static {
        Properties properties = new Properties();
        try {
            properties.load(AboutDialog.class.getResourceAsStream("/resources/about.properties"));
        } catch (IOException e) {
            log.error("*** Error retrieving the Master Resource URL! ***", e);
        }

        String defaultMasterUrl =
                properties.getProperty("master-resource-url",
                "http://www.broadinstitute.org/igv/resources/dataServerRegistry.txt");

        if (defaultMasterUrl.startsWith("@")) {
            DEFAULT_DATA_SERVER_URL =
                    "http://www.broadinstitute.org/igv/resources/dataServerRegistry.txt";
        } else {
            DEFAULT_DATA_SERVER_URL = defaultMasterUrl;
        }
    }
    private static final int INITIAL_TRACK_HEIGHT = 15;
    private final String SYSTEM_DEFAULT_FOR_DIRECT_DRAW = System.getProperty("sun.java2d.noddraw");
    /**
     * Property keys below
     */
    public static final String TRACK_PROPERTIES_URL_KEY = "TRACK_PROPERTIES_URL";
    /** Field description */
    public static final String COLOR_SCALE_KEY = "COLOR_SCALE_";
    /** Field description */
    final public static String FRAME_BOUNDS_KEY = "IGVMainFrame.Bounds";
    // Key for attributes the app is allowed to display
    /** Field description */
    final public static String DISPLAYABLE_ATTRIBUTES_KEY = "IGVMainFrame.Attributes.Displayable";
    // Session Key Constants
    /** Field description */
    final public static String DRAW_EXON_NUMBERS_KEY = "DRAW_EXON_NUMBERS";
    /** Field description */
    final public static String SESSION_NAME_KEY = "IGVMainFrame.Session.Name";
    /** Field description */
    final public static String SESSION_GENOME_KEY = "IGVMainFrame.Session.Genome";
    /** Field description */
    final public static String SESSION_TRACK_FILES_KEY = "IGVMainFrame.Session.Track.Files";
    /** Field description */
    final public static String SESSION_ATTRIBUTE_FILE_KEY = "IGVMainFrame.Session.Attribute.File";
    /** Field description */
    final public static String RECENT_SESSION_KEY = "IGVMainFrame.Session.recent.sessions";
    // View preference key constants
    /** Field description */
    final public static String TRACK_HEIGHT_KEY = "IGVMainFrame.track.height";
    /** Field description */
    final public static String CHART_TRACK_HEIGHT_KEY = "IGVMainFrame.chart.track.height";
    /** Field description */
    final public static String SHOW_MISSING_DATA_KEY = "IGVMainFrame.track.show.missing.data";
    /** Field description */
    final public static String SHOW_ATTRIBUTE_VIEWS_KEY = "IGVMainFrame.track.show.attribute.views";
    /** Field description */
    final public static String SHOW_ROI_KEY = "IGVMainFrame.track.show.regionsOfInterest";
    /** Field description */
    final public static String SHOW_SINGLE_TRACK_PANE_KEY = "IGVMainFrame.single.track.pane";
    /** Field description */
    final public static String FEATURE_FISHBONE_STYLE = "IGVMainFrame.feature.fishbone.style";
    /** Field description */
    final public static String GENOMES_SEQUENCE_URL = "IGVMainFrame.genome.sequence.dir";
    /** Field description */
    final public static String JOIN_ADJACENT_SEGMENTS_KEY = "IGVMainFrame.join.adjacent.segments";
    /** Field description */
    final public static String SHOW_REGION_TOOL_KEY = "IGVMainFrame.show.region.tool";
    /** Field description */
    final public static String LAST_EXPORTED_REGION_DIRECTORY = "LAST_EXPORTED_REGION_DIRECTORY";
    /** Field description */
    final static public String LAST_TRACK_DIRECTORY = "LAST_TRACK_DIRECTORY";
    /** Field description */
    final static public String LAST_ATTRIBUTE_FILE_DIRECTORY = "LAST_ATTRIBUTE_FILE_DIRECTORY";
    /** Field description */
    final static public String LAST_SNAPSHOT_DIRECTORY = "LAST_SNAPSHOT_DIRECTORY";
    /** Field description */
    final static public String LAST_GENOME_IMPORT_DIRECTORY = "LAST_GENOME_IMPORT_DIRECTORY";
    /** Field description */
    final static public String LAST_SESSION_DIRECTORY = "LAST_SESSION_DIRECTORY";
    /** Field description */
    final static public String DEFAULT_GENOME_KEY = "DEFAULT_GENOME_KEY";
    /** Field description */
    final static public String LAST_CHROMOSOME_VIEWED_KEY = "LAST_CHROMOSOME_VIEWED_KEY";
    /** Field description */
    final public static String MUTATION_INDEL_COLOR_KEY = "MUTATION_INDEL_COLOR_KEY";
    /** Field description */
    final public static String MUTATION_MISSENSE_COLOR_KEY = "MUTATION_MISSENSE_COLOR_KEY";
    /** Field description */
    final public static String MUTATION_NONSENSE_COLOR_KEY = "MUTATION_NONSENSE_COLOR_KEY";
    /** Field description */
    final public static String MUTATION_SPLICE_SITE_COLOR_KEY = "MUTATION_SPLICE_SITE_COLOR_KEY";
    /** Field description */
    final public static String MUTATION_SYNONYMOUS_COLOR_KEY = "MUTATION_SYNONYMOUS_COLOR_KEY";
    /** Field description */
    final public static String MUTATION_TARGETED_REGION_COLOR_KEY =
            "MUTATION_TARGETED_REGION_COLOR_KEY";
    /** Field description */
    final public static String MUTATION_UNKNOWN_COLOR_KEY = "MUTATION_UNKNOWN_COLOR_KEY";
    /** Field description */
    final public static String ENABLE_IMAGE_CACHING_KEY = "ENABLE_IMAGE_CACHING_KEY";
    /** Field description */
    final public static String DIRECT_DRAW_DISABLED_KEY = "DIRECT_DRAW_DISABLED_KEY";
    /** Field description */
    final public static String OVERLAY_TRACKS_KEY = "OVERLAY_TRACKS_KEY";
    /** Field description */
    final public static String DISPLAY_OVERLAY_TRACKS_KEY = "DISPLAY_OVERLAY_TRACKS_KEY";
    /** Field description */
    final public static String OVERLAY_ATTRIBUTE_KEY = "OVERLAY_ATTRIBUTE_KEY";
    /** Field description */
    final public static String COLOR_OVERLAY_KEY = "COVER_OVERLAY_KEY";
    final public static String ENABLE_LINKED_SORTING = "ENABLE_LINKED_SORTING";
    /** Field description */
    final public static String TRACK_ATTRIBUTE_NAME_KEY = "TRACK_ATTRIBUTE_NAME_KEY";
    /** Field description */
    final public static String DATA_SERVER_URL_KEY = "MASTER_RESOURCE_FILE_KEY";
    /** Field description */
    final public static String CHECKED_RESOURCES_KEY = "CHECKED_RESOURCES_KEY";
    /** Field description */
    final public static String DEFINE_GENOME_INPUT_DIRECTORY_KEY =
            "DEFINE_GENOME_INPUT_DIRECTORY_KEY";
    /** Field description */
    final public static String LAST_CYTOBAND_DIRECTORY_KEY = "LAST_CYTOBAND_DIRECTORY_KEY";
    /** Field description */
    final public static String LAST_REFFLAT_DIRECTORY_KEY = "LAST_REFFLAT_DIRECTORY_KEY";
    /** Field description */
    final public static String LAST_FASTA_DIRECTORY_KEY = "LAST_FASTA_DIRECTORY_KEY";
    /** Field description */
    final public static String LAST_SEQUENCE_DIRECTORY_KEY = "LAST_SEQUENCE_DIRECTORY_KEY";
    final public static String MAF_SPECIES_KEY = "MAF_SPECIES_KEY";
    final public static String PROBE_MAPPING_KEY = "PROBE_MAPPING_KEY";
    final static private Preferences IGV_PREFERENCE_MANAGER =
            Preferences.userNodeForPackage(PreferenceManager.class);
    /**
     * Cache of preference values.  Profiling reveals that Preferences.get()
     * is taking huge amounts of time.  There are hundereds of thousands of
     * calls to this to get the track height,  this is possibly a bad design
     * decision, however caching the preference values solves the peformance
     * problem for now.
     */
    public Map<String, String> stringCache = new HashMap(1000);
    /**
     * See comment for stringCache.
     */
    private Map<String, Boolean> booleanCache = new Hashtable();
    private Map<String, Object> objectCache = new Hashtable();
    private Map<TrackType, ColorScale> colorScaleCache = new Hashtable();
    /**
     * Instance method for the Preference Manager.
     */
    private static PreferenceManager instance;

    private PreferenceManager() {
    }

    /**
     * Method description
     *
     *
     * @return
     */
    synchronized public static PreferenceManager getInstance() {

        if (instance == null) {
            instance = new PreferenceManager();
        }
        return instance;
    }

    /**
     * Return the preference as a boolean value.
     *
     * @param key
     * @param defaultValue
     * @return
     */
    public boolean getBooleanPreference(String key, boolean defaultValue) {
        Boolean boolValue = booleanCache.get(key);
        if (boolValue == null) {

            String value = get(key, null);
            if ((value != null) && !value.trim().equals("")) {

                // If value ok
                boolValue = new Boolean(value);
            } else {
                boolValue = new Boolean(defaultValue);
            }
            booleanCache.put(key, boolValue);
        }
        return boolValue.booleanValue();
    }

    public boolean isMapProbesToGenes() {
        return getBooleanPreference(PROBE_MAPPING_KEY, false);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isJoinAdjacentSegments() {
        return getBooleanPreference(JOIN_ADJACENT_SEGMENTS_KEY, true);
    }

    /**
     * Update any cached values with the new key/value pair
     * @param key
     * @param value
     */
    private void updateCaches(String key, String value) {
        stringCache.put(key, value);
        if (booleanCache.containsKey(key)) {
            booleanCache.put(key, new Boolean(value));
        }
    }

    private void clearCaches() {
        stringCache.clear();
        booleanCache.clear();
        objectCache.clear();
    }

    /**
     * Method description
     *
     *
     * @param key
     * @param value
     */
    public void put(String key, String value) {
        IGV_PREFERENCE_MANAGER.put(key, value);
        updateCaches(key, value);
    }

    /**
     * Method description
     *
     *
     * @param updatedPrefs
     */
    public void putAll(Map<String, String> updatedPrefs) {
        for (Map.Entry<String, String> entry : updatedPrefs.entrySet()) {
            put(entry.getKey(), entry.getValue());
        }
        clearCaches();

    }

    /**
     * Method description
     *
     *
     * @param key
     * @param defaultString
     *
     * @return
     */
    public String get(String key, String defaultString) {
        String rtn = stringCache.get(key);
        if (rtn == null) {
            rtn = IGV_PREFERENCE_MANAGER.get(key, defaultString);
            stringCache.put(key, rtn);
        }
        return rtn;
    }

    /**
     * Method description
     *
     *
     * @param key
     */
    public void remove(String key) {
        IGV_PREFERENCE_MANAGER.remove(key);
        stringCache.remove(key);
        booleanCache.remove(key);
        objectCache.clear();
    }

    /**
     * Method description
     *
     *
     * @throws Exception
     */
    public void clear() {
        try {
            IGV_PREFERENCE_MANAGER.clear();

        } catch (BackingStoreException ex) {
            log.error("Error clearing preferences", ex);
        }
        stringCache.clear();
        booleanCache.clear();
        objectCache.clear();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getGenomeListURL() {
        return get(PreferenceManager.GENOMES_SEQUENCE_URL,
                IGVConstants.DEFAULT_SERVER_GENOME_ARCHIVE_LIST);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean getOverlayTracks() {
        return getBooleanPreference(OVERLAY_TRACKS_KEY, true);
    }

    /**
     * Method description
     *
     *
     * @param value
     */
    public void setOverlayTracks(boolean value) {
        this.put(OVERLAY_TRACKS_KEY, String.valueOf(value));
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean getDrawExonNumbers() {
        return getBooleanPreference(DRAW_EXON_NUMBERS_KEY, false);
    }

    public List<String> getMafSpecies() {
        String tmp = get(MAF_SPECIES_KEY, null);

        String[] species = null;
        if (tmp == null) {
            species = MAFManager.species;
        } else {
            species = tmp.split(":");
        }
        return Arrays.asList(species);
    }

    public void setMafSpecies(List<String> species) {
        StringBuffer buf = new StringBuffer(species.size() * 7);
        Iterator<String> iter = species.iterator();
        while (iter.hasNext()) {
            buf.append(iter.next());
            if (iter.hasNext()) {
                buf.append(":");
            }
        }
        put(MAF_SPECIES_KEY, buf.toString());

    }

    /**
     * Method description
     *
     *
     * @param value
     */
    public void setDrawExonNumbers(boolean value) {
        this.put(DRAW_EXON_NUMBERS_KEY, String.valueOf(value));
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean getColorOverlay() {
        return getBooleanPreference(COLOR_OVERLAY_KEY, false);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean getDiplayOverlayTracks() {
        return getBooleanPreference(DISPLAY_OVERLAY_TRACKS_KEY, true);
    }

    /**
     * Method description
     *
     *
     * @param value
     */
    public void setDisplayOverlayTracks(boolean value) {
        this.remove(DISPLAY_OVERLAY_TRACKS_KEY);
        this.put(DISPLAY_OVERLAY_TRACKS_KEY, String.valueOf(value));
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean getDirectDrawDisabled() {

        boolean systemDefault = (SYSTEM_DEFAULT_FOR_DIRECT_DRAW == null)
                ? true : Boolean.getBoolean(SYSTEM_DEFAULT_FOR_DIRECT_DRAW);

        return getBooleanPreference(DIRECT_DRAW_DISABLED_KEY, systemDefault);
    }

    /**
     * Method description
     *
     *
     * @param value
     */
    public void setDirectDrawDisabled(boolean value) {

        if (value) {
            System.setProperty("sun.java2d.noddraw", "true");
        } else if (System.getProperty("sun.java2d.noddraw") != null) {
            System.clearProperty("sun.java2d.noddraw");
        }

        put(DIRECT_DRAW_DISABLED_KEY, String.valueOf(value));
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getOverlayAttribute() {
        return get(OVERLAY_ATTRIBUTE_KEY, "LINKING_ID");
    }

    /**
     * Method description
     *
     *
     * @param value
     */
    public void setOverlayAttribute(String value) {
        put(OVERLAY_ATTRIBUTE_KEY, value);
    }

    public boolean isLinkedSortingEnabled() {
        return this.getBooleanPreference(ENABLE_LINKED_SORTING, false);
    }

    public void setLinkedSortingEnabled(boolean value) {
        put(ENABLE_LINKED_SORTING, Boolean.toString(value));
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getCheckedResources() {
        return IGVPreferences.get(CHECKED_RESOURCES_KEY, null);
    }

    /**
     * Method description
     *
     *
     * @param value
     */
    public void setCheckedResources(String value) {
        IGVPreferences.put(CHECKED_RESOURCES_KEY, value);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean getShowSingleTrackPane() {
        return getBooleanPreference(SHOW_SINGLE_TRACK_PANE_KEY, false);
    }

    /**
     * Method description
     *
     *
     * @param bounds
     */
    public void setApplicationFrameBounds(Rectangle bounds) {

        StringBuffer buffer = new StringBuffer();
        buffer.append(bounds.x);
        buffer.append(",");
        buffer.append(bounds.y);
        buffer.append(",");
        buffer.append(bounds.width);
        buffer.append(",");
        buffer.append(bounds.height);
        put(FRAME_BOUNDS_KEY, buffer.toString());
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Rectangle getApplicationFrameBounds() {

        Rectangle bounds = null;

        // Set the application's previous location and size
        String applicationBounds = IGV_PREFERENCE_MANAGER.get(FRAME_BOUNDS_KEY, null);

        if (applicationBounds != null) {
            String[] values = applicationBounds.split(",");
            int x = Integer.parseInt(values[0]);
            int y = Integer.parseInt(values[1]);
            int width = Integer.parseInt(values[2]);
            int height = Integer.parseInt(values[3]);
            bounds = new Rectangle(x, y, width, height);
        }
        return bounds;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastExportedRegionDirectory(File directory) {

        put(LAST_EXPORTED_REGION_DIRECTORY, directory.getAbsolutePath());
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastExportedRegionDirectory() {

        File exportedRegionDirectory = null;

        String lastFilePath = get(LAST_EXPORTED_REGION_DIRECTORY, null);

        if (lastFilePath != null) {

            // Create the exported region directory
            exportedRegionDirectory = new File(lastFilePath);
        }

        return exportedRegionDirectory;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastSnapshotDirectory(File directory) {

        put(LAST_SNAPSHOT_DIRECTORY, directory.getAbsolutePath());
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastSnapshotDirectory() {

        File snapshotDirectory = null;

        String lastFilePath = get(LAST_SNAPSHOT_DIRECTORY, null);

        if (lastFilePath != null) {

            // Create the snapshot directory
            snapshotDirectory = new File(lastFilePath);
        }

        return snapshotDirectory;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setDefineGenomeInputDirectory(File directory) {

        put(DEFINE_GENOME_INPUT_DIRECTORY_KEY, directory.getAbsolutePath());
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getDefineGenomeInputDirectory() {

        File directory = null;

        String lastFilePath = get(DEFINE_GENOME_INPUT_DIRECTORY_KEY, IGVConstants.getDefaultUserDirectory());

        if (lastFilePath != null) {
            directory = new File(lastFilePath);
        }

        return directory;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastGenomeImportDirectory(File directory) {

        put(LAST_GENOME_IMPORT_DIRECTORY, directory.getAbsolutePath());
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastGenomeImportDirectory() {

        File genomeImportDirectory = null;

        String lastFilePath = get(LAST_GENOME_IMPORT_DIRECTORY, IGVConstants.getDefaultUserDirectory());

        if (lastFilePath != null) {
            genomeImportDirectory = new File(lastFilePath);
        }

        return genomeImportDirectory;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastCytobandDirectory(File directory) {

        if (directory != null) {
            put(LAST_CYTOBAND_DIRECTORY_KEY, directory.getAbsolutePath());
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastCytobandDirectory() {

        File directory = null;

        String lastFilePath = get(LAST_CYTOBAND_DIRECTORY_KEY, IGVConstants.getDefaultUserDirectory());

        if (lastFilePath != null) {
            directory = new File(lastFilePath);
        }

        return directory;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastRefFlatDirectory(File directory) {

        if (directory != null) {
            put(LAST_REFFLAT_DIRECTORY_KEY, directory.getAbsolutePath());
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastRefFlatDirectory() {

        File directory = null;

        String lastFilePath = get(LAST_REFFLAT_DIRECTORY_KEY, IGVConstants.getDefaultUserDirectory());

        if (lastFilePath != null) {
            directory = new File(lastFilePath);
        }

        return directory;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastFastaDirectory(File directory) {

        if (directory != null) {
            put(LAST_FASTA_DIRECTORY_KEY, directory.getAbsolutePath());
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastFastaDirectory() {

        File directory = null;

        String lastFilePath = get(LAST_FASTA_DIRECTORY_KEY, IGVConstants.getDefaultUserDirectory());

        if (lastFilePath != null) {
            directory = new File(lastFilePath);
        }

        return directory;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastSequenceDirectory(File directory) {

        if (directory != null) {
            put(LAST_SEQUENCE_DIRECTORY_KEY, directory.getAbsolutePath());
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastSequenceDirectory() {

        File directory = null;

        String lastFilePath = get(LAST_SEQUENCE_DIRECTORY_KEY, IGVConstants.getDefaultUserDirectory());

        if (lastFilePath != null) {
            directory = new File(lastFilePath);
        }

        return directory;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastSessionDirectory(File directory) {

        put(LAST_SESSION_DIRECTORY, directory.getAbsolutePath());
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastSessionDirectory() {

        File sessionDirectory = null;

        String lastFilePath = get(LAST_SESSION_DIRECTORY, null);

        if (lastFilePath != null) {

            // Create the session directory
            sessionDirectory = new File(lastFilePath);
        }

        return sessionDirectory;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean getShowRegionTool() {
        return getBooleanPreference(SHOW_REGION_TOOL_KEY, true);

    }

    /**
     * Method description
     *
     *
     * @param recentSessions
     */
    public void setRecentSessions(String recentSessions) {
        put(RECENT_SESSION_KEY, recentSessions);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getRecentSessions() {

        return get(RECENT_SESSION_KEY, null);
    }

    /**
     * Method description
     *
     *
     * @param masterResourceFile
     */
    public void setDataServerURL(String masterResourceFile) {
        put(DATA_SERVER_URL_KEY, masterResourceFile);
    }

    public String getDataServerURL() {

        String masterResourceFile = get(DATA_SERVER_URL_KEY, DEFAULT_DATA_SERVER_URL).
                replace("www.broadinstitute.org", "www.broadinstitute.org");
        return masterResourceFile;
    }

    /**
     * Method description
     *
     *
     * @param value
     */
    public void setShowRegionTool(boolean value) {
        put(SHOW_REGION_TOOL_KEY, Boolean.toString(value));
    }

    /**
     * Method description
     *
     *
     * @param isShowable
     */
    public void setShowAttributeView(boolean isShowable) {

        put(PreferenceManager.SHOW_ATTRIBUTE_VIEWS_KEY, Boolean.toString(isShowable));
    }

    public boolean getShowAttributeView() {
        return getBooleanPreference(SHOW_ATTRIBUTE_VIEWS_KEY, true);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean getShowMissingData() {
        return getBooleanPreference(SHOW_MISSING_DATA_KEY, false);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isShowSingleTrackPane() {

        String showSingleTrackPane =
                PreferenceManager.getInstance().get(PreferenceManager.SHOW_SINGLE_TRACK_PANE_KEY,
                "false");

        return Boolean.parseBoolean(showSingleTrackPane);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isDisplayableAttributesNotSet() {

        boolean isSet = false;

        String displayableAttributes = getInstance().get(DISPLAYABLE_ATTRIBUTES_KEY, null);

        // If first time a preference was set default all
        // attributes to dispayable
        if (displayableAttributes == null) {
            isSet = true;
        } else {
            isSet = true;    // TODO For now alsways default to true until we store this attribute in the future// isSet = false;
        }
        return isSet;
    }

    /**
     * Method description
     *
     *
     * @param attributeNames
     */
    public void setDisplayableAttributes(HashSet<String> attributeNames) {

        StringBuffer buffer = new StringBuffer();
        if (attributeNames == null) {
            remove(PreferenceManager.DISPLAYABLE_ATTRIBUTES_KEY);
            return;
        }

        int size = attributeNames.size();
        if (!attributeNames.isEmpty()) {

            for (String name : attributeNames) {
                buffer.append(name);

                // Skip last comma
                if (size > 1) {
                    buffer.append(",");
                    --size;
                }
            }
            put(PreferenceManager.DISPLAYABLE_ATTRIBUTES_KEY, buffer.toString());
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDisplayableAttributes() {

        String displayableAttributes = getInstance().get(DISPLAYABLE_ATTRIBUTES_KEY, null);

        return displayableAttributes;
    }

    /**
     * Method description
     *
     *
     * @param chromosome
     */
    public void setLastChromosomeViewed(String chromosome) {
        put(LAST_CHROMOSOME_VIEWED_KEY, chromosome);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getLastChromosomeViewed() {
        String chromosome = get(LAST_CHROMOSOME_VIEWED_KEY, "chr1");
        return chromosome;
    }

    /**
     * Method description
     *
     *
     * @param attributeName
     */
    public void setTrackAttributeName(String attributeName) {
        put(TRACK_ATTRIBUTE_NAME_KEY, attributeName);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getTrackAttributeName() {
        return get(TRACK_ATTRIBUTE_NAME_KEY, null);
    }

    /**
     * Method description
     *
     *
     * @param url
     */
    public void setTrackPropertiesURL(String url) {
        put(TRACK_PROPERTIES_URL_KEY, url);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getTrackPropertiesURL() {
        return get(TRACK_PROPERTIES_URL_KEY, null);
    }

    /**
     * Method description
     *
     *
     * @param file
     */
    public void setTrackPropertiesFile(String file) {
        put(TRACK_PROPERTIES_FILE, file);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getTrackPropertiesFile() {
        return get(TRACK_PROPERTIES_FILE, null);
    }

    /**
     * Method description
     *
     *
     * @param genomeId
     */
    public void setDefaultGenome(String genomeId) {
        put(DEFAULT_GENOME_KEY, genomeId);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDefaultGenome() {

        String genome = get(DEFAULT_GENOME_KEY, "hg18");
        return genome;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastTrackDirectory(File directory) {
        String lastDirectory = directory.getAbsolutePath();
        put(LAST_TRACK_DIRECTORY, lastDirectory);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastTrackDirectory() {

        String lastDirectoryPath = get(LAST_TRACK_DIRECTORY, null);

        File lastDirectoryFile = null;
        if (lastDirectoryPath != null) {
            lastDirectoryFile = new File(lastDirectoryPath);
        }

        return lastDirectoryFile;
    }

    /**
     * Method description
     *
     *
     * @param directory
     */
    public void setLastAttributeDirectory(File directory) {
        String lastDirectory = directory.getAbsolutePath();
        put(LAST_ATTRIBUTE_FILE_DIRECTORY, lastDirectory);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public File getLastAttributeDirectory() {

        String lastDirectoryPath = get(LAST_ATTRIBUTE_FILE_DIRECTORY, null);

        File lastDirectoryFile = null;
        if (lastDirectoryPath != null) {
            lastDirectoryFile = new File(lastDirectoryPath);
        }

        return lastDirectoryFile;
    }

    /**
     * Method description
     *
     *
     * @param height
     */
    public void setDefaultTrackHeight(String height) {
        put(TRACK_HEIGHT_KEY, height);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getDefaultTrackHeight() {

        int defaultTrackHeight = (int) (Double.parseDouble(
                PreferenceManager.getInstance().get(
                PreferenceManager.TRACK_HEIGHT_KEY,
                String.valueOf(INITIAL_TRACK_HEIGHT))));

        return defaultTrackHeight;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getDefaultChartTrackHeight() {

        int defaultTrackHeight = (int) (Double.parseDouble(
                PreferenceManager.getInstance().get(
                PreferenceManager.CHART_TRACK_HEIGHT_KEY,
                DEFAULT_CHART_TRACK_HEIGHT)));

        return defaultTrackHeight;
    }

    /**
     * Set the color scheme for the track type.
     *
     * @param type
     * @param colorScaleString
     */
    public void setColorScaleString(TrackType type, String colorScaleString) {

        put(COLOR_SCALE_KEY + type.toString(), colorScaleString);

        // Force cache update
        colorScaleCache.remove(type);
    }

    /**
     *  Set the color scheme for the track type.  Its unfortunate that this is a public
     *  method,  color schemes are managed by ColorScaleFactory and that is the
     *  only object that should call this method.
     *
     *  @param type
     * @param colorScale
     */
    public void setColorScale(TrackType type, ColorScale colorScale) {
        String colorScaleString = colorScale.asString();
        put(COLOR_SCALE_KEY + type.toString(), colorScaleString);
        colorScaleCache.put(type, colorScale);
    }

    /**
     * Get the color scheme for the track type.  In earlier versions of IGV a heatmap scale
     * could have (erroneously) been written for the "GENERIC" track type.  To prevent this
     * scale from being used the GENERIC type is filterd out.
     *
     * @param type
     *
     * @return
     */
    public ColorScale getColorScale(TrackType type) {
        ColorScale scale = colorScaleCache.get(type);
        if (scale == null) {

            String colorScaleString = get(COLOR_SCALE_KEY + type.toString(), null);
            if ((type != TrackType.GENERIC) && (colorScaleString != null)) {
                scale = ColorScaleFactory.getScaleFromString(colorScaleString);
            } else {
                scale = getDefaultColorScale(type);

            }
            if (scale != null) {
                colorScaleCache.put(type, scale);
            }
        }
        return scale;
    }

    /**
     * Return the default color scale.  This si the scale for track type "generic",
     * as well as any track type without a specific scale.
     *
     *
     * @param type
     * @return
     */
    public ColorScale getDefaultColorScale(TrackType type) {
        switch (type) {
            case LOH:
                return new ContinuousColorScale(0, -1, 0, 1, Color.red, IGVConstants.LIGHT_YELLOW,
                        Color.blue);

            case RNAI:
            case POOLED_RNAI:
                ContinuousColorScale cs = new ContinuousColorScale(0, -3, 0, 3, Color.red,
                        Color.white,
                        Color.blue);
                cs.setNoDataColor(new Color(225, 225, 225));
                return cs;

            case DNA_METHYLATION:
                cs = new ContinuousColorScale(0.4, 1, Color.WHITE, Color.GREEN.darker());
                cs.setNoDataColor(new Color(225, 225, 225));
                return cs;

            case GENE_EXPRESSION:
                cs = new ContinuousColorScale(-0.1, -1.5, 0.1, 1.5, Color.BLUE, Color.WHITE,
                        Color.RED);
                cs.setNoDataColor(new Color(225, 225, 225));
                return cs;

            case COPY_NUMBER:
            case ALLELE_SPECIFIC_COPY_NUMBER:
                return new ContinuousColorScale(-0.1, -1.5, 0.1, 1.5, Color.BLUE, Color.WHITE,
                        Color.RED);

            default:
                return null;
        }
    }

    /**
     * Method description
     *
     *
     * @param colorScheme
     */
    public void setMutationColorScheme(Map<Mutation.Type, Color> colorScheme) {

        setMutationColorScheme(
                getcommaSeparatedRGBString(colorScheme.get(Mutation.Type.Indel)),
                getcommaSeparatedRGBString(colorScheme.get(Mutation.Type.Missense)),
                getcommaSeparatedRGBString(colorScheme.get(Mutation.Type.Nonsense)),
                getcommaSeparatedRGBString(colorScheme.get(Mutation.Type.Splice_site)),
                getcommaSeparatedRGBString(colorScheme.get(Mutation.Type.Synonymous)),
                getcommaSeparatedRGBString(colorScheme.get(Mutation.Type.Targeted_Region)),
                getcommaSeparatedRGBString(colorScheme.get(Mutation.Type.Unknown)));
    }

    /**
     * Method description
     *
     *
     * @param indel
     * @param missense
     * @param nonsense
     * @param spliceSite
     * @param synonymous
     * @param targetedRegion
     * @param unknown
     */
    public void setMutationColorScheme(String indel, String missense, String nonsense,
            String spliceSite, String synonymous, String targetedRegion,
            String unknown) {

        put(MUTATION_INDEL_COLOR_KEY, indel);
        put(MUTATION_MISSENSE_COLOR_KEY, missense);
        put(MUTATION_NONSENSE_COLOR_KEY, nonsense);
        put(MUTATION_SPLICE_SITE_COLOR_KEY, spliceSite);
        put(MUTATION_SYNONYMOUS_COLOR_KEY, synonymous);
        put(MUTATION_TARGETED_REGION_COLOR_KEY, targetedRegion);
        put(MUTATION_UNKNOWN_COLOR_KEY, unknown);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Map<Mutation.Type, Color> getMutationColorScheme() {

        String indelColor = get(MUTATION_INDEL_COLOR_KEY, null);
        String missenseColor = get(MUTATION_MISSENSE_COLOR_KEY, null);
        String nonsenseColor = get(MUTATION_NONSENSE_COLOR_KEY, null);
        String spliceSiteColor = get(MUTATION_SPLICE_SITE_COLOR_KEY, null);
        String synonymousColor = get(MUTATION_SYNONYMOUS_COLOR_KEY, null);
        String targetedRegionColor = get(MUTATION_TARGETED_REGION_COLOR_KEY, null);
        String unknownColor = get(MUTATION_UNKNOWN_COLOR_KEY, null);

        HashMap<Mutation.Type, Color> colorScheme = null;
        if ((indelColor != null) && (missenseColor != null) && (nonsenseColor != null) && (spliceSiteColor != null) && (synonymousColor != null) && (targetedRegionColor != null) && (unknownColor != null)) {

            colorScheme = new HashMap<Mutation.Type, Color>();

            String rgb[] = indelColor.split(",");
            Color color1 = new Color(Integer.parseInt(rgb[0]), Integer.parseInt(rgb[1]),
                    Integer.parseInt(rgb[2]));
            colorScheme.put(Mutation.Type.Indel, color1);

            rgb = missenseColor.split(",");
            Color color2 = new Color(Integer.parseInt(rgb[0]), Integer.parseInt(rgb[1]),
                    Integer.parseInt(rgb[2]));
            colorScheme.put(Mutation.Type.Missense, color2);

            rgb = nonsenseColor.split(",");
            Color color3 = new Color(Integer.parseInt(rgb[0]), Integer.parseInt(rgb[1]),
                    Integer.parseInt(rgb[2]));
            colorScheme.put(Mutation.Type.Nonsense, color3);

            rgb = spliceSiteColor.split(",");
            Color color4 = new Color(Integer.parseInt(rgb[0]), Integer.parseInt(rgb[1]),
                    Integer.parseInt(rgb[2]));
            colorScheme.put(Mutation.Type.Splice_site, color4);

            rgb = synonymousColor.split(",");
            Color color5 = new Color(Integer.parseInt(rgb[0]), Integer.parseInt(rgb[1]),
                    Integer.parseInt(rgb[2]));
            colorScheme.put(Mutation.Type.Synonymous, color5);

            rgb = targetedRegionColor.split(",");
            Color color6 = new Color(Integer.parseInt(rgb[0]), Integer.parseInt(rgb[1]),
                    Integer.parseInt(rgb[2]));
            colorScheme.put(Mutation.Type.Targeted_Region, color6);

            rgb = unknownColor.split(",");
            Color color7 = new Color(Integer.parseInt(rgb[0]), Integer.parseInt(rgb[1]),
                    Integer.parseInt(rgb[2]));
            colorScheme.put(Mutation.Type.Unknown, color7);
        }

        return colorScheme;
    }

    /**
     * Method description
     *
     *
     * @throws Exception
     */
    public void export() throws Exception {
        IGV_PREFERENCE_MANAGER.exportSubtree(System.out);
    }

    /**
     * Method description
     *
     *
     * @param outputStream
     *
     * @throws Exception
     */
    public void export(OutputStream outputStream) throws Exception {
        IGV_PREFERENCE_MANAGER.exportSubtree(outputStream);
    }

    /**
     * Method description
     *
     *
     * @param inStream
     *
     * @throws Exception
     */
    public void importPreferences(InputStream inStream) throws Exception {
        Preferences.importPreferences(inStream);
    }

    /**
     */
    public ChartPreferences getChartPreferences() {
        ChartPreferences prefs =
                (ChartPreferences) objectCache.get(ChartPreferences.class.getName());
        if (prefs == null) {
            prefs = new ChartPreferences();
            objectCache.put(ChartPreferences.class.getName(), prefs);
        }
        prefs.setDrawTopBorder(getBooleanPreference(ChartPreferences.DRAW_TOP_BORDER, false));
        prefs.setDrawBottomBorder(getBooleanPreference(ChartPreferences.DRAW_BOTTOM_BORDER, false));
        prefs.setColorBorders(getBooleanPreference(ChartPreferences.COLOR_BORDERS, true));
        prefs.setDrawAxis(getBooleanPreference(ChartPreferences.Y_AXIS, false));
        prefs.setDrawTrackName(getBooleanPreference(ChartPreferences.DRAW_TRACK_NAME, false));
        prefs.setColorTrackName(getBooleanPreference(ChartPreferences.COLOR_TRACK_NAME, true));
        return prefs;
    }

    /**
     */
    public void setChartPreferences(ChartPreferences prefs) {
        objectCache.put(ChartPreferences.class.getName(), prefs);
        put(ChartPreferences.DRAW_TOP_BORDER, String.valueOf(prefs.isDrawTopBorder()));
        put(ChartPreferences.DRAW_BOTTOM_BORDER, String.valueOf(prefs.isDrawBottomBorder()));
        put(ChartPreferences.COLOR_BORDERS, String.valueOf(prefs.isColorBorders()));
        put(ChartPreferences.Y_AXIS, String.valueOf(prefs.isDrawAxis()));
        put(ChartPreferences.DRAW_TRACK_NAME, String.valueOf(prefs.isDrawTrackName()));
        put(ChartPreferences.COLOR_TRACK_NAME, String.valueOf(prefs.isColorTrackName()));
    }

    /**
     * Class description
     *
     *
     * @version    Enter version here..., 08/11/02
     * @author     Enter your name here...
     */
    public static class ChartPreferences {

        public static final String DRAW_TOP_BORDER = "CHART.DRAW_TOP_BORDER";
        public static final String DRAW_BOTTOM_BORDER = "CHART.DRAW_BOTTOM_BORDER";
        public static final String COLOR_BORDERS = "CHART.COLOR_BORDERS";
        public static final String Y_AXIS = "CHART.DRAW_AXIS";
        public static final String DRAW_TRACK_NAME = "CHART.DRAW_TRACK_NAME";
        public static final String COLOR_TRACK_NAME = "CHART.COLOR_TRACK_NAME";
        private boolean drawTopBorder;
        private boolean drawBottomBorder;
        private boolean colorBorders;
        private boolean drawAxis;
        private boolean drawTrackName;
        private boolean colorTrackName;

        public ChartPreferences() {
        }

        public boolean isDrawTopBorder() {
            return drawTopBorder;
        }

        public void setDrawTopBorder(boolean drawBorders) {
            this.drawTopBorder = drawBorders;
        }

        public boolean isColorBorders() {
            return colorBorders;
        }

        public void setColorBorders(boolean colorBorders) {
            this.colorBorders = colorBorders;
        }

        public boolean isDrawAxis() {
            return drawAxis;
        }

        public void setDrawAxis(boolean drawAxis) {
            this.drawAxis = drawAxis;
        }

        public boolean isDrawBottomBorder() {
            return drawBottomBorder;
        }

        public void setDrawBottomBorder(boolean drawBottomBorder) {
            this.drawBottomBorder = drawBottomBorder;
        }

        public boolean isDrawTrackName() {
            return drawTrackName;
        }

        public void setDrawTrackName(boolean drawTrackName) {
            this.drawTrackName = drawTrackName;
        }

        public boolean isColorTrackName() {
            return colorTrackName;
        }

        public void setColorTrackName(boolean colorTrackName) {
            this.colorTrackName = colorTrackName;
        }
    }

    /**
     */
    public SAMPreferences getSAMPreferences() {
        SAMPreferences prefs =
                (SAMPreferences) objectCache.get(SAMPreferences.class.getName());
        if (prefs == null) {
            prefs = new SAMPreferences();
            objectCache.put(SAMPreferences.class.getName(), prefs);

            prefs.setMaxVisibleRange(Float.parseFloat(get(SAMPreferences.MAX_VISIBLE_RANGE, "30")));
            prefs.setShowDuplicates(getBooleanPreference(SAMPreferences.SHOW_DUPLICATES, false));
            prefs.setQualityThreshold(Integer.parseInt(get(SAMPreferences.QUALITY_THRESHOLD, "0")));
            prefs.setInsertSizeThreshold(Integer.parseInt(get(SAMPreferences.INSERT_SIZE_THRESHOLD, "10000")));
            prefs.setFlagUnmappedPair(getBooleanPreference(SAMPreferences.FLAG_UNMAPPED_PAIR, false));
            prefs.setAutosort(getBooleanPreference(SAMPreferences.AUTO_SORT, false));
            prefs.setShadeCenterFlag(getBooleanPreference(SAMPreferences.SHADE_CENTER, true));
            prefs.setMaxLevels(getIntegerPreference(SAMPreferences.MAX_LEVELS, 1000));
            prefs.setShadeBaseQuality(getBooleanPreference(SAMPreferences.SHADE_BASE_QUALITY, true));
            prefs.setBaseQualityMin(getIntegerPreference(SAMPreferences.BASE_QUALITY_MIN, 5));
            prefs.setBaseQualityMax(getIntegerPreference(SAMPreferences.BASE_QUALITY_MAX, 20));
        }

        return prefs;
    }

    public void updateSAMPreferences() {
        objectCache.remove(SAMPreferences.class.getName());
    }

    public int getIntegerPreference(String key, int def) {
        try {
            String prefString = get(key, null);
            if (prefString != null) {
                return Integer.parseInt(prefString);
            }
        } catch (Exception e) {
            log.error("Number format error for preference: " + key);

        }
        return def;
    }

    public static class SAMPreferences {

        public static final String MAX_VISIBLE_RANGE = "SAM.MAX_VISIBLE_RANGE";
        public static final String SHOW_ZERO_QUALITY = "SAM.SHOW_VISIBLE_QUALITY";
        public static final String SHOW_DUPLICATES = "SAM.SHOW_DUPLICATES";
        public static final String FLAG_UNMAPPED_PAIR = "SAM.FLAG_UNMAPPED_PAIR";
        public static final String MAX_LEVELS = "SAM.MAX_LEVELS";
        public static final String QUALITY_THRESHOLD = "SAM.QUALITY_THRESHOLD";
        public static final String INSERT_SIZE_THRESHOLD = "SAM.INSERT_SIZE_THRESHOLD";
        public static final String AUTO_SORT = "SAM.AUTOSORT";
        public static final String SHADE_CENTER = "SAM.SHADE_CENTER";
        public static final String SHADE_BASE_QUALITY = "SAM.SHADE_BASE_QUALITY";
        public static final String BASE_QUALITY_MIN = "SAM.BASE_QUALITY_MIN";
        public static final String BASE_QUALITY_MAX = "SAM.BASE_QUALITY_MAX";
        private float maxVisibleRange;
        private int maxLevels;
        private int qualityThreshold;
        private boolean showDuplicates;
        private boolean flagUnmappedPair;
        private boolean autosort;
        private boolean shadeBaseQuality;
        private int insertSizeThreshold;
        private int baseQualityMin = 0;
        private int baseQualityMax = 40;
        private boolean shadeCenterFlag;

        /**
         * @return the maxVisibleRange
         */
        public float getMaxVisibleRange() {
            return maxVisibleRange;
        }

        /**
         * @param maxVisibleRange the maxVisibleRange to set
         */
        public void setMaxVisibleRange(float maxVisibleRange) {
            this.maxVisibleRange = maxVisibleRange;
        }

        /**
         * @return the showDuplicates
         */
        public boolean isShowDuplicates() {
            return showDuplicates;
        }

        /**
         * @param showDuplicates the showDuplicates to set
         */
        public void setShowDuplicates(boolean showDuplicates) {
            this.showDuplicates = showDuplicates;
        }

        /**
         * @return the flagUnmappedPair
         */
        public boolean isFlagUnmappedPair() {
            return flagUnmappedPair;
        }

        /**
         * @param flagUnmappedPair the flagUnmappedPair to set
         */
        public void setFlagUnmappedPair(boolean flagUnmappedPair) {
            this.flagUnmappedPair = flagUnmappedPair;
        }

        /**
         * @return the autosort
         */
        public boolean isAutosort() {
            return autosort;
        }

        /**
         * @param autosort the autosort to set
         */
        public void setAutosort(boolean autosort) {
            this.autosort = autosort;
        }

        /**
         * @return the maxLevels
         */
        public int getMaxLevels() {
            return maxLevels;
        }

        /**
         * @param maxLevels the maxLevels to set
         */
        public void setMaxLevels(int maxLevels) {
            this.maxLevels = maxLevels;
        }

        /**
         * @return the qualityThreshold
         */
        public int getQualityThreshold() {
            return qualityThreshold;
        }

        /**
         * @param qualityThreshold the qualityThreshold to set
         */
        public void setQualityThreshold(int qualityThreshold) {
            this.qualityThreshold = qualityThreshold;
        }

        public int getInsertSizeThreshold() {
            return insertSizeThreshold;
        }

        public void setInsertSizeThreshold(int insertSizeThreshold) {
            this.insertSizeThreshold = insertSizeThreshold;
        }

        public boolean isShadeBaseQuality() {
            return shadeBaseQuality;
        }

        public void setShadeBaseQuality(boolean shadeBaseQuality) {
            this.shadeBaseQuality = shadeBaseQuality;
        }

        public int getBaseQualityMin() {
            return baseQualityMin;
        }

        public void setBaseQualityMin(int baseQualityMin) {
            this.baseQualityMin = baseQualityMin;
        }

        public int getBaseQualityMax() {
            return baseQualityMax;
        }

        public void setBaseQualityMax(int baseQualityMax) {
            this.baseQualityMax = baseQualityMax;
        }

        /**
         * @return the shadeCenterFlag
         */
        public boolean isShadeCenterFlag() {
            return shadeCenterFlag;
        }

        /**
         * @param shadeCenterFlag the shadeCenterFlag to set
         */
        public void setShadeCenterFlag(boolean shadeCenterFlag) {
            this.shadeCenterFlag = shadeCenterFlag;
        }
    }
}
