/*
* The Broad Institute
* SOFTWARE COPYRIGHT NOTICE AGREEMENT
* This is copyright (2007-2008) by the Broad Institute/Massachusetts Institute
* of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
* (the "License"); you may not use this file except in compliance with
*  the License.  You may obtain a copy of the License at
*
*    http://www.opensource.org/licenses/gpl-2.0.php
*
* This software is supplied without any warranty or guaranteed support
* whatsoever. Neither the Broad Institute nor MIT can be responsible for its
* use, misuse, or functionality.
*/



package org.broad.igv.data.wgs;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.feature.Genome;
import org.broad.igv.feature.GenomeManager;
import org.broad.igv.track.TrackType;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author jrobinso
 */
public class SequenceDataset {

    String genomeId;
    Genome genome;
    String trackName;
    
    int [] locations;
    float[] data;
    
    File directory;
    String color;
    String [] chromosomes;
    
    /**
     * Constructs ...
     *
     *
     * @param path
     */
    public SequenceDataset(String path) {

        directory = new File(path);
        //if (!(directory.exists() && directory.isDirectory()))
        //{
        //    System.err.println("Directory: " + path + " does not exist or is not a directory");
        //}

        readProperties();
        
        scanForChromosomes();

        genome = GenomeManager.getInstance().getGenome(genomeId);

        // locations = new int[250000000];
        // data = new float[locations.length];
    }

    private void readProperties() {
        File propFile = new File(directory, "property.txt");
        if (propFile.exists())
        {
            InputStream is = null;
            try
            {
                Properties props = new Properties();
                is = new FileInputStream(propFile);
                props.load(is);
                is.close();

                genomeId = props.getProperty("genome");
                trackName = props.getProperty("name");
                color = props.getProperty("color");
            }
            catch (IOException ex)
            {
                ex.printStackTrace();
            }
            finally
            {
                try
                {
                    is.close();
                }
                catch (IOException ex)
                {
                    ex.printStackTrace();
                }
            }
        }
        else
        {
            genomeId = "hg18";
            trackName = directory.getName();
            color = "0,0,255";
        }
    }
    
    private void scanForChromosomes() {
        File [] files = directory.listFiles(new FileFilter() {

            public boolean accept(File arg0) {
                return arg0.getName().startsWith("chr") && arg0.getName().endsWith(".bin");
            }
        });
        chromosomes = new String[files.length];
        for(int i=0; i<files.length; i++) {
            chromosomes[i] = files[i].getName().replace(".bin", "").trim();
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getName() {
        return trackName;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public TrackType getType() {
        return TrackType.OTHER;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Genome getGenome() {
        return genome;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String[] getChromosomes() {
        return chromosomes;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getTrackName() {
        return trackName;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getWindowSpan() {
        return 1;
    }

    /**
     * Method description
     *
     *
     * @param chr
     *
     * @return
     */
    public float[] getData(String chr) {

        DataInputStream dis = null;
        File inputFile = new File(directory, chr + ".bin");
        if (!inputFile.exists())
        {
            return null;
        }
        int chrLength = genome.getChromosome(chr).getLength();
        float[] data = new float[chrLength];
        try
        {
            dis = new DataInputStream(new BufferedInputStream(new FileInputStream(inputFile)));
            for (int i = 0; i < chrLength; i++)
            {
                if (dis.available() >= (Short.SIZE / 8))
                {
                    data[i] = dis.readShort();
                }
            }
            dis.close();
        }
        catch (IOException e)
        {
            e.printStackTrace();

        }
        finally
        {
            try
            {
                dis.close();
            }
            catch (IOException ex)
            {
                ex.printStackTrace();
            }
        }
        return data;

    }



    /**
     * Method description
     *
     *
     * @param args
     */
    public static void main(String[] args) {
        SequenceDataset ds = new SequenceDataset("/Users/jrobinso/IGV/SeqTrack/");
        
        float [] data = ds.getData("chrX");
        
        System.out.println(data.length);
    }
}
