/*
 * Decompiled with CFR 0.152.
 */
package org.igv.hic;

import htsjdk.samtools.seekablestream.SeekableStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import org.igv.feature.Chromosome;
import org.igv.feature.genome.Genome;
import org.igv.hic.ContactRecord;
import org.igv.hic.Matrix;
import org.igv.hic.MatrixZoomData;
import org.igv.hic.NVI;
import org.igv.hic.NormalizationVector;
import org.igv.hic.Region;
import org.igv.hic.StaticBlockIndex;
import org.igv.logging.LogManager;
import org.igv.logging.Logger;
import org.igv.util.CompressionUtils;
import org.igv.util.collections.CaseInsensitiveMap;
import org.igv.util.collections.LRUCache;
import org.igv.util.stream.IGVSeekableStreamFactory;

public class HicFile {
    private static final Logger log = LogManager.getLogger(HicFile.class);
    private final SeekableStream fileChannel;
    private final Map<String, Object> config;
    private final String path;
    private final Genome genome;
    private boolean initialized = false;
    private String magic;
    private Integer version;
    private long footerPosition;
    private String genomeId;
    private long normVectorIndexPosition;
    private int normVectorIndexSize;
    private Map<String, IndexEntry> masterIndex;
    private Map<String, Long> expectedValueVectors;
    private Map<String, Object> attributes;
    private List<Chromosome> chromosomes = new ArrayList<Chromosome>();
    private Map<String, Integer> chromosomeIndexMap = new CaseInsensitiveMap<Integer>();
    private Integer wgResolution = null;
    private List<Integer> bpResolutions = new ArrayList<Integer>();
    private List<Integer> fragResolutions = new ArrayList<Integer>();
    private Map<String, String> chrAliasTable = new HashMap<String, String>();
    private Map<String, IndexEntry> normVectorIndex;
    private LRUCache<String, NormalizationVector> normVectorCache = new LRUCache(10);
    private LRUCache<String, Matrix> matrixCache = new LRUCache(10);
    private BlockCache blockCache = new BlockCache();
    private List<String> normalizationTypes = new ArrayList<String>(Collections.singletonList("NONE"));
    private Long normExpectedValueVectorsPosition;
    private final Map<String, CompletableFuture<Block>> pendingBlockRequests = new ConcurrentHashMap<String, CompletableFuture<Block>>();

    public HicFile(String path, Genome genome) throws IOException {
        this.path = path;
        this.genome = genome;
        this.fileChannel = IGVSeekableStreamFactory.getInstance().getStreamFor(path);
        this.config = Collections.emptyMap();
        this.init();
    }

    private synchronized void init() throws IOException {
        if (this.initialized) {
            return;
        }
        this.readHeaderAndFooter();
        this.initialized = true;
    }

    public int getVersion() {
        return this.version;
    }

    public String getNVIString() {
        if (this.normVectorIndexPosition > 0L && this.normVectorIndexSize > 0) {
            return this.normVectorIndexPosition + "," + this.normVectorIndexSize;
        }
        return null;
    }

    private static ByteBuffer wrap(byte[] data) {
        ByteBuffer bb = ByteBuffer.wrap(data);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        return bb;
    }

    private void readHeaderAndFooter() throws IOException {
        byte[] header = this.readBytes(0L, 16L);
        if (header == null || header.length == 0) {
            throw new IOException("File content is empty");
        }
        ByteBuffer bp = HicFile.wrap(header);
        this.magic = HicFile.getString(bp);
        this.version = bp.getInt();
        if (this.version < 5) {
            throw new IOException("Unsupported hic version: " + this.version);
        }
        this.footerPosition = bp.getLong();
        this.readFooter();
        long bodyPosition = Long.MAX_VALUE;
        for (IndexEntry e : this.masterIndex.values()) {
            bodyPosition = Math.min(bodyPosition, e.start);
        }
        long remainingSize = bodyPosition - 16L;
        byte[] body = this.readBytes(16L, (int)remainingSize);
        ByteBuffer bodyParser = HicFile.wrap(body);
        this.genomeId = HicFile.getString(bodyParser);
        if (this.version >= 9) {
            this.normVectorIndexPosition = bodyParser.getLong();
            this.normVectorIndexSize = (int)bodyParser.getLong();
        } else {
            String nviStr = NVI.getNVI(this.path);
            if (nviStr != null) {
                try {
                    String[] parts = nviStr.split(",");
                    this.normVectorIndexPosition = Long.parseLong(parts[0]);
                    this.normVectorIndexSize = Integer.parseInt(parts[1]);
                }
                catch (NumberFormatException e) {
                    log.error("Error parsing NVI string: " + nviStr, e);
                }
            }
        }
        this.attributes = new HashMap<String, Object>();
        int nAttributes = bodyParser.getInt();
        while (nAttributes-- > 0) {
            String k = HicFile.getString(bodyParser);
            String v = HicFile.getString(bodyParser);
            this.attributes.put(k, v);
        }
        this.chromosomes = new ArrayList<Chromosome>();
        this.chromosomeIndexMap = new CaseInsensitiveMap<Integer>();
        int nChrs = bodyParser.getInt();
        for (int i = 0; i < nChrs; ++i) {
            String name = HicFile.getString(bodyParser);
            long size = this.version < 9 ? (long)bodyParser.getInt() : bodyParser.getLong();
            Chromosome chr = new Chromosome(i, name, (int)size);
            this.chromosomes.add(chr);
            String canonicalName = this.genome == null ? name : this.genome.getCanonicalChrName(name);
            this.chrAliasTable.put(canonicalName, name);
            this.chromosomeIndexMap.put(name, i);
        }
        int nBp = bodyParser.getInt();
        for (int i = 0; i < nBp; ++i) {
            this.bpResolutions.add(bodyParser.getInt());
        }
        boolean loadFragData = false;
        if (loadFragData) {
            int nFrag = bodyParser.getInt();
            for (int i = 0; i < nFrag; ++i) {
                this.fragResolutions.add(bodyParser.getInt());
            }
        }
    }

    private void readFooter() throws IOException {
        int skip = this.version < 9 ? 8 : 12;
        byte[] data = this.readBytes(this.footerPosition, skip);
        if (data == null) {
            return;
        }
        ByteBuffer bp = HicFile.wrap(data);
        long nBytes = this.version < 9 ? (long)bp.getInt() : bp.getLong();
        int nEntries = bp.getInt();
        int miSize = nEntries * 196;
        byte[] miData = this.readBytes(this.footerPosition + (long)skip, Math.min(miSize, (int)nBytes));
        ByteBuffer miParser = HicFile.wrap(miData);
        this.masterIndex = new HashMap<String, IndexEntry>();
        while (nEntries-- > 0) {
            String key = HicFile.getString(miParser);
            long pos = miParser.getLong();
            int size = miParser.getInt();
            this.masterIndex.put(key, new IndexEntry(pos, size));
        }
        if (this.version > 5) {
            int skip2 = this.version < 9 ? 4 : 8;
            this.normExpectedValueVectorsPosition = this.footerPosition + (long)skip2 + nBytes;
        }
    }

    private Matrix getMatrix(int chrIdx1, int chrIdx2) throws IOException {
        String key = Matrix.getKey(chrIdx1, chrIdx2);
        if (this.matrixCache.containsKey(key)) {
            return this.matrixCache.get(key);
        }
        Matrix m = this.readMatrix(chrIdx1, chrIdx2);
        if (m != null) {
            this.matrixCache.put(key, m);
        }
        return m;
    }

    private Matrix readMatrix(int chrIdx1, int chrIdx2) throws IOException {
        String key;
        IndexEntry idx;
        if (chrIdx1 > chrIdx2) {
            int tmp = chrIdx1;
            chrIdx1 = chrIdx2;
            chrIdx2 = tmp;
        }
        if ((idx = this.masterIndex.get(key = Matrix.getKey(chrIdx1, chrIdx2))) == null) {
            return null;
        }
        byte[] data = this.readBytes(idx.start, idx.size);
        if (data == null) {
            return null;
        }
        return Matrix.parseMatrix(data, this.chromosomes);
    }

    public List<Integer> getBpResolutions() {
        return this.bpResolutions;
    }

    public List<ContactRecord> getContactRecords(Region region1, Region region2, String units, int binSize, String normalization, boolean allRecords) throws IOException {
        return this.getContactRecords(region1, region2, units, binSize, normalization, allRecords, 1);
    }

    public List<ContactRecord> getContactRecords(Region region1, Region region2, String units, int binSize, String normalization, boolean allRecords, int countsThreshold) throws IOException {
        NormalizationVector nv;
        List<Block> blocks;
        int idx2;
        boolean transpose;
        int idx1 = this.chromosomeIndexMap.getOrDefault(this.getFileChrName(region1.chr()), -1);
        boolean bl = transpose = idx1 > (idx2 = this.chromosomeIndexMap.getOrDefault(this.getFileChrName(region2.chr()), -1).intValue()) || idx1 == idx2 && region1.start() >= region2.end();
        if (transpose) {
            Region tmp = region1;
            region1 = region2;
            region2 = tmp;
        }
        if ((blocks = this.getBlocks(region1, region2, units, binSize)) == null || blocks.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<ContactRecord> contactRecords = new ArrayList<ContactRecord>();
        double x1 = (double)region1.start() / (double)binSize;
        double x2 = (double)region1.end() / (double)binSize;
        double y1 = (double)region2.start() / (double)binSize;
        double y2 = (double)region2.end() / (double)binSize;
        boolean useNormalization = normalization != null && !"NONE".equals(normalization);
        NormalizationVector normalizationVector = nv = useNormalization ? this.getNormalizationVector(normalization, region1.chr(), "BP", binSize) : null;
        if (nv == null) {
            useNormalization = false;
        }
        for (Block block : blocks) {
            if (block == null) continue;
            double[] normVector = null;
            int binMin = Integer.MAX_VALUE;
            int binMax = Integer.MIN_VALUE;
            if (useNormalization) {
                for (ContactRecord rec : block.records) {
                    binMin = Math.min(binMin, Math.min(rec.bin1(), rec.bin2()));
                    binMax = Math.max(binMax, Math.max(rec.bin1(), rec.bin2()));
                }
                normVector = nv.getValues(binMin, binMax);
            }
            for (ContactRecord rec : block.records) {
                if (!allRecords && (!((double)rec.bin1() >= x1) || !((double)rec.bin1() < x2) || !((double)rec.bin2() >= y1) || !((double)rec.bin2() < y2)) || !(rec.counts() > (float)countsThreshold)) continue;
                if (normVector == null) {
                    contactRecords.add(rec);
                    continue;
                }
                double nvnv = normVector[rec.bin1() - binMin] * normVector[rec.bin2() - binMin];
                if (Double.isNaN(nvnv)) continue;
                float normCounts = (float)((double)rec.counts() / nvnv);
                ContactRecord normRec = new ContactRecord(rec.bin1(), rec.bin2(), rec.counts(), normCounts);
                contactRecords.add(normRec);
            }
        }
        return contactRecords;
    }

    public int getWGResolution() {
        if (this.wgResolution == null) {
            try {
                Integer idx = this.chromosomeIndexMap.get("all");
                if (idx == null) {
                    return -1;
                }
                Matrix matrix = this.getMatrix(idx, idx);
                if (matrix == null) {
                    return -1;
                }
                List<MatrixZoomData> zdArray = matrix.getBpZoomData();
                if (zdArray.isEmpty()) {
                    return -1;
                }
                this.wgResolution = zdArray.get(0).getZoom().binSize();
            }
            catch (IOException e) {
                log.error(e.getMessage());
                this.wgResolution = -1;
            }
        }
        return this.wgResolution;
    }

    public int getBinSize(String chr, double bpPerPixel) {
        if ("all".equalsIgnoreCase(chr)) {
            return this.getWGResolution();
        }
        List<Integer> resolutions = this.getBpResolutions();
        int index = 0;
        for (int i = resolutions.size() - 1; i >= 0; --i) {
            if (!((double)resolutions.get(i).intValue() >= bpPerPixel)) continue;
            index = i;
            break;
        }
        int binSize = resolutions.get(index);
        return binSize;
    }

    private List<Block> getBlocks(Region region1, Region region2, String unit, int binSize) throws IOException {
        String key;
        this.init();
        String chr1 = this.getFileChrName(region1.chr());
        String chr2 = this.getFileChrName(region2.chr());
        Integer idx1 = this.chromosomeIndexMap.get(chr1);
        Integer idx2 = this.chromosomeIndexMap.get(chr2);
        if (idx1 == null || idx2 == null) {
            return Collections.emptyList();
        }
        Matrix matrix = this.getMatrix(idx1, idx2);
        if (matrix == null) {
            return Collections.emptyList();
        }
        MatrixZoomData zd = matrix.getZoomData(binSize, unit);
        if (zd == null) {
            log.info("No data available for resolution: " + binSize + " for chromosome pair: " + chr1 + ", " + chr2);
            return Collections.emptyList();
        }
        List<Integer> blockNumbers = zd.getBlockNumbers(region1, region2, this.version);
        ArrayList<Block> blocks = new ArrayList<Block>();
        ArrayList<Integer> toQuery = new ArrayList<Integer>();
        for (Integer num : blockNumbers) {
            key = zd.getKey() + "_" + num;
            if (this.blockCache.has(binSize, key)) {
                blocks.add(this.blockCache.get(binSize, key));
                continue;
            }
            toQuery.add(num);
        }
        for (Integer bn : toQuery) {
            key = zd.getKey() + "_" + bn;
            Block block = this.getBlockWithDeduplication(key, bn, binSize, zd);
            blocks.add(block);
        }
        return blocks;
    }

    private Block getBlockWithDeduplication(String key, int blockNumber, int binSize, MatrixZoomData zd) throws IOException {
        if (this.blockCache.has(binSize, key)) {
            return this.blockCache.get(binSize, key);
        }
        CompletableFuture<Block> newFuture = new CompletableFuture<Block>();
        CompletableFuture existingFuture = this.pendingBlockRequests.putIfAbsent(key, newFuture);
        if (existingFuture != null) {
            try {
                return (Block)existingFuture.get();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException("Interrupted while waiting for block data", e);
            }
            catch (ExecutionException e) {
                Throwable cause = e.getCause();
                if (cause instanceof IOException) {
                    throw (IOException)cause;
                }
                throw new IOException("Error reading block", cause);
            }
        }
        try {
            Block block = this.readBlock(blockNumber, zd);
            if (block != null) {
                this.blockCache.set(binSize, key, block);
            }
            newFuture.complete(block);
            Block block2 = block;
            return block2;
        }
        catch (IOException e) {
            newFuture.completeExceptionally(e);
            throw e;
        }
        finally {
            this.pendingBlockRequests.remove(key);
        }
    }

    private Block readBlock(int blockNumber, MatrixZoomData zd) throws IOException {
        StaticBlockIndex.BlockIndexEntry idx = zd.getBlockIndex().getBlockIndexEntry(blockNumber);
        if (idx == null) {
            return null;
        }
        byte[] data = this.readBytes(idx.filePosition, idx.size);
        if (data == null) {
            return null;
        }
        byte[] plain = new CompressionUtils().decompress(data);
        ByteBuffer parser = HicFile.wrap(plain);
        int nRecords = parser.getInt();
        ArrayList<ContactRecord> records = new ArrayList<ContactRecord>();
        if (this.version < 7) {
            for (int i = 0; i < nRecords; ++i) {
                int binX = parser.getInt();
                int binY = parser.getInt();
                float counts = parser.getFloat();
                records.add(new ContactRecord(binX, binY, counts));
            }
        } else {
            boolean useIntXPos;
            boolean useFloatContact;
            int binXOffset = parser.getInt();
            int binYOffset = parser.getInt();
            boolean bl = useFloatContact = parser.get() == 1;
            boolean bl2 = this.version < 9 ? false : (useIntXPos = parser.get() == 1);
            boolean useIntYPos = this.version < 9 ? false : parser.get() == 1;
            byte type = parser.get();
            int Short_MIN_VALUE = Short.MIN_VALUE;
            if (type == 1) {
                int rowCount = useIntYPos ? parser.getInt() : (int)parser.getShort();
                for (int i = 0; i < rowCount; ++i) {
                    int dy = useIntYPos ? parser.getInt() : (int)parser.getShort();
                    int binY = binYOffset + dy;
                    int colCount = useIntXPos ? parser.getInt() : (int)parser.getShort();
                    for (int j = 0; j < colCount; ++j) {
                        int dx = useIntXPos ? parser.getInt() : (int)parser.getShort();
                        int binX = binXOffset + dx;
                        float counts = useFloatContact ? parser.getFloat() : (float)parser.getShort();
                        records.add(new ContactRecord(binX, binY, counts));
                    }
                }
            } else if (type == 2) {
                int nPts = parser.getInt();
                short w = parser.getShort();
                for (int i = 0; i < nPts; ++i) {
                    int row = i / w;
                    int col = i - row * w;
                    int bin1 = binXOffset + col;
                    int bin2 = binYOffset + row;
                    if (useFloatContact) {
                        float counts = parser.getFloat();
                        if (Float.isNaN(counts)) continue;
                        records.add(new ContactRecord(bin1, bin2, counts));
                        continue;
                    }
                    short counts = parser.getShort();
                    if (counts == Short.MIN_VALUE) continue;
                    records.add(new ContactRecord(bin1, bin2, counts));
                }
            } else {
                throw new IOException("Unknown block type: " + type);
            }
        }
        return new Block(blockNumber, zd, records, idx);
    }

    public boolean hasNormalizationVector(String type, String chr, String unit, int binSize) {
        int chrIdx = this.chromosomeIndexMap.getOrDefault(this.getFileChrName(chr), -1);
        String key = HicFile.getNormalizationVectorKey(type, chrIdx, unit, binSize);
        if (this.normVectorCache.containsKey(key)) {
            return true;
        }
        try {
            Map<String, IndexEntry> nvi = this.getNormVectorIndex();
            return nvi == null ? false : nvi.containsKey(key);
        }
        catch (IOException e) {
            log.error("Error reading norm vector index", e);
            return false;
        }
    }

    public NormalizationVector getNormalizationVector(String type, String chr, String unit, int binSize) throws IOException {
        this.init();
        int chrIdx = this.chromosomeIndexMap.getOrDefault(this.getFileChrName(chr), -1);
        String key = HicFile.getNormalizationVectorKey(type, chrIdx, unit, binSize);
        if (this.normVectorCache.containsKey(key)) {
            return this.normVectorCache.get(key);
        }
        Map<String, IndexEntry> nvi = this.getNormVectorIndex();
        if (nvi == null) {
            return null;
        }
        IndexEntry idx = nvi.get(key);
        if (idx == null) {
            return null;
        }
        byte[] header = this.readBytes(idx.start(), 8L);
        if (header == null) {
            return null;
        }
        ByteBuffer parser = HicFile.wrap(header);
        long nValues = this.version < 9 ? (long)parser.getInt() : parser.getLong();
        DataType dataType = this.version < 9 ? DataType.DOUBLE : DataType.FLOAT;
        long filePos = this.version < 9 ? idx.start() + 4L : idx.start() + 8L;
        NormalizationVector nv = new NormalizationVector(this.fileChannel, filePos, (int)nValues, dataType);
        this.normVectorCache.put(key, nv);
        return nv;
    }

    private Map<String, IndexEntry> getNormVectorIndex() throws IOException {
        if (this.version < 6) {
            return null;
        }
        if (this.normVectorIndex == null) {
            if (this.normVectorIndexPosition > 0L && this.normVectorIndexSize > 0) {
                this.readNormVectorIndex(new IndexEntry(this.normVectorIndexPosition, this.normVectorIndexSize));
            } else {
                try {
                    this.readNormExpectedValuesAndNormVectorIndex();
                }
                catch (IOException e) {
                    log.warn("Error reading norm vector index.  This could indicate normalization vectors are not present", e);
                }
            }
        }
        return this.normVectorIndex;
    }

    private void readNormVectorIndex(IndexEntry range) throws IOException {
        this.init();
        byte[] data = this.readBytes(range.start, range.size);
        ByteBuffer bp = HicFile.wrap(data);
        this.normVectorIndex = new HashMap<String, IndexEntry>();
        int nEntries = bp.getInt();
        while (nEntries-- > 0) {
            this.parseNormVectorEntry(bp);
        }
    }

    private void readNormExpectedValuesAndNormVectorIndex() throws IOException {
        this.init();
        if (this.normExpectedValueVectorsPosition == null) {
            return;
        }
        long nviStart = this.skipExpectedValues(this.normExpectedValueVectorsPosition);
        byte[] data = this.readBytes(nviStart, 4L);
        if (data == null || data.length == 0) {
            return;
        }
        ByteBuffer bp = HicFile.wrap(data);
        int nEntries = bp.getInt();
        int sizeEstimate = nEntries * 30;
        data = this.readBytes(nviStart + 4L, sizeEstimate);
        this.normVectorIndex = new HashMap<String, IndexEntry>();
        int byteCount = 4;
        ByteBuffer parser = HicFile.wrap(data);
        while (nEntries-- > 0) {
            if (parser.remaining() < 100) {
                sizeEstimate = Math.max(1000, ++nEntries * 30);
                data = this.readBytes(nviStart + (long)(byteCount += parser.position()), sizeEstimate);
                parser = HicFile.wrap(data);
            }
            this.parseNormVectorEntry(parser);
        }
    }

    private long skipExpectedValues(long start) throws IOException {
        int INT_SIZE = 4;
        int DOUBLE_SIZE = 8;
        int FLOAT_SIZE = 4;
        byte[] data = this.readBytes(start, 4L);
        if (data == null) {
            return start;
        }
        ByteBuffer buffer = HicFile.wrap(data);
        int nEntries = buffer.getInt();
        if (nEntries == 0) {
            return start + 4L;
        }
        long currentPosition = start + 4L;
        for (int i = 0; i < nEntries; ++i) {
            byte[] chunkHeader = this.readBytes(currentPosition, 500L);
            if (chunkHeader == null) {
                throw new IOException("Unexpected end of file while reading expected values.");
            }
            ByteBuffer chunkBuffer = HicFile.wrap(chunkHeader);
            HicFile.getString(chunkBuffer);
            HicFile.getString(chunkBuffer);
            chunkBuffer.getInt();
            long nValues = this.version < 9 ? (long)chunkBuffer.getInt() : chunkBuffer.getLong();
            long valuesSize = nValues * (long)(this.version < 9 ? 8 : 4);
            long posAfterValues = currentPosition + (long)chunkBuffer.position() + valuesSize;
            byte[] scaleFactorsHeader = this.readBytes(posAfterValues, 4L);
            if (scaleFactorsHeader == null) {
                throw new IOException("Unexpected end of file while reading scale factors.");
            }
            ByteBuffer scaleFactorsBuffer = HicFile.wrap(scaleFactorsHeader);
            int nChrScaleFactors = scaleFactorsBuffer.getInt();
            long scaleFactorsSize = (long)nChrScaleFactors * (long)(4 + (this.version < 9 ? 8 : 4));
            currentPosition = posAfterValues + 4L + scaleFactorsSize;
        }
        return currentPosition;
    }

    private void parseNormVectorEntry(ByteBuffer parser) {
        String type = HicFile.getString(parser);
        int chrIdx = parser.getInt();
        String unit = HicFile.getString(parser);
        int binSize = parser.getInt();
        long filePosition = parser.getLong();
        long sizeInBytes = this.version < 9 ? (long)parser.getInt() : parser.getLong();
        String key = HicFile.getNormalizationVectorKey(type, chrIdx, unit, binSize);
        if (!this.normalizationTypes.contains(type)) {
            this.normalizationTypes.add(type);
        }
        this.normVectorIndex.put(key, new IndexEntry(filePosition, (int)sizeInBytes));
    }

    private String getFileChrName(String chrName) {
        return this.chrAliasTable.getOrDefault(chrName, chrName);
    }

    public List<String> getNormalizationTypes() {
        try {
            this.getNormVectorIndex();
        }
        catch (IOException e) {
            log.error("Error reading norm vector index", e);
        }
        return this.normalizationTypes;
    }

    private static String getNormalizationVectorKey(String type, int chrIdx, String unit, int resolution) {
        return type + "_" + chrIdx + "_" + unit + "_" + resolution;
    }

    private byte[] readBytes(long position, long size) throws IOException {
        int r;
        if (size <= 0L) {
            return new byte[0];
        }
        byte[] byteArray = new byte[(int)size];
        int read = 0;
        this.fileChannel.seek(position);
        while ((long)read < size && (r = this.fileChannel.read(byteArray, read, (int)(size - (long)read))) >= 0) {
            read += r;
        }
        if (read == 0) {
            return null;
        }
        return byteArray;
    }

    private static String getString(ByteBuffer buffer) {
        byte b;
        ByteArrayOutputStream bis = new ByteArrayOutputStream(1000);
        while ((b = buffer.get()) != 0) {
            bis.write(b);
        }
        return new String(bis.toByteArray());
    }

    public void setNVIString(String nviString) {
        String[] parts = nviString.split(",");
        if (parts.length < 2) {
            log.error("Invalid NVI string: " + nviString);
            this.normVectorIndexPosition = -1L;
            this.normVectorIndexSize = -1;
            return;
        }
        try {
            this.normVectorIndexPosition = Long.parseLong(parts[0]);
            this.normVectorIndexSize = Integer.parseInt(parts[1]);
        }
        catch (NumberFormatException e) {
            log.error("Error parsing NVI string: " + nviString, e);
            this.normVectorIndexPosition = -1L;
            this.normVectorIndexSize = -1;
        }
    }

    private static class BlockCache {
        private Integer resolution = null;
        private LRUCache<String, Block> map = new LRUCache(6);

        private BlockCache() {
        }

        public void set(int resolution, String key, Block value) {
            if (this.resolution == null || this.resolution != resolution) {
                this.map.clear();
                this.resolution = resolution;
            }
            this.map.put(key, value);
        }

        public Block get(int resolution, String key) {
            return this.resolution != null && this.resolution == resolution ? this.map.get(key) : null;
        }

        public boolean has(int resolution, String key) {
            return this.resolution != null && this.resolution == resolution && this.map.containsKey(key);
        }
    }

    record IndexEntry(long start, int size) {
    }

    private static class Block {
        public final int blockNumber;
        public final MatrixZoomData zoomData;
        public final List<ContactRecord> records;
        public final StaticBlockIndex.BlockIndexEntry idx;

        public Block(int blockNumber, MatrixZoomData zd, List<ContactRecord> records, StaticBlockIndex.BlockIndexEntry idx) {
            this.blockNumber = blockNumber;
            this.zoomData = zd;
            this.records = records;
            this.idx = idx;
        }
    }

    static enum DataType {
        FLOAT(4),
        DOUBLE(8);

        private final int byteSize;

        private DataType(int byteSize) {
            this.byteSize = byteSize;
        }

        public int getByteSize() {
            return this.byteSize;
        }
    }
}

