/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.cram.structure.block;

import htsjdk.samtools.cram.CRAMException;
import htsjdk.samtools.cram.common.CRAMVersion;
import htsjdk.samtools.cram.common.CramVersions;
import htsjdk.samtools.cram.compression.ExternalCompressor;
import htsjdk.samtools.cram.compression.GZIPExternalCompressor;
import htsjdk.samtools.cram.io.CRC32InputStream;
import htsjdk.samtools.cram.io.CRC32OutputStream;
import htsjdk.samtools.cram.io.CramInt;
import htsjdk.samtools.cram.io.ITF8;
import htsjdk.samtools.cram.io.InputStreamUtils;
import htsjdk.samtools.cram.structure.CompressorCache;
import htsjdk.samtools.cram.structure.block.BlockCompressionMethod;
import htsjdk.samtools.cram.structure.block.BlockContentType;
import htsjdk.samtools.util.RuntimeIOException;
import htsjdk.utils.ValidationUtils;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

public class Block {
    public static final int NO_CONTENT_ID = 0;
    private final BlockCompressionMethod compressionMethod;
    private final BlockContentType contentType;
    private final int contentId;
    private final byte[] compressedContent;
    private final int uncompressedLength;

    protected Block(BlockCompressionMethod compressionMethod, BlockContentType contentType, int contentId, byte[] compressedContent, int uncompressedLength) {
        this.compressionMethod = compressionMethod;
        this.contentType = contentType;
        this.contentId = contentId;
        this.compressedContent = compressedContent;
        this.uncompressedLength = uncompressedLength;
        if (contentType != BlockContentType.EXTERNAL && contentId != 0) {
            throw new CRAMException("Cannot set a Content ID for non-external blocks.");
        }
    }

    private static Block createRawNonExternalBlock(BlockContentType contentType, byte[] rawContent) {
        if (contentType == BlockContentType.EXTERNAL) {
            throw new CRAMException("Code error: cannot use the non-external factory method for EXTERNAL blocks.");
        }
        return new Block(BlockCompressionMethod.RAW, contentType, 0, rawContent, rawContent.length);
    }

    public static Block createGZIPFileHeaderBlock(byte[] rawContent) {
        return new Block(BlockCompressionMethod.GZIP, BlockContentType.FILE_HEADER, 0, new GZIPExternalCompressor().compress(rawContent, null), rawContent.length);
    }

    public static Block createRawCompressionHeaderBlock(byte[] rawContent) {
        return Block.createRawNonExternalBlock(BlockContentType.COMPRESSION_HEADER, rawContent);
    }

    public static Block createRawSliceHeaderBlock(byte[] rawContent) {
        return Block.createRawNonExternalBlock(BlockContentType.MAPPED_SLICE, rawContent);
    }

    public static Block createRawCoreDataBlock(byte[] rawContent) {
        return Block.createRawNonExternalBlock(BlockContentType.CORE, rawContent);
    }

    public static Block createExternalBlock(BlockCompressionMethod compressionMethod, int contentId, byte[] compressedContent, int uncompressedLength) {
        ValidationUtils.validateArg(contentId >= 0, "Invalid external block content id");
        return new Block(compressionMethod, BlockContentType.EXTERNAL, contentId, compressedContent, uncompressedLength);
    }

    public final BlockCompressionMethod getCompressionMethod() {
        return this.compressionMethod;
    }

    public final BlockContentType getContentType() {
        return this.contentType;
    }

    public int getContentId() {
        return this.contentId;
    }

    public final byte[] getRawContent() {
        ValidationUtils.validateArg(this.getCompressionMethod() == BlockCompressionMethod.RAW, "getRawContent should only be called on blocks with RAW compression method");
        return this.compressedContent;
    }

    public final byte[] getUncompressedContent(CompressorCache compressorCache) {
        ExternalCompressor compressor = compressorCache.getCompressorForMethod(this.compressionMethod, -1);
        byte[] uncompressedContent = compressor.uncompress(this.compressedContent);
        if (uncompressedContent.length != this.uncompressedLength) {
            throw new CRAMException(String.format("Block uncompressed length did not match expected length: %04x vs %04x", this.uncompressedLength, uncompressedContent.length));
        }
        return uncompressedContent;
    }

    public int getUncompressedContentSize() {
        return this.uncompressedLength;
    }

    public final byte[] getCompressedContent() {
        return this.compressedContent;
    }

    public final int getCompressedContentSize() {
        return this.compressedContent.length;
    }

    public static Block read(CRAMVersion cramVersion, InputStream inputStream) {
        boolean v3OrHigher;
        boolean bl = v3OrHigher = cramVersion.getMajor() >= CramVersions.CRAM_v3.getMajor();
        if (v3OrHigher) {
            inputStream = new CRC32InputStream(inputStream);
        }
        try {
            BlockCompressionMethod compressionMethod = BlockCompressionMethod.byId(inputStream.read());
            BlockContentType contentType = BlockContentType.byId(inputStream.read());
            int contentId = ITF8.readUnsignedITF8(inputStream);
            int compressedSize = ITF8.readUnsignedITF8(inputStream);
            int uncompressedSize = ITF8.readUnsignedITF8(inputStream);
            byte[] compressedContent = new byte[compressedSize];
            InputStreamUtils.readFully(inputStream, compressedContent, 0, compressedSize);
            if (v3OrHigher) {
                int actualChecksum = ((CRC32InputStream)inputStream).getCRC32();
                int checksum = CramInt.readInt32(inputStream);
                if (checksum != actualChecksum) {
                    throw new RuntimeException(String.format("Block CRC32 mismatch, actual: %04x expected: %04x", checksum, actualChecksum));
                }
            }
            return new Block(compressionMethod, contentType, contentId, compressedContent, uncompressedSize);
        }
        catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    public final void write(CRAMVersion cramVersion, OutputStream outputStream) {
        try {
            if (cramVersion.getMajor() >= CramVersions.CRAM_v3.getMajor()) {
                CRC32OutputStream crc32OutputStream = new CRC32OutputStream(outputStream);
                this.doWrite(crc32OutputStream);
                outputStream.write(crc32OutputStream.getCrc32_LittleEndian());
            } else {
                this.doWrite(outputStream);
            }
        }
        catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private void doWrite(OutputStream outputStream) throws IOException {
        outputStream.write(this.getCompressionMethod().getMethodId());
        outputStream.write(this.getContentType().getContentTypeId());
        ITF8.writeUnsignedITF8(this.getContentId(), outputStream);
        ITF8.writeUnsignedITF8(this.getCompressedContentSize(), outputStream);
        ITF8.writeUnsignedITF8(this.getUncompressedContentSize(), outputStream);
        outputStream.write(this.getCompressedContent());
    }

    public String toString() {
        return String.format("method=%s, type=%s, id=%d, raw size=%d, compressed size=%d", this.getCompressionMethod().name(), this.getContentType().name(), this.getContentId(), this.getUncompressedContentSize(), this.getCompressedContentSize());
    }
}

