/*
 * Decompiled with CFR 0.152.
 */
package picard.analysis;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileReader;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMRecordIterator;
import htsjdk.samtools.SAMTag;
import htsjdk.samtools.SamPairUtil;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import picard.PicardException;
import picard.analysis.JumpingLibraryMetrics;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.Metrics;
import picard.sam.DuplicationMetrics;

@CommandLineProgramProperties(usage="Computes jumping library metrics.  Gets all data for computation from the firstread in each pair and assumes that the MQ tag is set with the mate's mapping quality.  If the MQ tag is not set, then the program assumes that the mate's mapping quality is >= MINIMUM_MAPPING_QUALITY", usageShort="Produces jumping library metrics for the provided SAM/BAMs", programGroup=Metrics.class)
public class CollectJumpingLibraryMetrics
extends CommandLineProgram {
    @Option(shortName="I", doc="BAM file(s) of reads with duplicates marked")
    public List<File> INPUT = new ArrayList<File>();
    @Option(shortName="O", doc="File to which metrics should be written")
    public File OUTPUT;
    @Option(shortName="MQ", doc="Mapping quality minimum cutoff")
    public Integer MINIMUM_MAPPING_QUALITY = 0;
    @Option(shortName="T", doc="When calculating mean and stdev stop when the bins in the tail of the distribution contain fewer than mode/TAIL_LIMIT items")
    public int TAIL_LIMIT = 10000;
    @Option(doc="Jumps greater than or equal to the greater of this value or 2 times the mode of the outward-facing pairs are considered chimeras")
    public int CHIMERA_KB_MIN = 100000;
    private static final int SAMPLE_FOR_MODE = 50000;

    public static void main(String[] args) {
        System.exit(new CollectJumpingLibraryMetrics().instanceMain(args));
    }

    @Override
    protected int doWork() {
        for (File f : this.INPUT) {
            IOUtil.assertFileIsReadable(f);
        }
        IOUtil.assertFileIsWritable(this.OUTPUT);
        Histogram<Integer> innieHistogram = new Histogram<Integer>();
        Histogram<Integer> outieHistogram = new Histogram<Integer>();
        int fragments = 0;
        int innies = 0;
        int outies = 0;
        int innieDupes = 0;
        int outieDupes = 0;
        int crossChromPairs = 0;
        int superSized = 0;
        int tandemPairs = 0;
        double chimeraSizeMinimum = Math.max(this.getOutieMode(), (double)this.CHIMERA_KB_MIN);
        for (File f : this.INPUT) {
            SAMFileReader reader = new SAMFileReader(f);
            if (reader.getFileHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
                throw new PicardException("SAM file must " + f.getName() + " must be sorted in coordintate order");
            }
            for (SAMRecord sam : reader) {
                if (!sam.getFirstOfPairFlag()) continue;
                if (sam.getReadUnmappedFlag()) {
                    if (!sam.getMateUnmappedFlag()) {
                        ++fragments;
                        continue;
                    }
                    if (sam.getReferenceIndex() != -1) continue;
                    break;
                }
                if (sam.getMateUnmappedFlag()) {
                    ++fragments;
                    continue;
                }
                if (sam.getAttribute(SAMTag.MQ.name()) != null && sam.getIntegerAttribute(SAMTag.MQ.name()) < this.MINIMUM_MAPPING_QUALITY || sam.getMappingQuality() < this.MINIMUM_MAPPING_QUALITY) continue;
                int absInsertSize = Math.abs(sam.getInferredInsertSize());
                if ((double)absInsertSize > chimeraSizeMinimum) {
                    ++superSized;
                    continue;
                }
                if (sam.getMateNegativeStrandFlag() == sam.getReadNegativeStrandFlag()) {
                    ++tandemPairs;
                    continue;
                }
                if (!sam.getMateReferenceIndex().equals(sam.getReferenceIndex())) {
                    ++crossChromPairs;
                    continue;
                }
                SamPairUtil.PairOrientation pairOrientation = SamPairUtil.getPairOrientation(sam);
                if (pairOrientation == SamPairUtil.PairOrientation.RF) {
                    outieHistogram.increment(absInsertSize);
                    ++outies;
                    if (!sam.getDuplicateReadFlag()) continue;
                    ++outieDupes;
                    continue;
                }
                if (pairOrientation == SamPairUtil.PairOrientation.FR) {
                    innieHistogram.increment(absInsertSize);
                    ++innies;
                    if (!sam.getDuplicateReadFlag()) continue;
                    ++innieDupes;
                    continue;
                }
                throw new IllegalStateException("This should never happen");
            }
            reader.close();
        }
        MetricsFile metricsFile = this.getMetricsFile();
        JumpingLibraryMetrics metrics = new JumpingLibraryMetrics();
        metrics.JUMP_PAIRS = outies;
        metrics.JUMP_DUPLICATE_PAIRS = outieDupes;
        metrics.JUMP_DUPLICATE_PCT = outies != 0 ? (double)outieDupes / (double)outies : 0.0;
        metrics.JUMP_LIBRARY_SIZE = outies > 0 && outieDupes > 0 ? DuplicationMetrics.estimateLibrarySize(outies, outies - outieDupes) : 0L;
        outieHistogram.trimByTailLimit(this.TAIL_LIMIT);
        metrics.JUMP_MEAN_INSERT_SIZE = outieHistogram.getMean();
        metrics.JUMP_STDEV_INSERT_SIZE = outieHistogram.getStandardDeviation();
        metrics.NONJUMP_PAIRS = innies;
        metrics.NONJUMP_DUPLICATE_PAIRS = innieDupes;
        metrics.NONJUMP_DUPLICATE_PCT = innies != 0 ? (double)innieDupes / (double)innies : 0.0;
        metrics.NONJUMP_LIBRARY_SIZE = innies > 0 && innieDupes > 0 ? DuplicationMetrics.estimateLibrarySize(innies, innies - innieDupes) : 0L;
        innieHistogram.trimByTailLimit(this.TAIL_LIMIT);
        metrics.NONJUMP_MEAN_INSERT_SIZE = innieHistogram.getMean();
        metrics.NONJUMP_STDEV_INSERT_SIZE = innieHistogram.getStandardDeviation();
        metrics.CHIMERIC_PAIRS = crossChromPairs + superSized + tandemPairs;
        metrics.FRAGMENTS = fragments;
        double totalPairs = (long)(outies + innies) + metrics.CHIMERIC_PAIRS;
        metrics.PCT_JUMPS = totalPairs != 0.0 ? (double)outies / totalPairs : 0.0;
        metrics.PCT_NONJUMPS = totalPairs != 0.0 ? (double)innies / totalPairs : 0.0;
        metrics.PCT_CHIMERAS = totalPairs != 0.0 ? (double)metrics.CHIMERIC_PAIRS / totalPairs : 0.0;
        metricsFile.addMetric(metrics);
        metricsFile.write(this.OUTPUT);
        return 0;
    }

    private double getOutieMode() {
        int samplePerFile = 50000 / this.INPUT.size();
        Histogram<Integer> histo = new Histogram<Integer>();
        for (File f : this.INPUT) {
            SAMFileReader reader = new SAMFileReader(f);
            int sampled = 0;
            SAMRecordIterator it = reader.iterator();
            while (it.hasNext() && sampled < samplePerFile) {
                SAMRecord sam = (SAMRecord)it.next();
                if (!sam.getFirstOfPairFlag()) continue;
                if (sam.getReadUnmappedFlag() && sam.getReferenceIndex() == -1) break;
                if (sam.getReadUnmappedFlag() || sam.getMateUnmappedFlag() || sam.getAttribute(SAMTag.MQ.name()) != null && sam.getIntegerAttribute(SAMTag.MQ.name()) < this.MINIMUM_MAPPING_QUALITY || sam.getMappingQuality() < this.MINIMUM_MAPPING_QUALITY || sam.getMateNegativeStrandFlag() == sam.getReadNegativeStrandFlag() || !sam.getMateReferenceIndex().equals(sam.getReferenceIndex()) || SamPairUtil.getPairOrientation(sam) != SamPairUtil.PairOrientation.RF) continue;
                histo.increment(Math.abs(sam.getInferredInsertSize()));
                ++sampled;
            }
            reader.close();
        }
        return histo.size() > 0 ? histo.getMode() : 0.0;
    }
}

