/*
 * Decompiled with CFR 0.152.
 */
package picard.sam;

import htsjdk.samtools.SAMFileReader;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMUtils;
import htsjdk.samtools.SAMValidationError;
import htsjdk.samtools.fastq.FastqRecord;
import htsjdk.samtools.fastq.FastqWriter;
import htsjdk.samtools.fastq.FastqWriterFactory;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Lazy;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.SamOrBam;

@CommandLineProgramProperties(usage="Extracts read sequences and qualities from the input SAM/BAM file and writes them into the output file in Sanger fastq format. In the RC mode (default is True), if the read is aligned and the alignment is to the reverse strand on the genome, the read's sequence from input SAM file will be reverse-complemented prior to writing it to fastq in order restore correctlythe original read sequence as it was generated by the sequencer.", usageShort="Converts a SAM/BAM into a FASTQ", programGroup=SamOrBam.class)
public class SamToFastq
extends CommandLineProgram {
    @Option(doc="Input SAM/BAM file to extract reads from", shortName="I")
    public File INPUT;
    @Option(shortName="F", doc="Output fastq file (single-end fastq or, if paired, first end of the pair fastq).", mutex={"OUTPUT_PER_RG"})
    public File FASTQ;
    @Option(shortName="F2", doc="Output fastq file (if paired, second end of the pair fastq).", optional=true, mutex={"OUTPUT_PER_RG"})
    public File SECOND_END_FASTQ;
    @Option(shortName="FU", doc="Output fastq file for unpaired reads; may only be provided in paired-fastq mode", optional=true, mutex={"OUTPUT_PER_RG"})
    public File UNPAIRED_FASTQ;
    @Option(shortName="OPRG", doc="Output a fastq file per read group (two fastq files per read group if the group is paired).", optional=true, mutex={"FASTQ", "SECOND_END_FASTQ", "UNPAIRED_FASTQ"})
    public boolean OUTPUT_PER_RG;
    @Option(shortName="ODIR", doc="Directory in which to output the fastq file(s).  Used only when OUTPUT_PER_RG is true.", optional=true)
    public File OUTPUT_DIR;
    @Option(shortName="RC", doc="Re-reverse bases and qualities of reads with negative strand flag set before writing them to fastq", optional=true)
    public boolean RE_REVERSE = true;
    @Option(shortName="INTER", doc="Will generate an interleaved fastq if paired, each line will have /1 or /2 to describe which end it came from")
    public boolean INTERLEAVE = false;
    @Option(shortName="NON_PF", doc="Include non-PF reads from the SAM file into the output FASTQ files. PF means 'passes filtering'. Reads whose 'not passing quality controls' flag is set are non-PF reads.")
    public boolean INCLUDE_NON_PF_READS = false;
    @Option(shortName="CLIP_ATTR", doc="The attribute that stores the position at which the SAM record should be clipped", optional=true)
    public String CLIPPING_ATTRIBUTE;
    @Option(shortName="CLIP_ACT", doc="The action that should be taken with clipped reads: 'X' means the reads and qualities should be trimmed at the clipped position; 'N' means the bases should be changed to Ns in the clipped region; and any integer means that the base qualities should be set to that value in the clipped region.", optional=true)
    public String CLIPPING_ACTION;
    @Option(shortName="R1_TRIM", doc="The number of bases to trim from the beginning of read 1.")
    public int READ1_TRIM = 0;
    @Option(shortName="R1_MAX_BASES", doc="The maximum number of bases to write from read 1 after trimming. If there are fewer than this many bases left after trimming, all will be written.  If this value is null then all bases left after trimming will be written.", optional=true)
    public Integer READ1_MAX_BASES_TO_WRITE;
    @Option(shortName="R2_TRIM", doc="The number of bases to trim from the beginning of read 2.")
    public int READ2_TRIM = 0;
    @Option(shortName="R2_MAX_BASES", doc="The maximum number of bases to write from read 2 after trimming. If there are fewer than this many bases left after trimming, all will be written.  If this value is null then all bases left after trimming will be written.", optional=true)
    public Integer READ2_MAX_BASES_TO_WRITE;
    @Option(doc="If true, include non-primary alignments in the output.  Support of non-primary alignments in SamToFastq is not comprehensive, so there may be exceptions if this is set to true and there are paired reads with non-primary alignments.")
    public boolean INCLUDE_NON_PRIMARY_ALIGNMENTS = false;
    private final Log log = Log.getInstance(SamToFastq.class);

    public static void main(String[] argv) {
        System.exit(new SamToFastq().instanceMain(argv));
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable(this.INPUT);
        SAMFileReader reader = new SAMFileReader(IOUtil.openFileForReading(this.INPUT));
        HashMap<String, SAMRecord> firstSeenMates = new HashMap<String, SAMRecord>();
        FastqWriterFactory factory = new FastqWriterFactory();
        factory.setCreateMd5(this.CREATE_MD5_FILE);
        Map<SAMReadGroupRecord, FastqWriters> writers = this.generateWriters(reader.getFileHeader().getReadGroups(), factory);
        ProgressLogger progress = new ProgressLogger(this.log);
        for (SAMRecord currentRecord : reader) {
            if (currentRecord.isSecondaryOrSupplementary() && !this.INCLUDE_NON_PRIMARY_ALIGNMENTS || currentRecord.getReadFailsVendorQualityCheckFlag() && !this.INCLUDE_NON_PF_READS) continue;
            FastqWriters fq = writers.get(currentRecord.getReadGroup());
            if (currentRecord.getReadPairedFlag()) {
                String currentReadName = currentRecord.getReadName();
                SAMRecord firstRecord = (SAMRecord)firstSeenMates.remove(currentReadName);
                if (firstRecord == null) {
                    firstSeenMates.put(currentReadName, currentRecord);
                } else {
                    this.assertPairedMates(firstRecord, currentRecord);
                    SAMRecord read1 = currentRecord.getFirstOfPairFlag() ? currentRecord : firstRecord;
                    SAMRecord read2 = currentRecord.getFirstOfPairFlag() ? firstRecord : currentRecord;
                    this.writeRecord(read1, 1, fq.getFirstOfPair(), this.READ1_TRIM, this.READ1_MAX_BASES_TO_WRITE);
                    FastqWriter secondOfPairWriter = fq.getSecondOfPair();
                    if (secondOfPairWriter == null) {
                        throw new PicardException("Input contains paired reads but no SECOND_END_FASTQ specified.");
                    }
                    this.writeRecord(read2, 2, secondOfPairWriter, this.READ2_TRIM, this.READ2_MAX_BASES_TO_WRITE);
                }
            } else {
                this.writeRecord(currentRecord, null, fq.getUnpaired(), this.READ1_TRIM, this.READ1_MAX_BASES_TO_WRITE);
            }
            progress.record(currentRecord);
        }
        reader.close();
        for (FastqWriters writerMapping : new HashSet<FastqWriters>(writers.values())) {
            writerMapping.closeAll();
        }
        if (firstSeenMates.size() > 0) {
            SAMUtils.processValidationError(new SAMValidationError(SAMValidationError.Type.MATE_NOT_FOUND, "Found " + firstSeenMates.size() + " unpaired mates", null), this.VALIDATION_STRINGENCY);
        }
        return 0;
    }

    private Map<SAMReadGroupRecord, FastqWriters> generateWriters(List<SAMReadGroupRecord> samReadGroupRecords, final FastqWriterFactory factory) {
        HashMap<SAMReadGroupRecord, FastqWriters> writerMap = new HashMap<SAMReadGroupRecord, FastqWriters>();
        if (!this.OUTPUT_PER_RG) {
            FastqWriter secondOfPairWriter;
            IOUtil.assertFileIsWritable(this.FASTQ);
            FastqWriter firstOfPairWriter = factory.newWriter(this.FASTQ);
            if (this.INTERLEAVE) {
                secondOfPairWriter = firstOfPairWriter;
            } else if (this.SECOND_END_FASTQ != null) {
                IOUtil.assertFileIsWritable(this.SECOND_END_FASTQ);
                secondOfPairWriter = factory.newWriter(this.SECOND_END_FASTQ);
            } else {
                secondOfPairWriter = null;
            }
            FastqWriter unpairedWriter = this.UNPAIRED_FASTQ == null ? firstOfPairWriter : factory.newWriter(this.UNPAIRED_FASTQ);
            FastqWriters fastqWriters = new FastqWriters(firstOfPairWriter, secondOfPairWriter, unpairedWriter);
            writerMap.put(null, fastqWriters);
            for (SAMReadGroupRecord rg : samReadGroupRecords) {
                writerMap.put(rg, fastqWriters);
            }
        } else {
            for (final SAMReadGroupRecord rg : samReadGroupRecords) {
                final FastqWriter firstOfPairWriter = factory.newWriter(this.makeReadGroupFile(rg, "_1"));
                Lazy lazySecondOfPairWriter = new Lazy((Lazy.LazyInitializer)new Lazy.LazyInitializer<FastqWriter>(){

                    @Override
                    public FastqWriter make() {
                        return SamToFastq.this.INTERLEAVE ? firstOfPairWriter : factory.newWriter(SamToFastq.this.makeReadGroupFile(rg, "_2"));
                    }
                });
                writerMap.put(rg, new FastqWriters(firstOfPairWriter, lazySecondOfPairWriter, firstOfPairWriter));
            }
        }
        return writerMap;
    }

    private File makeReadGroupFile(SAMReadGroupRecord readGroup, String preExtSuffix) {
        String fileName = readGroup.getPlatformUnit();
        if (fileName == null) {
            fileName = readGroup.getReadGroupId();
        }
        fileName = IOUtil.makeFileNameSafe(fileName);
        if (preExtSuffix != null) {
            fileName = fileName + preExtSuffix;
        }
        fileName = fileName + ".fastq";
        File result = this.OUTPUT_DIR != null ? new File(this.OUTPUT_DIR, fileName) : new File(fileName);
        IOUtil.assertFileIsWritable(result);
        return result;
    }

    void writeRecord(SAMRecord read, Integer mateNumber, FastqWriter writer, int basesToTrim, Integer maxBasesToWrite) {
        Integer clipPoint;
        String seqHeader = mateNumber == null ? read.getReadName() : read.getReadName() + "/" + mateNumber;
        String readString = read.getReadString();
        String baseQualities = read.getBaseQualityString();
        if (this.CLIPPING_ATTRIBUTE != null && (clipPoint = (Integer)read.getAttribute(this.CLIPPING_ATTRIBUTE)) != null) {
            if (this.CLIPPING_ACTION.equalsIgnoreCase("X")) {
                readString = this.clip(readString, clipPoint, null, !read.getReadNegativeStrandFlag());
                baseQualities = this.clip(baseQualities, clipPoint, null, !read.getReadNegativeStrandFlag());
            } else if (this.CLIPPING_ACTION.equalsIgnoreCase("N")) {
                readString = this.clip(readString, clipPoint, Character.valueOf('N'), !read.getReadNegativeStrandFlag());
            } else {
                char newQual = SAMUtils.phredToFastq(new byte[]{(byte)Integer.parseInt(this.CLIPPING_ACTION)}).charAt(0);
                baseQualities = this.clip(baseQualities, clipPoint, Character.valueOf(newQual), !read.getReadNegativeStrandFlag());
            }
        }
        if (this.RE_REVERSE && read.getReadNegativeStrandFlag()) {
            readString = SequenceUtil.reverseComplement(readString);
            baseQualities = StringUtil.reverseString(baseQualities);
        }
        if (basesToTrim > 0) {
            readString = readString.substring(basesToTrim);
            baseQualities = baseQualities.substring(basesToTrim);
        }
        if (maxBasesToWrite != null && maxBasesToWrite < readString.length()) {
            readString = readString.substring(0, maxBasesToWrite);
            baseQualities = baseQualities.substring(0, maxBasesToWrite);
        }
        writer.write(new FastqRecord(seqHeader, readString, "", baseQualities));
    }

    private String clip(String src, int point, Character replacement, boolean posStrand) {
        String result;
        block4: {
            int len = src.length();
            String string = result = posStrand ? src.substring(0, point - 1) : src.substring(len - point + 1);
            if (replacement == null) break block4;
            if (posStrand) {
                for (int i = point; i <= len; ++i) {
                    result = result + replacement;
                }
            } else {
                for (int i = 0; i <= len - point; ++i) {
                    result = replacement + result;
                }
            }
        }
        return result;
    }

    private void assertPairedMates(SAMRecord record1, SAMRecord record2) {
        if (!(record1.getFirstOfPairFlag() && record2.getSecondOfPairFlag() || record2.getFirstOfPairFlag() && record1.getSecondOfPairFlag())) {
            throw new PicardException("Illegal mate state: " + record1.getReadName());
        }
    }

    @Override
    protected String[] customCommandLineValidation() {
        if (this.INTERLEAVE && this.SECOND_END_FASTQ != null) {
            return new String[]{"Cannot set INTERLEAVE to true and pass in a SECOND_END_FASTQ"};
        }
        if (this.UNPAIRED_FASTQ != null && this.SECOND_END_FASTQ == null) {
            return new String[]{"UNPAIRED_FASTQ may only be set when also emitting read1 and read2 fastqs (so SECOND_END_FASTQ must also be set)."};
        }
        if (this.CLIPPING_ATTRIBUTE != null && this.CLIPPING_ACTION == null || this.CLIPPING_ATTRIBUTE == null && this.CLIPPING_ACTION != null) {
            return new String[]{"Both or neither of CLIPPING_ATTRIBUTE and CLIPPING_ACTION should be set."};
        }
        if (this.CLIPPING_ACTION != null && !this.CLIPPING_ACTION.equals("N") && !this.CLIPPING_ACTION.equals("X")) {
            try {
                Integer.parseInt(this.CLIPPING_ACTION);
            }
            catch (NumberFormatException nfe) {
                return new String[]{"CLIPPING ACTION must be one of: N, X, or an integer"};
            }
        }
        if (this.OUTPUT_PER_RG && this.OUTPUT_DIR == null || !this.OUTPUT_PER_RG && this.OUTPUT_DIR != null) {
            return new String[]{"If OUTPUT_PER_RG is true, then OUTPUT_DIR should be set. If "};
        }
        return null;
    }

    static class FastqWriters {
        private final FastqWriter firstOfPair;
        private final FastqWriter unpaired;
        private final Lazy<FastqWriter> secondOfPair;

        private FastqWriters(FastqWriter firstOfPair, Lazy<FastqWriter> secondOfPair, FastqWriter unpaired) {
            this.firstOfPair = firstOfPair;
            this.unpaired = unpaired;
            this.secondOfPair = secondOfPair;
        }

        private FastqWriters(FastqWriter firstOfPair, final FastqWriter secondOfPair, FastqWriter unpaired) {
            this(firstOfPair, new Lazy((Lazy.LazyInitializer)new Lazy.LazyInitializer<FastqWriter>(){

                @Override
                public FastqWriter make() {
                    return secondOfPair;
                }
            }), unpaired);
        }

        public FastqWriter getFirstOfPair() {
            return this.firstOfPair;
        }

        public FastqWriter getSecondOfPair() {
            return this.secondOfPair.get();
        }

        public FastqWriter getUnpaired() {
            return this.unpaired;
        }

        public void closeAll() {
            HashSet<FastqWriter> fastqWriters = new HashSet<FastqWriter>();
            fastqWriters.add(this.firstOfPair);
            fastqWriters.add(this.unpaired);
            if (this.secondOfPair.isInitialized()) {
                fastqWriters.add(this.secondOfPair.get());
            }
            for (FastqWriter fastqWriter : fastqWriters) {
                fastqWriter.close();
            }
        }
    }
}

