/*
 * Decompiled with CFR 0.152.
 */
package picard.illumina;

import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProcessExecutor;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.Option;
import picard.cmdline.Usage;
import picard.illumina.IlluminaBasecallsConverter;
import picard.illumina.parser.IlluminaDataProviderFactory;
import picard.illumina.parser.IlluminaDataType;
import picard.illumina.parser.IlluminaFileUtil;
import picard.illumina.parser.OutputMapping;
import picard.illumina.parser.ParameterizedFileUtil;
import picard.illumina.parser.ReadStructure;

public class CheckIlluminaDirectory
extends CommandLineProgram {
    private static final Log log = Log.getInstance(CheckIlluminaDirectory.class);
    @Usage
    public String USAGE = this.getStandardUsagePreamble() + "Check that the files to provide the data specified by DATA_TYPES are available, exist, and are reasonably sized for every tile/cycle.  " + "Reasonably sized means non-zero sized for files that exist per tile and equal size for binary files that exist per cycle/per tile. " + "CheckIlluminaDirectory DOES NOT check that the individual records in a file are well-formed.\n";
    @Option(doc="The basecalls output directory. ", shortName="B")
    public File BASECALLS_DIR;
    @Option(doc="The data types that should be checked for each tile/cycle.  If no values are provided then the data types checked are those required by IlluminaBaseCallsToSam (which is a superset of those used in ExtractIlluminaBarcodes).  These data types vary slightly depending onwhether or not the run is barcoded so READ_STRUCTURE should be the same as that which will be passed to IlluminaBasecallsToSam.  If this option is left unspecified then both ExtractIlluminaBarcodes and IlluminaBaseCallsToSam should complete successfully UNLESS the individual records of the files themselves are spurious. ", shortName="DT", optional=true)
    public final Set<IlluminaDataType> DATA_TYPES = new TreeSet<IlluminaDataType>();
    @Option(doc="A description of the logical structure of clusters in an Illumina Run, i.e. a description of the structure IlluminaBasecallsToSam assumes the  data to be in. It should consist of integer/character pairs describing the number of cycles and the type of those cycles (B for Barcode, T for Template, and S for skip).  E.g. If the input data consists of 80 base clusters and we provide a read structure of \"36T8B8S28T\" then, before being converted to SAM records those bases will be split into 4 reads where read one consists of 36 cycles of template, read two consists of 8 cycles of barcode, read three will be an 8 base read of skipped cycles and read four is another 28 cycle template read.  The read consisting of skipped cycles would NOT be included in output SAM/BAM file read groups.  Note:  If you want to check whether or not a future IlluminaBasecallsToSam or ExtractIlluminaBarcodes run will fail then be sure to use the exact same READ_STRUCTURE that you would pass to these programs for this run.", shortName="RS")
    public String READ_STRUCTURE;
    @Option(doc="The number of the lane(s) to check. ", shortName="L", minElements=1)
    public List<Integer> LANES;
    @Option(doc="The number(s) of the tile(s) to check. ", shortName="T", optional=true)
    public List<Integer> TILE_NUMBERS;
    @Option(doc="A flag to determine whether or not to create fake versions of the missing files.", shortName="F", optional=true)
    public Boolean FAKE_FILES = false;
    @Option(doc="A flag to create symlinks to the loc file for the X Ten for each tile.", shortName="X", optional=true)
    public Boolean LINK_LOCS = false;

    public static void main(String[] argv) {
        new CheckIlluminaDirectory().instanceMainWithExit(argv);
    }

    @Override
    protected int doWork() {
        ReadStructure readStructure = new ReadStructure(this.READ_STRUCTURE);
        if (this.DATA_TYPES.isEmpty()) {
            this.DATA_TYPES.addAll(Arrays.asList(IlluminaBasecallsConverter.DATA_TYPES_NO_BARCODE));
        }
        ArrayList<Integer> failingLanes = new ArrayList<Integer>();
        int totalFailures = 0;
        int[] expectedCycles = new OutputMapping(readStructure).getOutputCycles();
        log.info("Checking lanes(" + StringUtil.join(",", this.LANES) + " in basecalls directory (" + this.BASECALLS_DIR.getAbsolutePath() + ")\n");
        log.info("Expected cycles: " + StringUtil.intValuesToString(expectedCycles));
        for (Integer lane : this.LANES) {
            IlluminaFileUtil fileUtil = new IlluminaFileUtil(this.BASECALLS_DIR, lane);
            List<Integer> expectedTiles = fileUtil.getExpectedTiles();
            if (!this.TILE_NUMBERS.isEmpty()) {
                expectedTiles.retainAll(this.TILE_NUMBERS);
            }
            if (this.LINK_LOCS.booleanValue()) {
                this.createLocFileSymlinks(fileUtil, lane);
                fileUtil = new IlluminaFileUtil(this.BASECALLS_DIR, lane);
            }
            log.info("Checking lane " + lane);
            log.info("Expected tiles: " + StringUtil.join(", ", expectedTiles));
            int numFailures = CheckIlluminaDirectory.verifyLane(fileUtil, expectedTiles, expectedCycles, this.DATA_TYPES, this.FAKE_FILES);
            if (numFailures > 0) {
                log.info("Lane " + lane + " FAILED " + " Total Errors: " + numFailures);
                failingLanes.add(lane);
                totalFailures += numFailures;
                continue;
            }
            log.info("Lane " + lane + " SUCCEEDED ");
        }
        int status = 0;
        if (totalFailures == 0) {
            log.info("SUCCEEDED!  All required files are present and non-empty.");
        } else {
            status = totalFailures;
            log.info("FAILED! There were " + totalFailures + " in the following lanes: " + StringUtil.join(", ", failingLanes));
        }
        return status;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void createLocFileSymlinks(IlluminaFileUtil fileUtil, int lane) {
        File baseFile = new File(this.BASECALLS_DIR.getParentFile().getAbsolutePath() + File.separator + "s.locs");
        File newFileBase = new File(baseFile.getParent() + File.separator + IlluminaFileUtil.longLaneStr(lane) + File.separator);
        if (!baseFile.exists()) throw new PicardException(String.format("Locations file %s does not exist.", baseFile.getAbsolutePath()));
        boolean success = true;
        if (!newFileBase.exists()) {
            success = newFileBase.mkdirs();
        }
        if (!success) throw new PicardException(String.format("Could not create lane directory: %s.", newFileBase.getAbsolutePath()));
        for (Integer tile : fileUtil.getExpectedTiles()) {
            String newName = newFileBase + File.separator + String.format("s_%d_%d.locs", lane, tile);
            ProcessExecutor.ExitStatusAndOutput output = ProcessExecutor.executeAndReturnInterleavedOutput(new String[]{"ln", "-fs", baseFile.getAbsolutePath(), newName});
            if (output.exitStatus == 0) continue;
            throw new PicardException("Could not create symlink: " + output.stdout);
        }
    }

    private static final int verifyLane(IlluminaFileUtil fileUtil, List<Integer> expectedTiles, int[] cycles, Set<IlluminaDataType> dataTypes, boolean fakeFiles) {
        if (expectedTiles.isEmpty()) {
            throw new PicardException("0 input tiles were specified!  Check to make sure this lane is in the InterOp file!");
        }
        if (cycles.length == 0) {
            throw new PicardException("0 output cycles were specified!");
        }
        int numFailures = 0;
        Map<IlluminaFileUtil.SupportedIlluminaFormat, Set<IlluminaDataType>> formatToDataTypes = IlluminaDataProviderFactory.determineFormats(dataTypes, fileUtil);
        Set<IlluminaDataType> unmatchedDataTypes = IlluminaDataProviderFactory.findUnmatchedTypes(dataTypes, formatToDataTypes);
        if (!unmatchedDataTypes.isEmpty()) {
            if (fakeFiles) {
                for (IlluminaDataType dataType : unmatchedDataTypes) {
                    IlluminaFileUtil.SupportedIlluminaFormat format = IlluminaDataProviderFactory.findPreferredFormat(dataType, fileUtil);
                    fileUtil.getUtil(format).fakeFiles(expectedTiles, cycles, format);
                }
            }
            log.info("Could not find a format with available files for the following data types: " + StringUtil.join(", ", new ArrayList<IlluminaDataType>(unmatchedDataTypes)));
            numFailures += unmatchedDataTypes.size();
        }
        for (IlluminaFileUtil.SupportedIlluminaFormat format : formatToDataTypes.keySet()) {
            ParameterizedFileUtil util = fileUtil.getUtil(format);
            List<String> failures = util.verify(expectedTiles, cycles);
            if (!failures.isEmpty() && fakeFiles) {
                util.fakeFiles(expectedTiles, cycles, format);
            }
            numFailures += failures.size();
            for (String failure : failures) {
                log.info(failure);
            }
        }
        return numFailures;
    }

    @Override
    protected String[] customCommandLineValidation() {
        IOUtil.assertDirectoryIsReadable(this.BASECALLS_DIR);
        ArrayList<String> errors = new ArrayList<String>();
        for (Integer lane : this.LANES) {
            if (lane >= 1) continue;
            errors.add("LANES must be greater than or equal to 1.  LANES passed in " + StringUtil.join(", ", this.LANES));
            break;
        }
        if (errors.isEmpty()) {
            return null;
        }
        return errors.toArray(new String[errors.size()]);
    }
}

