/*
 * Decompiled with CFR 0.152.
 */
package picard.analysis;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.util.Arrays;
import java.util.List;
import picard.PicardException;
import picard.analysis.SinglePassSamProgram;
import picard.cmdline.Option;
import picard.util.RExecutor;

public class MeanQualityByCycle
extends SinglePassSamProgram {
    public final String USAGE = this.getStandardUsagePreamble() + "Program to generate a data table and chart of " + "mean quality by cycle from a SAM or BAM file.  Works best on a single lane/run of data, but can be applied to" + "merged BAMs.";
    @Option(shortName="CHART", doc="A file (with .pdf extension) to write the chart to.")
    public File CHART_OUTPUT;
    @Option(doc="If set to true, calculate mean quality over aligned reads only.")
    public boolean ALIGNED_READS_ONLY = false;
    @Option(doc="If set to true calculate mean quality over PF reads only.")
    public boolean PF_READS_ONLY = false;
    private final HistogramGenerator q = new HistogramGenerator(false);
    private final HistogramGenerator oq = new HistogramGenerator(true);
    private String plotSubtitle = "";
    private final Log log = Log.getInstance(MeanQualityByCycle.class);

    public static void main(String[] args) {
        System.exit(new MeanQualityByCycle().instanceMain(args));
    }

    @Override
    protected void setup(SAMFileHeader header, File samFile) {
        IOUtil.assertFileIsWritable(this.CHART_OUTPUT);
        List<SAMReadGroupRecord> readGroups = header.getReadGroups();
        if (readGroups.size() == 1) {
            this.plotSubtitle = StringUtil.asEmptyIfNull(readGroups.get(0).getLibrary());
        }
    }

    @Override
    protected void acceptRead(SAMRecord rec, ReferenceSequence ref) {
        if (this.PF_READS_ONLY && rec.getReadFailsVendorQualityCheckFlag()) {
            return;
        }
        if (this.ALIGNED_READS_ONLY && rec.getReadUnmappedFlag()) {
            return;
        }
        if (rec.isSecondaryOrSupplementary()) {
            return;
        }
        this.q.addRecord(rec);
        this.oq.addRecord(rec);
    }

    @Override
    protected void finish() {
        MetricsFile metrics = this.getMetricsFile();
        metrics.addHistogram(this.q.getMeanQualityHistogram());
        if (!this.oq.isEmpty()) {
            metrics.addHistogram(this.oq.getMeanQualityHistogram());
        }
        metrics.write(this.OUTPUT);
        if (this.q.isEmpty() && this.oq.isEmpty()) {
            this.log.warn("No valid bases found in input file. No plot will be produced.");
        } else {
            int rResult = RExecutor.executeFromClasspath("picard/analysis/meanQualityByCycle.R", this.OUTPUT.getAbsolutePath(), this.CHART_OUTPUT.getAbsolutePath(), this.INPUT.getName(), this.plotSubtitle);
            if (rResult != 0) {
                throw new PicardException("R script meanQualityByCycle.R failed with return code " + rResult);
            }
        }
    }

    private static class HistogramGenerator {
        final boolean useOriginalQualities;
        int maxLengthSoFar = 0;
        double[] firstReadTotalsByCycle = new double[this.maxLengthSoFar];
        long[] firstReadCountsByCycle = new long[this.maxLengthSoFar];
        double[] secondReadTotalsByCycle = new double[this.maxLengthSoFar];
        long[] secondReadCountsByCycle = new long[this.maxLengthSoFar];

        private HistogramGenerator(boolean useOriginalQualities) {
            this.useOriginalQualities = useOriginalQualities;
        }

        void addRecord(SAMRecord rec) {
            byte[] quals;
            byte[] byArray = quals = this.useOriginalQualities ? rec.getOriginalBaseQualities() : rec.getBaseQualities();
            if (quals == null) {
                return;
            }
            int length = quals.length;
            boolean rc = rec.getReadNegativeStrandFlag();
            this.ensureArraysBigEnough(length + 1);
            for (int i2 = 0; i2 < length; ++i2) {
                int cycle;
                int n2 = cycle = rc ? length - i2 : i2 + 1;
                if (rec.getReadPairedFlag() && rec.getSecondOfPairFlag()) {
                    int n3 = cycle;
                    this.secondReadTotalsByCycle[n3] = this.secondReadTotalsByCycle[n3] + (double)quals[i2];
                    int n4 = cycle;
                    this.secondReadCountsByCycle[n4] = this.secondReadCountsByCycle[n4] + 1L;
                    continue;
                }
                int n5 = cycle;
                this.firstReadTotalsByCycle[n5] = this.firstReadTotalsByCycle[n5] + (double)quals[i2];
                int n6 = cycle;
                this.firstReadCountsByCycle[n6] = this.firstReadCountsByCycle[n6] + 1L;
            }
        }

        private void ensureArraysBigEnough(int length) {
            if (length > this.maxLengthSoFar) {
                this.firstReadTotalsByCycle = Arrays.copyOf(this.firstReadTotalsByCycle, length);
                this.firstReadCountsByCycle = Arrays.copyOf(this.firstReadCountsByCycle, length);
                this.secondReadTotalsByCycle = Arrays.copyOf(this.secondReadTotalsByCycle, length);
                this.secondReadCountsByCycle = Arrays.copyOf(this.secondReadCountsByCycle, length);
                this.maxLengthSoFar = length;
            }
        }

        Histogram<Integer> getMeanQualityHistogram() {
            String label = this.useOriginalQualities ? "MEAN_ORIGINAL_QUALITY" : "MEAN_QUALITY";
            Histogram<Integer> meanQualities = new Histogram<Integer>("CYCLE", label);
            int firstReadLength = 0;
            for (int cycle = 0; cycle < this.firstReadTotalsByCycle.length; ++cycle) {
                if (!(this.firstReadTotalsByCycle[cycle] > 0.0)) continue;
                meanQualities.increment(cycle, this.firstReadTotalsByCycle[cycle] / (double)this.firstReadCountsByCycle[cycle]);
                firstReadLength = cycle;
            }
            for (int i2 = 0; i2 < this.secondReadTotalsByCycle.length; ++i2) {
                if (this.secondReadCountsByCycle[i2] <= 0L) continue;
                int cycle = firstReadLength + i2;
                meanQualities.increment(cycle, this.secondReadTotalsByCycle[i2] / (double)this.secondReadCountsByCycle[i2]);
            }
            return meanQualities;
        }

        boolean isEmpty() {
            return this.maxLengthSoFar == 0;
        }
    }
}

