/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.haplotype;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import net.sf.samtools.Cigar;
import net.sf.samtools.CigarElement;
import org.apache.commons.lang.ArrayUtils;
import org.apache.log4j.Logger;
import org.broadinstitute.sting.utils.BaseUtils;
import org.broadinstitute.sting.utils.GenomeLoc;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.haplotype.Haplotype;
import org.broadinstitute.sting.utils.sam.AlignmentUtils;
import org.broadinstitute.variant.variantcontext.Allele;
import org.broadinstitute.variant.variantcontext.VariantContext;
import org.broadinstitute.variant.variantcontext.VariantContextBuilder;

public class EventMap
extends TreeMap<Integer, VariantContext> {
    private static final Logger logger = Logger.getLogger(EventMap.class);
    protected static final int MIN_NUMBER_OF_EVENTS_TO_COMBINE_INTO_BLOCK_SUBSTITUTION = 3;
    public static final Allele SYMBOLIC_UNASSEMBLED_EVENT_ALLELE = Allele.create("<UNASSEMBLED_EVENT>", false);
    private final Haplotype haplotype;
    private final byte[] ref;
    private final GenomeLoc refLoc;
    private final String sourceNameToAdd;

    public EventMap(Haplotype haplotype, byte[] ref, GenomeLoc refLoc, String sourceNameToAdd) {
        this.haplotype = haplotype;
        this.ref = ref;
        this.refLoc = refLoc;
        this.sourceNameToAdd = sourceNameToAdd;
        this.processCigarForInitialEvents();
    }

    protected EventMap(Collection<VariantContext> stateForTesting) {
        this.haplotype = null;
        this.ref = null;
        this.refLoc = null;
        this.sourceNameToAdd = null;
        for (VariantContext vc : stateForTesting) {
            this.addVC(vc);
        }
    }

    protected void processCigarForInitialEvents() {
        Cigar cigar = this.haplotype.getCigar();
        byte[] alignment = this.haplotype.getBases();
        int refPos = this.haplotype.getAlignmentStartHapwrtRef();
        if (refPos < 0) {
            return;
        }
        int alignmentPos = 0;
        block6: for (int cigarIndex = 0; cigarIndex < cigar.numCigarElements(); ++cigarIndex) {
            CigarElement ce = cigar.getCigarElement(cigarIndex);
            int elementLength = ce.getLength();
            switch (ce.getOperator()) {
                case I: {
                    if (refPos > 0) {
                        ArrayList<Allele> insertionAlleles = new ArrayList<Allele>();
                        int insertionStart = this.refLoc.getStart() + refPos - 1;
                        byte refByte = this.ref[refPos - 1];
                        if (BaseUtils.isRegularBase(refByte)) {
                            insertionAlleles.add(Allele.create(refByte, true));
                        }
                        if (cigarIndex != 0 && cigarIndex != cigar.getCigarElements().size() - 1) {
                            byte[] insertionBases = new byte[]{};
                            insertionBases = ArrayUtils.add((byte[])insertionBases, (byte)this.ref[refPos - 1]);
                            if (BaseUtils.isAllRegularBases(insertionBases = ArrayUtils.addAll((byte[])insertionBases, (byte[])Arrays.copyOfRange(alignment, alignmentPos, alignmentPos + elementLength)))) {
                                insertionAlleles.add(Allele.create(insertionBases, false));
                            }
                        }
                        if (insertionAlleles.size() == 2) {
                            this.addVC(new VariantContextBuilder(this.sourceNameToAdd, this.refLoc.getContig(), insertionStart, insertionStart, insertionAlleles).make());
                        }
                    }
                    alignmentPos += elementLength;
                    continue block6;
                }
                case S: {
                    alignmentPos += elementLength;
                    continue block6;
                }
                case D: {
                    if (refPos > 0) {
                        byte[] deletionBases = Arrays.copyOfRange(this.ref, refPos - 1, refPos + elementLength);
                        ArrayList<Allele> deletionAlleles = new ArrayList<Allele>();
                        int deletionStart = this.refLoc.getStart() + refPos - 1;
                        byte refByte = this.ref[refPos - 1];
                        if (BaseUtils.isRegularBase(refByte) && BaseUtils.isAllRegularBases(deletionBases)) {
                            deletionAlleles.add(Allele.create(deletionBases, true));
                            deletionAlleles.add(Allele.create(refByte, false));
                            this.addVC(new VariantContextBuilder(this.sourceNameToAdd, this.refLoc.getContig(), deletionStart, deletionStart + elementLength, deletionAlleles).make());
                        }
                    }
                    refPos += elementLength;
                    continue block6;
                }
                case M: 
                case EQ: 
                case X: {
                    for (int iii = 0; iii < elementLength; ++iii) {
                        byte refByte = this.ref[refPos];
                        byte altByte = alignment[alignmentPos];
                        if (refByte != altByte && BaseUtils.isRegularBase(refByte) && BaseUtils.isRegularBase(altByte)) {
                            ArrayList<Allele> snpAlleles = new ArrayList<Allele>();
                            snpAlleles.add(Allele.create(refByte, true));
                            snpAlleles.add(Allele.create(altByte, false));
                            this.addVC(new VariantContextBuilder(this.sourceNameToAdd, this.refLoc.getContig(), this.refLoc.getStart() + refPos, this.refLoc.getStart() + refPos, snpAlleles).make());
                        }
                        ++refPos;
                        ++alignmentPos;
                    }
                    continue block6;
                }
                default: {
                    throw new ReviewedStingException("Unsupported cigar operator created during SW alignment: " + (Object)((Object)ce.getOperator()));
                }
            }
        }
    }

    protected void addVC(VariantContext vc) {
        this.addVC(vc, true);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    protected void addVC(VariantContext vc, boolean merge) {
        if (vc == null) {
            throw new IllegalArgumentException("vc cannot be null");
        }
        if (this.containsKey(vc.getStart())) {
            if (!merge) throw new IllegalStateException("Will not merge previously bound variant contexts as merge is false at " + vc);
            VariantContext prev = (VariantContext)this.get(vc.getStart());
            this.put(vc.getStart(), this.makeBlock(prev, vc));
            return;
        } else {
            this.put(vc.getStart(), vc);
        }
    }

    protected VariantContext makeBlock(VariantContext vc1, VariantContext vc2) {
        Allele alt;
        Allele ref;
        if (vc1.getStart() != vc2.getStart()) {
            throw new IllegalArgumentException("vc1 and 2 must have the same start but got " + vc1 + " and " + vc2);
        }
        if (!vc1.isBiallelic()) {
            throw new IllegalArgumentException("vc1 must be biallelic");
        }
        if (!vc1.isSNP()) {
            if (!(vc1.isSimpleDeletion() && vc2.isSimpleInsertion() || vc1.isSimpleInsertion() && vc2.isSimpleDeletion())) {
                throw new IllegalArgumentException("Can only merge single insertion with deletion (or vice versa) but got " + vc1 + " merging with " + vc2);
            }
        } else if (vc2.isSNP()) {
            throw new IllegalArgumentException("vc1 is " + vc1 + " but vc2 is a SNP, which implies there's been some terrible bug in the cigar " + vc2);
        }
        VariantContextBuilder b2 = new VariantContextBuilder(vc1);
        if (vc1.isSNP()) {
            if (vc1.getReference().equals(vc2.getReference())) {
                ref = vc1.getReference();
                alt = Allele.create(vc1.getAlternateAllele(0).getDisplayString() + vc2.getAlternateAllele(0).getDisplayString().substring(1), false);
            } else {
                ref = vc2.getReference();
                alt = vc1.getAlternateAllele(0);
                b2.stop(vc2.getEnd());
            }
        } else {
            VariantContext insertion = vc1.isSimpleInsertion() ? vc1 : vc2;
            VariantContext deletion = vc1.isSimpleInsertion() ? vc2 : vc1;
            ref = deletion.getReference();
            alt = insertion.getAlternateAllele(0);
            b2.stop(deletion.getEnd());
        }
        return b2.alleles((Collection<Allele>)Arrays.asList(ref, alt)).make();
    }

    @Requires(value={"getNumberOfEvents() > 0"})
    protected void replaceClumpedEventsWithBlockSubstititions() {
        if (this.getNumberOfEvents() >= 3) {
            int lastStart = -1;
            boolean foundOne = true;
            block0: while (foundOne) {
                foundOne = false;
                for (VariantContext vc : this.getVariantContexts()) {
                    if (vc.getStart() <= lastStart) continue;
                    lastStart = vc.getStart();
                    List<VariantContext> neighborhood = this.getNeighborhood(vc, 10);
                    if (!this.updateToBlockSubstitutionIfBetter(neighborhood)) continue;
                    foundOne = true;
                    continue block0;
                }
            }
        }
    }

    protected boolean updateToBlockSubstitutionIfBetter(List<VariantContext> neighbors) {
        if (neighbors.size() < 3) {
            return false;
        }
        VariantContext first = neighbors.get(0);
        int refStartOffset = first.getStart() - this.refLoc.getStart();
        int refEndOffset = neighbors.get(neighbors.size() - 1).getEnd() - this.refLoc.getStart();
        byte[] refBases = Arrays.copyOfRange(this.ref, refStartOffset, refEndOffset + 1);
        byte[] hapBases = AlignmentUtils.getBasesCoveringRefInterval(refStartOffset, refEndOffset, this.haplotype.getBases(), this.haplotype.getAlignmentStartHapwrtRef(), this.haplotype.getCigar());
        VariantContextBuilder builder = new VariantContextBuilder(first);
        builder.stop(first.getStart() + refBases.length - 1);
        builder.alleles((Collection<Allele>)Arrays.asList(Allele.create(refBases, true), Allele.create(hapBases)));
        VariantContext block = builder.make();
        for (VariantContext merged : neighbors) {
            if (this.remove(merged.getStart()) != null) continue;
            throw new IllegalArgumentException("Expected to remove variant context from the event map but remove said there wasn't any element there: " + merged);
        }
        logger.info("Transforming into block substitution at " + block);
        this.addVC(block, false);
        return true;
    }

    @Requires(value={"leftMost != null", "maxBPBetweenEvents >= 0"})
    @Ensures(value={"result != null", "! result.isEmpty()"})
    protected List<VariantContext> getNeighborhood(VariantContext leftMost, int maxBPBetweenEvents) {
        LinkedList<VariantContext> neighbors = new LinkedList<VariantContext>();
        VariantContext left = leftMost;
        for (VariantContext vc : this.getVariantContexts()) {
            if (vc.getStart() < leftMost.getStart() || vc.getStart() - left.getEnd() >= maxBPBetweenEvents) continue;
            neighbors.add(vc);
            left = vc;
        }
        return neighbors;
    }

    public Set<Integer> getStartPositions() {
        return this.keySet();
    }

    public Collection<VariantContext> getVariantContexts() {
        return this.values();
    }

    public int getNumberOfEvents() {
        return this.size();
    }

    @Override
    public String toString() {
        StringBuilder b2 = new StringBuilder("EventMap{");
        for (VariantContext vc : this.getVariantContexts()) {
            b2.append(String.format("%s:%d-%d %s,", vc.getChr(), vc.getStart(), vc.getEnd(), vc.getAlleles()));
        }
        b2.append("}");
        return b2.toString();
    }

    public static TreeSet<Integer> buildEventMapsForHaplotypes(List<Haplotype> haplotypes, byte[] ref, GenomeLoc refLoc, boolean debug) {
        TreeSet<Integer> startPosKeySet = new TreeSet<Integer>();
        int hapNumber = 0;
        if (debug) {
            logger.info("=== Best Haplotypes ===");
        }
        for (Haplotype h2 : haplotypes) {
            h2.setEventMap(new EventMap(h2, ref, refLoc, "HC" + hapNumber++));
            startPosKeySet.addAll(h2.getEventMap().getStartPositions());
            if (!debug) continue;
            logger.info(h2.toString());
            logger.info("> Cigar = " + h2.getCigar());
            logger.info(">> Events = " + h2.getEventMap());
        }
        return startPosKeySet;
    }

    public static TreeSet<VariantContext> getAllVariantContexts(List<Haplotype> haplotypes) {
        TreeSet<VariantContext> vcs = new TreeSet<VariantContext>(new VariantContextComparator());
        for (Haplotype h2 : haplotypes) {
            vcs.addAll(h2.getEventMap().getVariantContexts());
        }
        return vcs;
    }

    private static class VariantContextComparator
    implements Comparator<VariantContext> {
        private VariantContextComparator() {
        }

        @Override
        public int compare(VariantContext vc1, VariantContext vc2) {
            return vc1.getStart() - vc2.getStart();
        }
    }
}

