/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.crypt;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.broadinstitute.sting.utils.crypt.CryptUtils;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.exceptions.UserException;
import org.broadinstitute.sting.utils.io.IOUtils;

public class GATKKey {
    private PrivateKey privateKey;
    private PublicKey publicKey;
    private String emailAddress;
    private byte[] signature;
    private String signingAlgorithm;
    public static final String DEFAULT_SIGNING_ALGORITHM = "SHA1withRSA";
    public static final byte GATK_KEY_SECTIONAL_DELIMITER = 0;

    public GATKKey(PrivateKey privateKey, PublicKey publicKey, String emailAddress) {
        this(privateKey, publicKey, emailAddress, DEFAULT_SIGNING_ALGORITHM);
    }

    public GATKKey(PrivateKey privateKey, PublicKey publicKey, String emailAddress, String signingAlgorithm) {
        if (privateKey == null || publicKey == null || emailAddress == null || emailAddress.length() == 0 || signingAlgorithm == null) {
            throw new ReviewedStingException("Cannot construct GATKKey using null/empty arguments");
        }
        this.privateKey = privateKey;
        this.publicKey = publicKey;
        this.emailAddress = emailAddress;
        this.signingAlgorithm = signingAlgorithm;
        this.validateEmailAddress();
        this.generateSignature();
        if (!this.isValid()) {
            throw new ReviewedStingException("Newly-generated GATK key fails validation -- this should never happen!");
        }
    }

    public GATKKey(PublicKey publicKey, File keyFile) {
        this(publicKey, keyFile, DEFAULT_SIGNING_ALGORITHM);
    }

    public GATKKey(PublicKey publicKey, File keyFile, String signingAlgorithm) {
        if (publicKey == null || keyFile == null || signingAlgorithm == null) {
            throw new ReviewedStingException("Cannot construct GATKKey using null arguments");
        }
        this.publicKey = publicKey;
        this.signingAlgorithm = signingAlgorithm;
        this.readKey(keyFile);
    }

    public void writeKey(File destination) {
        try {
            byte[] keyBytes = this.marshalKeyData();
            IOUtils.writeByteArrayToStream(keyBytes, new GZIPOutputStream(new FileOutputStream(destination)));
        }
        catch (IOException e2) {
            throw new UserException.CouldNotCreateOutputFile(destination, (Exception)e2);
        }
    }

    public boolean isValid() {
        try {
            Signature sig = Signature.getInstance(this.signingAlgorithm);
            sig.initVerify(this.publicKey);
            sig.update(this.emailAddress.getBytes());
            return sig.verify(this.signature);
        }
        catch (NoSuchAlgorithmException e2) {
            throw new ReviewedStingException(String.format("Signing algorithm %s not found", this.signingAlgorithm), e2);
        }
        catch (InvalidKeyException e3) {
            throw new ReviewedStingException(String.format("Public key %s is invalid", this.publicKey), e3);
        }
        catch (SignatureException e4) {
            throw new UserException.UnreadableKeyException("Signature is invalid or signing algorithm was unable to process the input data", (Exception)e4);
        }
    }

    private void generateSignature() {
        try {
            Signature sig = Signature.getInstance(this.signingAlgorithm);
            sig.initSign(this.privateKey, CryptUtils.createRandomnessSource());
            sig.update(this.emailAddress.getBytes());
            this.signature = sig.sign();
        }
        catch (NoSuchAlgorithmException e2) {
            throw new ReviewedStingException(String.format("Signing algorithm %s not found", this.signingAlgorithm), e2);
        }
        catch (InvalidKeyException e3) {
            throw new ReviewedStingException(String.format("Private key %s is invalid", this.privateKey), e3);
        }
        catch (SignatureException e4) {
            throw new ReviewedStingException(String.format("Error creating signature for email address %s", this.emailAddress), e4);
        }
    }

    private void readKey(File source) {
        try {
            byte[] keyBytes = IOUtils.readStreamIntoByteArray(new GZIPInputStream(new FileInputStream(source)));
            if (keyBytes.length != IOUtils.getGZIPFileUncompressedSize(source)) {
                throw new UserException.UnreadableKeyException("Number of bytes read does not match the uncompressed size specified in the GZIP ISIZE field");
            }
            this.unmarshalKeyData(keyBytes);
        }
        catch (FileNotFoundException e2) {
            throw new UserException.CouldNotReadInputFile(source, (Exception)e2);
        }
        catch (IOException e3) {
            throw new UserException.UnreadableKeyException(source, (Exception)e3);
        }
        catch (UserException.CouldNotReadInputFile e4) {
            throw new UserException.UnreadableKeyException(source, (Exception)e4);
        }
    }

    private byte[] marshalKeyData() {
        byte[] emailAddressBytes = this.emailAddress.getBytes();
        byte[] assembledKey = new byte[emailAddressBytes.length + 1 + this.signature.length];
        System.arraycopy(emailAddressBytes, 0, assembledKey, 0, emailAddressBytes.length);
        assembledKey[emailAddressBytes.length] = 0;
        System.arraycopy(this.signature, 0, assembledKey, emailAddressBytes.length + 1, this.signature.length);
        return assembledKey;
    }

    private void unmarshalKeyData(byte[] keyBytes) {
        int delimiterPosition = -1;
        for (int i2 = 0; i2 < keyBytes.length; ++i2) {
            if (keyBytes[i2] != 0) continue;
            delimiterPosition = i2;
            break;
        }
        if (delimiterPosition == -1) {
            throw new UserException.UnreadableKeyException("Malformed GATK key contains no sectional delimiter");
        }
        if (delimiterPosition == 0) {
            throw new UserException.UnreadableKeyException("Malformed GATK key contains no email address");
        }
        if (delimiterPosition == keyBytes.length - 1) {
            throw new UserException.UnreadableKeyException("Malformed GATK key contains no signature");
        }
        byte[] emailAddressBytes = new byte[delimiterPosition];
        System.arraycopy(keyBytes, 0, emailAddressBytes, 0, delimiterPosition);
        this.emailAddress = new String(emailAddressBytes);
        this.signature = new byte[keyBytes.length - delimiterPosition - 1];
        System.arraycopy(keyBytes, delimiterPosition + 1, this.signature, 0, keyBytes.length - delimiterPosition - 1);
    }

    private void validateEmailAddress() {
        for (byte b2 : this.emailAddress.getBytes()) {
            if (b2 != 0) continue;
            throw new UserException(String.format("Email address must not contain a byte with value %d", (byte)0));
        }
    }
}

