/*
 * Decompiled with CFR 0.152.
 */
package net.sf.samtools;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.List;
import net.sf.samtools.BAMFileConstants;
import net.sf.samtools.BAMIndex;
import net.sf.samtools.BAMIndexContent;
import net.sf.samtools.BAMIndexMetaData;
import net.sf.samtools.Bin;
import net.sf.samtools.Chunk;
import net.sf.samtools.LinearIndex;
import net.sf.samtools.SAMException;
import net.sf.samtools.SAMSequenceDictionary;
import net.sf.samtools.seekablestream.SeekableStream;
import net.sf.samtools.util.RuntimeIOException;

public abstract class AbstractBAMFileIndex
implements BAMIndex {
    protected static final int BIN_GENOMIC_SPAN = 0x20000000;
    private static final int[] LEVEL_STARTS = new int[]{0, 1, 9, 73, 585, 4681};
    public static final int MAX_BINS = 37450;
    public static final int MAX_LINEAR_INDEX_SIZE = 37451 - LEVEL_STARTS[LEVEL_STARTS.length - 1];
    private final IndexFileBuffer mIndexBuffer;
    private SAMSequenceDictionary mBamDictionary = null;

    protected AbstractBAMFileIndex(SeekableStream stream, SAMSequenceDictionary dictionary) {
        this.mBamDictionary = dictionary;
        this.mIndexBuffer = new IndexStreamBuffer(stream);
    }

    protected AbstractBAMFileIndex(File file, SAMSequenceDictionary dictionary) {
        this(file, dictionary, true);
    }

    protected AbstractBAMFileIndex(File file, SAMSequenceDictionary dictionary, boolean useMemoryMapping) {
        this.mBamDictionary = dictionary;
        this.mIndexBuffer = useMemoryMapping ? new MemoryMappedFileBuffer(file) : new RandomAccessFileBuffer(file);
        this.seek(0);
        byte[] buffer = new byte[4];
        this.readBytes(buffer);
        if (!Arrays.equals(buffer, BAMFileConstants.BAM_INDEX_MAGIC)) {
            throw new RuntimeException("Invalid file header in BAM index " + file + ": " + new String(buffer));
        }
    }

    @Override
    public void close() {
        this.mIndexBuffer.close();
    }

    public static int getNumIndexLevels() {
        return LEVEL_STARTS.length;
    }

    public static int getFirstBinInLevel(int levelNumber) {
        return LEVEL_STARTS[levelNumber];
    }

    public int getLevelSize(int levelNumber) {
        if (levelNumber == AbstractBAMFileIndex.getNumIndexLevels()) {
            return 37451 - LEVEL_STARTS[levelNumber];
        }
        return LEVEL_STARTS[levelNumber + 1] - LEVEL_STARTS[levelNumber];
    }

    public int getLevelForBin(Bin bin) {
        if (bin.getBinNumber() >= 37450) {
            throw new SAMException("Tried to get level for invalid bin.");
        }
        for (int i2 = AbstractBAMFileIndex.getNumIndexLevels() - 1; i2 >= 0; --i2) {
            if (bin.getBinNumber() < LEVEL_STARTS[i2]) continue;
            return i2;
        }
        throw new SAMException("Unable to find correct bin for bin " + bin);
    }

    public int getFirstLocusInBin(Bin bin) {
        int level = this.getLevelForBin(bin);
        int levelStart = LEVEL_STARTS[level];
        int levelSize = (level == AbstractBAMFileIndex.getNumIndexLevels() - 1 ? 37449 : LEVEL_STARTS[level + 1]) - levelStart;
        return (bin.getBinNumber() - levelStart) * (0x20000000 / levelSize) + 1;
    }

    public int getLastLocusInBin(Bin bin) {
        int level = this.getLevelForBin(bin);
        int levelStart = LEVEL_STARTS[level];
        int levelSize = (level == AbstractBAMFileIndex.getNumIndexLevels() - 1 ? 37449 : LEVEL_STARTS[level + 1]) - levelStart;
        return (bin.getBinNumber() - levelStart + 1) * (0x20000000 / levelSize);
    }

    public int getNumberOfReferences() {
        this.seek(4);
        return this.readInteger();
    }

    @Override
    public long getStartOfLastLinearBin() {
        this.seek(4);
        int sequenceCount = this.readInteger();
        long lastLinearIndexPointer = -1L;
        for (int i2 = 0; i2 < sequenceCount; ++i2) {
            int nBins = this.readInteger();
            for (int j1 = 0; j1 < nBins; ++j1) {
                this.skipBytes(4);
                int nChunks = this.readInteger();
                this.skipBytes(16 * nChunks);
            }
            int nLinearBins = this.readInteger();
            if (nLinearBins <= 0) continue;
            this.skipBytes(8 * (nLinearBins - 1));
            lastLinearIndexPointer = this.readLong();
        }
        return lastLinearIndexPointer;
    }

    @Override
    public BAMIndexMetaData getMetaData(int reference) {
        this.seek(4);
        ArrayList<Chunk> metaDataChunks = new ArrayList<Chunk>();
        int sequenceCount = this.readInteger();
        if (reference >= sequenceCount) {
            return null;
        }
        this.skipToSequence(reference);
        int binCount = this.readInteger();
        for (int binNumber = 0; binNumber < binCount; ++binNumber) {
            int indexBin = this.readInteger();
            int nChunks = this.readInteger();
            Chunk lastChunk = null;
            if (indexBin == 37450) {
                for (int ci = 0; ci < nChunks; ++ci) {
                    long chunkBegin = this.readLong();
                    long chunkEnd = this.readLong();
                    lastChunk = new Chunk(chunkBegin, chunkEnd);
                    metaDataChunks.add(lastChunk);
                }
                continue;
            }
            this.skipBytes(16 * nChunks);
        }
        return new BAMIndexMetaData(metaDataChunks);
    }

    public Long getNoCoordinateCount() {
        this.seek(4);
        int sequenceCount = this.readInteger();
        this.skipToSequence(sequenceCount);
        try {
            return this.readLong();
        }
        catch (Exception e2) {
            return null;
        }
    }

    protected BAMIndexContent query(int referenceSequence, int startPos, int endPos) {
        this.seek(4);
        ArrayList<Chunk> metaDataChunks = new ArrayList<Chunk>();
        int sequenceCount = this.readInteger();
        if (referenceSequence >= sequenceCount) {
            return null;
        }
        BitSet regionBins = this.regionToBins(startPos, endPos);
        if (regionBins == null) {
            return null;
        }
        this.skipToSequence(referenceSequence);
        int binCount = this.readInteger();
        boolean metaDataSeen = false;
        Bin[] bins = new Bin[this.getMaxBinNumberForReference(referenceSequence) + 1];
        for (int binNumber = 0; binNumber < binCount; ++binNumber) {
            long chunkEnd;
            long chunkBegin;
            int indexBin = this.readInteger();
            int nChunks = this.readInteger();
            ArrayList<Chunk> chunks = new ArrayList<Chunk>(nChunks);
            Chunk lastChunk = null;
            if (regionBins.get(indexBin)) {
                for (int ci = 0; ci < nChunks; ++ci) {
                    chunkBegin = this.readLong();
                    chunkEnd = this.readLong();
                    lastChunk = new Chunk(chunkBegin, chunkEnd);
                    chunks.add(lastChunk);
                }
            } else {
                if (indexBin == 37450) {
                    for (int ci = 0; ci < nChunks; ++ci) {
                        chunkBegin = this.readLong();
                        chunkEnd = this.readLong();
                        lastChunk = new Chunk(chunkBegin, chunkEnd);
                        metaDataChunks.add(lastChunk);
                    }
                    metaDataSeen = true;
                    continue;
                }
                this.skipBytes(16 * nChunks);
            }
            Bin bin = new Bin(referenceSequence, indexBin);
            bin.setChunkList(chunks);
            bin.setLastChunk(lastChunk);
            bins[indexBin] = bin;
        }
        int nLinearBins = this.readInteger();
        int regionLinearBinStart = LinearIndex.convertToLinearIndexOffset(startPos);
        int regionLinearBinStop = endPos > 0 ? LinearIndex.convertToLinearIndexOffset(endPos) : nLinearBins - 1;
        int actualStop = Math.min(regionLinearBinStop, nLinearBins - 1);
        long[] linearIndexEntries = new long[]{};
        if (regionLinearBinStart < nLinearBins) {
            linearIndexEntries = new long[actualStop - regionLinearBinStart + 1];
            this.skipBytes(8 * regionLinearBinStart);
            for (int linearBin = regionLinearBinStart; linearBin <= actualStop; ++linearBin) {
                linearIndexEntries[linearBin - regionLinearBinStart] = this.readLong();
            }
        }
        LinearIndex linearIndex = new LinearIndex(referenceSequence, regionLinearBinStart, linearIndexEntries);
        return new BAMIndexContent(referenceSequence, bins, binCount - (metaDataSeen ? 1 : 0), new BAMIndexMetaData(metaDataChunks), linearIndex);
    }

    private int getMaxBinNumberForReference(int reference) {
        try {
            int sequenceLength = this.mBamDictionary.getSequence(reference).getSequenceLength();
            return AbstractBAMFileIndex.getMaxBinNumberForSequenceLength(sequenceLength);
        }
        catch (Exception e2) {
            return 37450;
        }
    }

    static int getMaxBinNumberForSequenceLength(int sequenceLength) {
        return AbstractBAMFileIndex.getFirstBinInLevel(AbstractBAMFileIndex.getNumIndexLevels() - 1) + (sequenceLength >> 14);
    }

    protected abstract BAMIndexContent getQueryResults(int var1);

    protected int getMaxAddressibleGenomicLocation() {
        return 0x20000000;
    }

    protected BitSet regionToBins(int startPos, int endPos) {
        int k2;
        int end;
        int maxPos = 0x1FFFFFFF;
        int start = startPos <= 0 ? 0 : startPos - 1 & 0x1FFFFFFF;
        int n2 = end = endPos <= 0 ? 0x1FFFFFFF : endPos - 1 & 0x1FFFFFFF;
        if (start > end) {
            return null;
        }
        BitSet bitSet = new BitSet(37450);
        bitSet.set(0);
        for (k2 = 1 + (start >> 26); k2 <= 1 + (end >> 26); ++k2) {
            bitSet.set(k2);
        }
        for (k2 = 9 + (start >> 23); k2 <= 9 + (end >> 23); ++k2) {
            bitSet.set(k2);
        }
        for (k2 = 73 + (start >> 20); k2 <= 73 + (end >> 20); ++k2) {
            bitSet.set(k2);
        }
        for (k2 = 585 + (start >> 17); k2 <= 585 + (end >> 17); ++k2) {
            bitSet.set(k2);
        }
        for (k2 = 4681 + (start >> 14); k2 <= 4681 + (end >> 14); ++k2) {
            bitSet.set(k2);
        }
        return bitSet;
    }

    protected List<Chunk> optimizeChunkList(List<Chunk> chunks, long minimumOffset) {
        Chunk lastChunk = null;
        Collections.sort(chunks);
        ArrayList<Chunk> result = new ArrayList<Chunk>();
        for (Chunk chunk : chunks) {
            if (chunk.getChunkEnd() <= minimumOffset) continue;
            if (result.isEmpty()) {
                result.add(chunk);
                lastChunk = chunk;
                continue;
            }
            if (!lastChunk.overlaps(chunk) && !lastChunk.isAdjacentTo(chunk)) {
                result.add(chunk);
                lastChunk = chunk;
                continue;
            }
            if (chunk.getChunkEnd() <= lastChunk.getChunkEnd()) continue;
            lastChunk.setChunkEnd(chunk.getChunkEnd());
        }
        return result;
    }

    private void skipToSequence(int sequenceIndex) {
        for (int i2 = 0; i2 < sequenceIndex; ++i2) {
            int nBins = this.readInteger();
            for (int j2 = 0; j2 < nBins; ++j2) {
                int bin = this.readInteger();
                int nChunks = this.readInteger();
                this.skipBytes(16 * nChunks);
            }
            int nLinearBins = this.readInteger();
            this.skipBytes(8 * nLinearBins);
        }
    }

    private void readBytes(byte[] bytes) {
        this.mIndexBuffer.readBytes(bytes);
    }

    private int readInteger() {
        return this.mIndexBuffer.readInteger();
    }

    private long readLong() {
        return this.mIndexBuffer.readLong();
    }

    private void skipBytes(int count) {
        this.mIndexBuffer.skipBytes(count);
    }

    private void seek(int position) {
        this.mIndexBuffer.seek(position);
    }

    private static class IndexStreamBuffer
    extends IndexFileBuffer {
        private final SeekableStream in;
        private final ByteBuffer tmpBuf;

        public IndexStreamBuffer(SeekableStream s2) {
            this.in = s2;
            this.tmpBuf = ByteBuffer.allocate(8);
            this.tmpBuf.order(ByteOrder.LITTLE_ENDIAN);
        }

        @Override
        public void close() {
            try {
                this.in.close();
            }
            catch (IOException e2) {
                throw new RuntimeIOException(e2);
            }
        }

        @Override
        public void readBytes(byte[] bytes) {
            try {
                this.in.read(bytes);
            }
            catch (IOException e2) {
                throw new RuntimeIOException(e2);
            }
        }

        @Override
        public void seek(int position) {
            try {
                this.in.seek(position);
            }
            catch (IOException e2) {
                throw new RuntimeIOException(e2);
            }
        }

        @Override
        public int readInteger() {
            try {
                int r2 = this.in.read(this.tmpBuf.array(), 0, 4);
                if (r2 != 4) {
                    throw new RuntimeIOException("Expected 4 bytes, got " + r2);
                }
            }
            catch (IOException e2) {
                throw new RuntimeIOException(e2);
            }
            return this.tmpBuf.getInt(0);
        }

        @Override
        public long readLong() {
            try {
                int r2 = this.in.read(this.tmpBuf.array(), 0, 8);
                if (r2 != 8) {
                    throw new RuntimeIOException("Expected 8 bytes, got " + r2);
                }
            }
            catch (IOException e2) {
                throw new RuntimeIOException(e2);
            }
            return this.tmpBuf.getLong(0);
        }

        @Override
        public void skipBytes(int count) {
            try {
                int skipped;
                for (int s2 = count; s2 > 0; s2 -= skipped) {
                    skipped = (int)this.in.skip(s2);
                    if (skipped > 0) continue;
                    throw new RuntimeIOException("Failed to skip " + s2);
                }
            }
            catch (IOException e2) {
                throw new RuntimeIOException(e2);
            }
        }
    }

    private static class RandomAccessFileBuffer
    extends IndexFileBuffer {
        private static final int PAGE_SIZE = 4096;
        private static final int PAGE_OFFSET_MASK = 4095;
        private static final int PAGE_MASK = -4096;
        private static final int INVALID_PAGE = 1;
        private File mFile;
        private RandomAccessFile mRandomAccessFile;
        private int mFileLength;
        private int mFilePointer = 0;
        private int mCurrentPage = 1;
        private final byte[] mBuffer = new byte[4096];

        RandomAccessFileBuffer(File file) {
            this.mFile = file;
            try {
                this.mRandomAccessFile = new RandomAccessFile(file, "r");
                long fileLength = this.mRandomAccessFile.length();
                if (fileLength > Integer.MAX_VALUE) {
                    throw new RuntimeException("BAM index file " + this.mFile + " is too large: " + fileLength);
                }
                this.mFileLength = (int)fileLength;
            }
            catch (IOException exc) {
                throw new RuntimeIOException(exc.getMessage(), exc);
            }
        }

        @Override
        void readBytes(byte[] bytes) {
            int resultOffset = 0;
            int resultLength = bytes.length;
            if (this.mFilePointer + resultLength > this.mFileLength) {
                throw new RuntimeException("Attempt to read past end of BAM index file (file is truncated?): " + this.mFile);
            }
            while (resultLength > 0) {
                this.loadPage(this.mFilePointer);
                int pageOffset = this.mFilePointer & 0xFFF;
                int copyLength = Math.min(resultLength, 4096 - pageOffset);
                System.arraycopy(this.mBuffer, pageOffset, bytes, resultOffset, copyLength);
                this.mFilePointer += copyLength;
                resultOffset += copyLength;
                resultLength -= copyLength;
            }
        }

        @Override
        int readInteger() {
            this.loadPage(this.mFilePointer);
            int pageOffset = this.mFilePointer & 0xFFF;
            this.mFilePointer += 4;
            return this.mBuffer[pageOffset + 0] & 0xFF | (this.mBuffer[pageOffset + 1] & 0xFF) << 8 | (this.mBuffer[pageOffset + 2] & 0xFF) << 16 | (this.mBuffer[pageOffset + 3] & 0xFF) << 24;
        }

        @Override
        long readLong() {
            long lower = this.readInteger();
            long upper = this.readInteger();
            return upper << 32 | lower & 0xFFFFFFFFL;
        }

        @Override
        void skipBytes(int count) {
            this.mFilePointer += count;
        }

        @Override
        void seek(int position) {
            this.mFilePointer = position;
        }

        @Override
        void close() {
            this.mFilePointer = 0;
            this.mCurrentPage = 1;
            if (this.mRandomAccessFile != null) {
                try {
                    this.mRandomAccessFile.close();
                }
                catch (IOException exc) {
                    throw new RuntimeIOException(exc.getMessage(), exc);
                }
                this.mRandomAccessFile = null;
            }
        }

        private void loadPage(int filePosition) {
            int page = filePosition & 0xFFFFF000;
            if (page == this.mCurrentPage) {
                return;
            }
            try {
                this.mRandomAccessFile.seek(page);
                int readLength = Math.min(this.mFileLength - page, 4096);
                this.mRandomAccessFile.readFully(this.mBuffer, 0, readLength);
                this.mCurrentPage = page;
            }
            catch (IOException exc) {
                throw new RuntimeIOException("Exception reading BAM index file " + this.mFile + ": " + exc.getMessage(), exc);
            }
        }
    }

    private static class MemoryMappedFileBuffer
    extends IndexFileBuffer {
        private MappedByteBuffer mFileBuffer;

        MemoryMappedFileBuffer(File file) {
            try {
                FileInputStream fileStream = new FileInputStream(file);
                FileChannel fileChannel = fileStream.getChannel();
                this.mFileBuffer = fileChannel.map(FileChannel.MapMode.READ_ONLY, 0L, fileChannel.size());
                this.mFileBuffer.order(ByteOrder.LITTLE_ENDIAN);
                fileChannel.close();
                fileStream.close();
            }
            catch (IOException exc) {
                throw new RuntimeIOException(exc.getMessage(), exc);
            }
        }

        @Override
        void readBytes(byte[] bytes) {
            this.mFileBuffer.get(bytes);
        }

        @Override
        int readInteger() {
            return this.mFileBuffer.getInt();
        }

        @Override
        long readLong() {
            return this.mFileBuffer.getLong();
        }

        @Override
        void skipBytes(int count) {
            this.mFileBuffer.position(this.mFileBuffer.position() + count);
        }

        @Override
        void seek(int position) {
            this.mFileBuffer.position(position);
        }

        @Override
        void close() {
            this.mFileBuffer = null;
        }
    }

    private static abstract class IndexFileBuffer {
        private IndexFileBuffer() {
        }

        abstract void readBytes(byte[] var1);

        abstract int readInteger();

        abstract long readLong();

        abstract void skipBytes(int var1);

        abstract void seek(int var1);

        abstract void close();
    }
}

