/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.fastq;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import net.sf.picard.PicardException;
import net.sf.picard.filter.AggregateFilter;
import net.sf.picard.filter.FailsVendorReadQualityFilter;
import net.sf.picard.filter.FilteringIterator;
import net.sf.picard.filter.SamRecordFilter;
import net.sf.picard.filter.TagFilter;
import net.sf.picard.filter.WholeReadClippedFilter;
import net.sf.picard.io.IoUtil;
import net.sf.picard.util.Log;
import net.sf.picard.util.PeekableIterator;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMRecordIterator;
import net.sf.samtools.util.BinaryCodec;

public class BamToBfqWriter {
    private static final int SEED_REGION_LENGTH = 28;
    private static final int MAX_SEED_REGION_NOCALL_FIXES = 2;
    private final File bamFile;
    private final String outputPrefix;
    private final String namePrefix;
    private final int nameTrim;
    private boolean pairedReads = false;
    private int wrote = 0;
    private int increment = 1;
    private int chunk = 0;
    private BinaryCodec codec1;
    private BinaryCodec codec2;
    private final Log log = Log.getInstance(BamToBfqWriter.class);
    private final boolean includeNonPfReads;
    private final boolean clipAdapters;
    private final Integer basesToWrite;

    public BamToBfqWriter(File bamFile, String outputPrefix, Integer total, Integer chunk, boolean pairedReads, String namePrefix, boolean includeNonPfReads, boolean clipAdapters, Integer basesToWrite) {
        IoUtil.assertFileIsReadable(bamFile);
        this.bamFile = bamFile;
        this.outputPrefix = outputPrefix;
        this.pairedReads = pairedReads;
        if (total != null) {
            double writeable = this.countWritableRecords();
            this.increment = (int)Math.floor(writeable / total.doubleValue());
            if (this.increment == 0) {
                this.increment = 1;
            }
        }
        if (chunk != null) {
            this.chunk = chunk;
        }
        this.namePrefix = namePrefix;
        this.nameTrim = namePrefix != null ? namePrefix.length() : 0;
        this.includeNonPfReads = includeNonPfReads;
        this.clipAdapters = clipAdapters;
        this.basesToWrite = basesToWrite;
    }

    public BamToBfqWriter(File bamFile, String outputPrefix, boolean pairedReads, String namePrefix, boolean includeNonPfReads) {
        this(bamFile, outputPrefix, null, null, pairedReads, namePrefix, includeNonPfReads, true, null);
    }

    public void writeBfqFiles() {
        SAMRecordIterator iterator = new SAMFileReader(IoUtil.openFileForReading(this.bamFile)).iterator();
        TagFilter tagFilter = new TagFilter("XN", 1);
        FailsVendorReadQualityFilter qualityFilter = new FailsVendorReadQualityFilter();
        WholeReadClippedFilter clippedFilter = new WholeReadClippedFilter();
        if (!this.pairedReads) {
            ArrayList<SamRecordFilter> filters = new ArrayList<SamRecordFilter>();
            filters.add(tagFilter);
            filters.add(clippedFilter);
            if (!this.includeNonPfReads) {
                filters.add(qualityFilter);
            }
            this.writeSingleEndBfqs(iterator, filters);
            this.codec1.close();
        } else {
            this.writePairedEndBfqs(iterator, tagFilter, qualityFilter, clippedFilter);
            this.codec1.close();
            this.codec2.close();
        }
        this.log.info("Wrote " + this.wrote + " bfq records.");
    }

    private void writePairedEndBfqs(Iterator<SAMRecord> iterator, TagFilter tagFilter, FailsVendorReadQualityFilter qualityFilter, SamRecordFilter ... otherFilters) {
        int fileIndex = 0;
        this.initializeNextBfqFiles(fileIndex++);
        int records = 0;
        block0: while (iterator.hasNext()) {
            SAMRecord first = iterator.next();
            if (!iterator.hasNext()) {
                throw new PicardException("Mismatched number of records in " + this.bamFile.getAbsolutePath());
            }
            SAMRecord second = iterator.next();
            if (!second.getReadName().equals(first.getReadName()) || first.getFirstOfPairFlag() == second.getFirstOfPairFlag()) {
                throw new PicardException("Unmatched read pairs in " + this.bamFile.getAbsolutePath() + ": " + first.getReadName() + ", " + second.getReadName() + ".");
            }
            if (tagFilter.filterOut(first) && tagFilter.filterOut(second) || !this.includeNonPfReads && (qualityFilter.filterOut(first) || qualityFilter.filterOut(second))) continue;
            for (SamRecordFilter filter : otherFilters) {
                if (filter.filterOut(first) || filter.filterOut(second)) continue block0;
            }
            if (++records % this.increment != 0) continue;
            first.setReadName(first.getReadName() + "/1");
            this.writeFastqRecord(first.getFirstOfPairFlag() ? this.codec1 : this.codec2, first);
            second.setReadName(second.getReadName() + "/2");
            this.writeFastqRecord(second.getFirstOfPairFlag() ? this.codec1 : this.codec2, second);
            ++this.wrote;
            if (this.wrote % 1000000 == 0) {
                this.log.info(this.wrote + " records written.");
            }
            if (this.chunk <= 0 || this.wrote % this.chunk != 0) continue;
            this.initializeNextBfqFiles(fileIndex++);
        }
    }

    private void writeSingleEndBfqs(Iterator<SAMRecord> iterator, List<SamRecordFilter> filters) {
        int fileIndex = 0;
        this.initializeNextBfqFiles(fileIndex++);
        int records = 0;
        FilteringIterator it = new FilteringIterator(iterator, new AggregateFilter(filters));
        while (it.hasNext()) {
            SAMRecord record = it.next();
            if (++records % this.increment != 0) continue;
            record.setReadName(record.getReadName() + "/1");
            this.writeFastqRecord(this.codec1, record);
            ++this.wrote;
            if (this.wrote % 1000000 == 0) {
                this.log.info(this.wrote + " records processed.");
            }
            if (this.chunk <= 0 || this.wrote % this.chunk != 0) continue;
            this.initializeNextBfqFiles(fileIndex++);
        }
    }

    private void initializeNextBfqFiles(int fileIndex) {
        if (this.codec1 != null) {
            this.codec1.close();
            if (this.pairedReads) {
                this.codec2.close();
            }
        }
        File bfq1 = this.getOutputFile(this.outputPrefix, 1, fileIndex);
        this.codec1 = new BinaryCodec(IoUtil.openFileForWriting(bfq1));
        this.log.info("Now writing to file " + bfq1.getAbsolutePath());
        if (this.pairedReads) {
            File bfq2 = this.getOutputFile(this.outputPrefix, 2, fileIndex);
            this.codec2 = new BinaryCodec(IoUtil.openFileForWriting(bfq2));
            this.log.info("Now writing to file " + bfq2.getAbsolutePath());
        }
    }

    private void writeFastqRecord(BinaryCodec codec, SAMRecord rec) {
        Integer trimPoint;
        String readName = rec.getReadName();
        if (this.namePrefix != null && readName.startsWith(this.namePrefix)) {
            readName = readName.substring(this.nameTrim);
        }
        codec.writeString(readName, true, true);
        char[] seqs = rec.getReadString().toCharArray();
        char[] quals = rec.getBaseQualityString().toCharArray();
        int retainedLength = seqs.length;
        if (this.clipAdapters && (trimPoint = rec.getIntegerAttribute("XT")) != null) {
            assert (rec.getReadLength() == seqs.length);
            retainedLength = Math.min(seqs.length, Math.max(28, trimPoint - 1));
        }
        codec.writeInt(this.basesToWrite != null ? this.basesToWrite : seqs.length);
        byte[] seqsAndQuals = this.encodeSeqsAndQuals(seqs, quals, retainedLength);
        codec.writeBytes(seqsAndQuals);
    }

    private byte[] encodeSeqsAndQuals(char[] seqs, char[] quals, int retainedLength) {
        int i2;
        byte[] seqsAndQuals = new byte[this.basesToWrite == null ? seqs.length : this.basesToWrite];
        int seedRegionNoCallFixes = 0;
        for (i2 = 0; i2 < retainedLength && i2 < seqsAndQuals.length; ++i2) {
            int base;
            int quality = Math.min(quals[i2] - 33, 63);
            switch (seqs[i2]) {
                case 'A': 
                case 'a': {
                    base = 0;
                    break;
                }
                case 'C': 
                case 'c': {
                    base = 1;
                    break;
                }
                case 'G': 
                case 'g': {
                    base = 2;
                    break;
                }
                case 'T': 
                case 't': {
                    base = 3;
                    break;
                }
                case '.': 
                case 'N': 
                case 'n': {
                    base = 0;
                    if (i2 < 28) {
                        if (seedRegionNoCallFixes < 2) {
                            quality = 1;
                            ++seedRegionNoCallFixes;
                            break;
                        }
                        quality = 0;
                        break;
                    }
                    quality = 1;
                    break;
                }
                default: {
                    throw new PicardException("Unknown base when writing bfq file: " + seqs[i2]);
                }
            }
            seqsAndQuals[i2] = this.encodeBaseAndQuality(base, quality);
        }
        for (i2 = retainedLength; i2 < seqsAndQuals.length; ++i2) {
            seqsAndQuals[i2] = this.encodeBaseAndQuality(0, 1);
        }
        return seqsAndQuals;
    }

    private byte encodeBaseAndQuality(int base, int quality) {
        return (byte)(base << 6 | quality);
    }

    private int countWritableRecords() {
        int count = 0;
        SAMFileReader reader = new SAMFileReader(IoUtil.openFileForReading(this.bamFile));
        if (!reader.getFileHeader().getSortOrder().equals((Object)SAMFileHeader.SortOrder.queryname)) {
            throw new PicardException("Input file (" + this.bamFile.getAbsolutePath() + ") needs to be sorted by queryname.");
        }
        PeekableIterator<SAMRecord> it = new PeekableIterator<SAMRecord>(reader.iterator());
        if (!this.pairedReads) {
            ArrayList<SamRecordFilter> filters = new ArrayList<SamRecordFilter>();
            filters.add(new TagFilter("XN", 1));
            if (!this.includeNonPfReads) {
                filters.add(new FailsVendorReadQualityFilter());
            }
            FilteringIterator itr = new FilteringIterator(it, new AggregateFilter(filters));
            while (itr.hasNext()) {
                itr.next();
                ++count;
            }
        } else {
            while (it.hasNext()) {
                SAMRecord first = it.next();
                SAMRecord second = it.next();
                if (first.getAttribute("XN") != null && second.getAttribute("XN") != null || !this.includeNonPfReads && (first.getReadFailsVendorQualityCheckFlag() || second.getReadFailsVendorQualityCheckFlag())) continue;
                ++count;
            }
        }
        it.close();
        return count;
    }

    private File getOutputFile(String outputPrefix, int read, int index) {
        File result = new File(outputPrefix + index + "." + read + ".bfq");
        IoUtil.assertFileIsWritable(result);
        return result;
    }
}

