/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.activeregion;

import com.google.java.contract.Ensures;
import com.google.java.contract.Invariant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import net.sf.picard.reference.IndexedFastaSequenceFile;
import org.broadinstitute.sting.utils.GenomeLoc;
import org.broadinstitute.sting.utils.GenomeLocParser;
import org.broadinstitute.sting.utils.GenomeLocSortedSet;
import org.broadinstitute.sting.utils.HasGenomeLocation;
import org.broadinstitute.sting.utils.activeregion.ActivityProfileState;
import org.broadinstitute.sting.utils.sam.GATKSAMRecord;

@Invariant(value={"extension >= 0", "activeRegionLoc != null", "genomeLocParser != null", "spanIncludingReads != null", "extendedLoc != null"})
public class ActiveRegion
implements HasGenomeLocation {
    private final List<GATKSAMRecord> reads = new ArrayList<GATKSAMRecord>();
    private final List<ActivityProfileState> supportingStates;
    private final GenomeLoc activeRegionLoc;
    private final GenomeLoc extendedLoc;
    private final int extension;
    private final GenomeLocParser genomeLocParser;
    private final boolean isActive;
    private GenomeLoc spanIncludingReads;

    public ActiveRegion(GenomeLoc activeRegionLoc, List<ActivityProfileState> supportingStates, boolean isActive, GenomeLocParser genomeLocParser, int extension) {
        if (activeRegionLoc == null) {
            throw new IllegalArgumentException("activeRegionLoc cannot be null");
        }
        if (activeRegionLoc.size() == 0) {
            throw new IllegalArgumentException("Active region cannot be of zero size, but got " + activeRegionLoc);
        }
        if (genomeLocParser == null) {
            throw new IllegalArgumentException("genomeLocParser cannot be null");
        }
        if (extension < 0) {
            throw new IllegalArgumentException("extension cannot be < 0 but got " + extension);
        }
        this.activeRegionLoc = activeRegionLoc;
        this.supportingStates = supportingStates == null ? Collections.emptyList() : new ArrayList<ActivityProfileState>(supportingStates);
        this.isActive = isActive;
        this.genomeLocParser = genomeLocParser;
        this.extension = extension;
        this.spanIncludingReads = this.extendedLoc = genomeLocParser.createGenomeLocOnContig(activeRegionLoc.getContig(), activeRegionLoc.getStart() - extension, activeRegionLoc.getStop() + extension);
        if (!this.supportingStates.isEmpty()) {
            if (this.supportingStates.size() != activeRegionLoc.size()) {
                throw new IllegalArgumentException("Supporting states wasn't empty but it doesn't have exactly one state per bp in the active region: states " + this.supportingStates.size() + " vs. bp in region = " + activeRegionLoc.size());
            }
            GenomeLoc lastStateLoc = null;
            for (ActivityProfileState state : this.supportingStates) {
                if (lastStateLoc != null && (state.getLoc().getStart() != lastStateLoc.getStart() + 1 || state.getLoc().getContigIndex() != lastStateLoc.getContigIndex())) {
                    throw new IllegalArgumentException("Supporting state has an invalid sequence: last state was " + lastStateLoc + " but next state was " + state);
                }
                lastStateLoc = state.getLoc();
            }
        }
    }

    public String toString() {
        return "ActiveRegion " + this.activeRegionLoc.toString() + " active?=" + this.isActive() + " nReads=" + this.reads.size();
    }

    public byte[] getActiveRegionReference(IndexedFastaSequenceFile referenceReader) {
        return this.getActiveRegionReference(referenceReader, 0);
    }

    @Ensures(value={"result != null"})
    public byte[] getActiveRegionReference(IndexedFastaSequenceFile referenceReader, int padding) {
        return this.getReference(referenceReader, padding, this.extendedLoc);
    }

    public byte[] getFullReference(IndexedFastaSequenceFile referenceReader) {
        return this.getFullReference(referenceReader, 0);
    }

    public byte[] getFullReference(IndexedFastaSequenceFile referenceReader, int padding) {
        return this.getReference(referenceReader, padding, this.spanIncludingReads);
    }

    @Ensures(value={"result != null"})
    private byte[] getReference(IndexedFastaSequenceFile referenceReader, int padding, GenomeLoc genomeLoc) {
        if (referenceReader == null) {
            throw new IllegalArgumentException("referenceReader cannot be null");
        }
        if (padding < 0) {
            throw new IllegalArgumentException("padding must be a positive integer but got " + padding);
        }
        if (genomeLoc == null) {
            throw new IllegalArgumentException("genomeLoc cannot be null");
        }
        if (genomeLoc.size() == 0) {
            throw new IllegalArgumentException("GenomeLoc must have size > 0 but got " + genomeLoc);
        }
        byte[] reference = referenceReader.getSubsequenceAt(genomeLoc.getContig(), Math.max(1, genomeLoc.getStart() - padding), Math.min(referenceReader.getSequenceDictionary().getSequence(genomeLoc.getContig()).getSequenceLength(), genomeLoc.getStop() + padding)).getBases();
        return reference;
    }

    @Override
    @Ensures(value={"result != null"})
    public GenomeLoc getLocation() {
        return this.activeRegionLoc;
    }

    @Ensures(value={"result != null"})
    public GenomeLoc getExtendedLoc() {
        return this.extendedLoc;
    }

    @Ensures(value={"result != null"})
    public GenomeLoc getReadSpanLoc() {
        return this.spanIncludingReads;
    }

    @Ensures(value={"result != null"})
    public List<ActivityProfileState> getSupportingStates() {
        return Collections.unmodifiableList(this.supportingStates);
    }

    @Ensures(value={"result >= 0"})
    public int getExtension() {
        return this.extension;
    }

    @Ensures(value={"result != null"})
    public List<GATKSAMRecord> getReads() {
        return Collections.unmodifiableList(this.reads);
    }

    @Ensures(value={"result >= 0"})
    public int size() {
        return this.reads.size();
    }

    @Ensures(value={"reads.size() == old(reads.size()) + 1"})
    public void add(GATKSAMRecord read) {
        if (read == null) {
            throw new IllegalArgumentException("Read cannot be null");
        }
        GenomeLoc readLoc = this.genomeLocParser.createGenomeLoc(read);
        if (!this.readOverlapsRegion(read)) {
            throw new IllegalArgumentException("Read location " + readLoc + " doesn't overlap with active region extended span " + this.extendedLoc);
        }
        this.spanIncludingReads = this.spanIncludingReads.union(readLoc);
        if (!this.reads.isEmpty()) {
            GATKSAMRecord lastRead = this.reads.get(this.size() - 1);
            if (!lastRead.getReferenceIndex().equals(read.getReferenceIndex())) {
                throw new IllegalArgumentException("Attempting to add a read to ActiveRegion not on the same contig as other reads: lastRead " + lastRead + " attempting to add " + read);
            }
            if (read.getAlignmentStart() < lastRead.getAlignmentStart()) {
                throw new IllegalArgumentException("Attempting to add a read to ActiveRegion out of order w.r.t. other reads: lastRead " + lastRead + " at " + lastRead.getAlignmentStart() + " attempting to add " + read + " at " + read.getAlignmentStart());
            }
        }
        this.reads.add(read);
    }

    public boolean readOverlapsRegion(GATKSAMRecord read) {
        GenomeLoc readLoc = this.genomeLocParser.createGenomeLoc(read);
        return readLoc.overlapsP(this.extendedLoc);
    }

    public void addAll(Collection<GATKSAMRecord> reads) {
        if (reads == null) {
            throw new IllegalArgumentException("reads cannot be null");
        }
        for (GATKSAMRecord read : reads) {
            this.add(read);
        }
    }

    @Ensures(value={"size() == 0"})
    public void clearReads() {
        this.spanIncludingReads = this.extendedLoc;
        this.reads.clear();
    }

    public void removeAll(Set<GATKSAMRecord> readsToRemove) {
        Iterator<GATKSAMRecord> it = this.reads.iterator();
        this.spanIncludingReads = this.extendedLoc;
        while (it.hasNext()) {
            GATKSAMRecord read = it.next();
            if (readsToRemove.contains(read)) {
                it.remove();
                continue;
            }
            this.spanIncludingReads = this.spanIncludingReads.union(this.genomeLocParser.createGenomeLoc(read));
        }
    }

    protected boolean equalExceptReads(ActiveRegion other) {
        if (this.activeRegionLoc.compareTo(other.activeRegionLoc) != 0) {
            return false;
        }
        if (this.isActive() != other.isActive()) {
            return false;
        }
        if (this.genomeLocParser != other.genomeLocParser) {
            return false;
        }
        if (this.extension != other.extension) {
            return false;
        }
        return this.extendedLoc.compareTo(other.extendedLoc) == 0;
    }

    public boolean isActive() {
        return this.isActive;
    }

    @Ensures(value={"result != null"})
    protected List<ActiveRegion> splitAndTrimToIntervals(GenomeLocSortedSet intervals) {
        List<GenomeLoc> allOverlapping = intervals.getOverlapping(this.getLocation());
        LinkedList<ActiveRegion> clippedRegions = new LinkedList<ActiveRegion>();
        for (GenomeLoc overlapping : allOverlapping) {
            clippedRegions.add(this.trim(overlapping, this.extension));
        }
        return clippedRegions;
    }

    public ActiveRegion trim(GenomeLoc newExtent, int newExtension) {
        if (newExtent == null) {
            throw new IllegalArgumentException("Active region extent cannot be null");
        }
        GenomeLoc subLoc = this.getLocation().intersect(newExtent);
        int subStart = subLoc.getStart() - this.getLocation().getStart();
        int subEnd = subStart + subLoc.size();
        List<ActivityProfileState> subStates = this.supportingStates.isEmpty() ? this.supportingStates : this.supportingStates.subList(subStart, subEnd);
        return new ActiveRegion(subLoc, subStates, this.isActive, this.genomeLocParser, newExtension);
    }

    public ActiveRegion trim(GenomeLoc newExtent) {
        if (newExtent == null) {
            throw new IllegalArgumentException("Active region extent cannot be null");
        }
        GenomeLoc subActive = this.getLocation().intersect(newExtent);
        int requiredOnRight = Math.max(newExtent.getStop() - subActive.getStop(), 0);
        int requiredOnLeft = Math.max(subActive.getStart() - newExtent.getStart(), 0);
        int requiredExtension = Math.min(Math.max(requiredOnLeft, requiredOnRight), this.getExtension());
        return new ActiveRegion(subActive, Collections.<ActivityProfileState>emptyList(), this.isActive, this.genomeLocParser, requiredExtension);
    }
}

