/*
 * Decompiled with CFR 0.152.
 */
package org.broad.igv.gwas;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashSet;
import java.util.Random;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.broad.igv.Globals;
import org.broad.igv.exceptions.ParserException;
import org.broad.igv.feature.genome.Genome;
import org.broad.igv.gwas.GWASData;
import org.broad.igv.util.ParsingUtils;
import org.broad.igv.util.ResourceLocator;
import org.broad.tribble.readers.AsciiLineReader;

public class GWASParser {
    private static final Logger log = Logger.getLogger(GWASParser.class);
    private ResourceLocator locator;
    Genome genome;
    private GWASColumns columns;

    public static boolean isGWASFile(String typeString) {
        return typeString.endsWith(".logistic") || typeString.endsWith(".linear") || typeString.endsWith(".assoc") || typeString.endsWith(".qassoc") || typeString.endsWith(".gwas");
    }

    public GWASParser(ResourceLocator locator, Genome genome) {
        this.locator = locator;
        this.genome = genome;
        this.columns = new GWASColumns();
    }

    public GWASData parseDescriptions(GWASData gData, String hitChr, long hitLocation, int searchStartRow) throws IOException {
        AsciiLineReader reader = null;
        String nextLine = null;
        boolean hitFound = false;
        int cacheSize = gData.getDescriptionCache().getMaxSize();
        int rowCounter = 0;
        try {
            reader = ParsingUtils.openAsciiReader(this.locator);
            String headerLine = reader.readLine();
            if (!this.columns.parseHeader(headerLine)) {
                throw new ParserException("Error while parsing columns line.", 0L, nextLine);
            }
            gData.getDescriptionCache().setHeaderTokens(headerLine);
            int cacheAfter = cacheSize / 2;
            int cacheCounter = 0;
            while (cacheCounter < cacheAfter && (nextLine = reader.readLine()) != null && nextLine.trim().length() > 0) {
                GWASEntry entry;
                nextLine = nextLine.trim();
                if (++rowCounter < searchStartRow || (entry = this.parseLine(nextLine, rowCounter)) == null) continue;
                if (entry.chr.equals(hitChr) && (long)entry.start == hitLocation) {
                    hitFound = true;
                }
                if (hitFound) {
                    ++cacheCounter;
                }
                gData.getDescriptionCache().add(entry.chr, entry.start, entry.p, nextLine);
            }
        }
        catch (ParserException e) {
            throw e;
        }
        catch (Exception e) {
            if (nextLine != null && rowCounter != 0) {
                throw new ParserException(e.getMessage(), (Throwable)e, rowCounter, nextLine);
            }
            throw new RuntimeException(e);
        }
        finally {
            if (reader != null) {
                reader.close();
            }
        }
        return gData;
    }

    public GWASData parse() throws IOException {
        AsciiLineReader reader = null;
        String nextLine = null;
        int rowCounter = 0;
        HashSet<String> chromos = new HashSet<String>();
        GWASEntry lastEntry = null;
        try {
            reader = ParsingUtils.openAsciiReader(this.locator);
            String headerLine = reader.readLine();
            if (!this.columns.parseHeader(headerLine)) {
                throw new ParserException("Error while parsing columns line.", 0L, nextLine);
            }
            GWASData gData = new GWASData();
            int indexCounter = 0;
            while ((nextLine = reader.readLine()) != null && nextLine.trim().length() > 0) {
                int indexSize;
                GWASEntry entry = this.parseLine(nextLine = nextLine.trim(), ++rowCounter);
                if (entry == null) continue;
                gData.addLocation(entry.chr, entry.start);
                gData.addValue(entry.chr, entry.p);
                if (lastEntry != null) {
                    if (entry.chr.equals(lastEntry.chr)) {
                        if (entry.start < lastEntry.start) {
                            throw new ParserException("File is not sorted, found start position lower than previous", rowCounter);
                        }
                    } else {
                        if (chromos.contains(entry.chr)) {
                            throw new ParserException("File is not sorted; chromosome repeated", rowCounter);
                        }
                        chromos.add(entry.chr);
                    }
                }
                if (++indexCounter == (indexSize = 10000)) {
                    gData.getFileIndex().add((int)reader.getPosition());
                    indexCounter = 0;
                }
                lastEntry = entry;
            }
            GWASData gWASData = gData;
            return gWASData;
        }
        catch (Exception e) {
            if (nextLine != null && rowCounter != 0) {
                throw new ParserException(e.getMessage(), (Throwable)e, rowCounter, nextLine);
            }
            throw new RuntimeException(e);
        }
        finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    private GWASEntry parseLine(String nextLine, long lineNumber) {
        String[] tokens = Globals.singleTabMultiSpacePattern.split(nextLine);
        if (tokens.length > 1) {
            int start;
            String chr = this.genome.getChromosomeAlias(tokens[this.columns.chrCol].trim());
            try {
                start = Integer.parseInt(tokens[this.columns.locationCol].trim());
            }
            catch (NumberFormatException e) {
                throw new ParserException("Column " + this.columns.locationCol + " must be a numeric value.", lineNumber, nextLine);
            }
            if (!tokens[this.columns.pCol].trim().equalsIgnoreCase("NA")) {
                double p;
                try {
                    p = Double.parseDouble(tokens[this.columns.pCol]);
                    if (p <= 0.0) {
                        throw new NumberFormatException();
                    }
                    p = -Math.log10(p);
                }
                catch (NumberFormatException e) {
                    throw new ParserException("Column " + this.columns.pCol + " must be a positive numeric value. Found " + tokens[this.columns.pCol], lineNumber, nextLine);
                }
                return new GWASEntry(chr, start, p, nextLine);
            }
        }
        return null;
    }

    public static void generateUnsortedGWASData() throws Exception {
        Random random = new Random(12345L);
        int numFeats = 100;
        String headerLine = StringUtils.join((Object[])new String[]{"Chr", "bp", "p", "snp"}, "\t");
        PrintWriter writer = new PrintWriter("random.gwas");
        writer.println(headerLine);
        for (int ii = 0; ii < numFeats; ++ii) {
            String chr = String.format("chr%d", random.nextInt(20));
            String location = String.format("%d", random.nextInt(Integer.MAX_VALUE));
            String pVal = String.format("%2.8f", random.nextDouble() / 1000.0);
            String rsID = String.format("rs%d", random.nextInt(100000));
            String line = StringUtils.join((Object[])new String[]{chr, location, pVal, rsID}, "\t");
            writer.println(line);
        }
        writer.flush();
        writer.close();
    }

    public static class GWASColumns {
        public int locationCol = -1;
        public int chrCol = -1;
        public int pCol = -1;
        public int SNPCol = -1;

        public boolean hasAllFields() {
            return this.locationCol >= 0 || this.chrCol >= 0 || this.pCol >= 0 || this.SNPCol >= 0;
        }

        public boolean parseHeader(String headerString) {
            String[] headers = Globals.singleTabMultiSpacePattern.split(headerString = headerString.trim());
            int headersSize = headers.length;
            if (headersSize < 4) {
                return false;
            }
            for (int colCounter = 0; colCounter < headersSize; ++colCounter) {
                String header = headers[colCounter];
                if ((header = header.toLowerCase()).equals("chr") || header.equals("chromosome")) {
                    this.chrCol = colCounter;
                }
                if (header.equals("bp") || header.equals("pos") || header.equals("position")) {
                    this.locationCol = colCounter;
                }
                if (header.equals("p") || header.equals("pval") || header.equals("p-value") || header.equals("pvalue") || header.equals("p.value")) {
                    this.pCol = colCounter;
                }
                if (!header.equals("snp") && !header.equals("rs") && !header.equals("rsid") && !header.equals("rsnum") && !header.equals("id") && !header.equals("marker") && !header.equals("markername")) continue;
                this.SNPCol = colCounter;
            }
            return this.hasAllFields();
        }
    }

    private static class GWASEntry {
        private final String chr;
        private final int start;
        private final double p;
        private final String description;

        private GWASEntry(String chr, int start, double p, String description) {
            this.chr = chr;
            this.start = start;
            this.p = p;
            this.description = description;
        }
    }
}

