/*
 * Decompiled with CFR 0.152.
 */
package org.broad.igv.feature.genome;

import java.io.File;
import java.io.IOException;
import java.net.SocketException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.broad.igv.DirectoryManager;
import org.broad.igv.event.GenomeChangeEvent;
import org.broad.igv.event.IGVEventBus;
import org.broad.igv.exceptions.DataLoadException;
import org.broad.igv.feature.FeatureDB;
import org.broad.igv.feature.genome.DotGenomeUtils;
import org.broad.igv.feature.genome.Genome;
import org.broad.igv.feature.genome.GenomeDownloadUtils;
import org.broad.igv.feature.genome.GenomeListItem;
import org.broad.igv.feature.genome.load.ChromAliasParser;
import org.broad.igv.feature.genome.load.GenomeConfig;
import org.broad.igv.feature.genome.load.GenomeLoader;
import org.broad.igv.feature.genome.load.JsonGenomeLoader;
import org.broad.igv.feature.genome.load.TrackConfig;
import org.broad.igv.jbrowse.CircularViewUtilities;
import org.broad.igv.logging.LogManager;
import org.broad.igv.logging.Logger;
import org.broad.igv.prefs.PreferencesManager;
import org.broad.igv.track.FeatureTrack;
import org.broad.igv.track.Track;
import org.broad.igv.ucsc.hub.Hub;
import org.broad.igv.ucsc.hub.HubParser;
import org.broad.igv.ucsc.hub.TrackHubSelectionDialog;
import org.broad.igv.ui.IGV;
import org.broad.igv.ui.PanelName;
import org.broad.igv.ui.WaitCursorManager;
import org.broad.igv.ui.commandbar.GenomeListManager;
import org.broad.igv.ui.panel.FrameManager;
import org.broad.igv.ui.util.MessageUtils;
import org.broad.igv.ui.util.UIUtilities;
import org.broad.igv.util.ParsingUtils;
import org.broad.igv.util.ResourceLocator;

public class GenomeManager {
    public static final String UPDATE_ANNOTATIONS_MESSAGE = "Select default annotation tracks for this genome.";
    public static final String SELECT_ANNOTATIONS_MESSAGE = "Select default annotation tracks for this genome.  You can change these selections later using the 'Genomes > Select Genome Annotations...' menu.";
    private static Logger log = LogManager.getLogger(GenomeManager.class);
    private static GenomeManager theInstance;
    private static GenomeListManager genomeListManager;
    private Genome currentGenome;

    public static synchronized GenomeManager getInstance() {
        if (theInstance == null) {
            theInstance = new GenomeManager();
        }
        return theInstance;
    }

    private GenomeManager() {
        genomeListManager = GenomeListManager.getInstance();
        GenomeLoader.loadSequenceMap();
    }

    public String getGenomeId() {
        return this.currentGenome == null ? null : this.currentGenome.getId();
    }

    public Genome getCurrentGenome() {
        return this.currentGenome;
    }

    public boolean loadGenomeById(String genomeId) throws IOException {
        String genomePath;
        Genome currentGenome = this.getCurrentGenome();
        if (currentGenome != null && genomeId.equals(currentGenome.getId())) {
            return false;
        }
        if (ParsingUtils.fileExists(genomeId)) {
            genomePath = genomeId;
        } else {
            GenomeListItem item = genomeListManager.getGenomeListItem(genomeId);
            if (item == null) {
                MessageUtils.showMessage("Could not locate genome with ID: " + genomeId);
                return false;
            }
            genomePath = item.getPath();
        }
        return this.loadGenome(genomePath) != null;
    }

    public Genome loadGenome(String genomePath) throws IOException {
        Genome genome;
        WaitCursorManager.CursorToken cursorToken = null;
        try {
            Map<String, List<String>> hubUrlMap;
            List<String> hubUrls;
            log.info("Loading genome: " + genomePath);
            if (IGV.hasInstance()) {
                IGV.getInstance().setStatusBarMessage("<html><font color=blue>Loading genome</font></html>");
                cursorToken = WaitCursorManager.showWaitCursor();
            }
            FeatureDB.clearFeatures();
            Genome newGenome = GenomeLoader.getLoader(genomePath).loadGenome();
            if (newGenome.getTrackHubs().isEmpty() && (hubUrls = (hubUrlMap = HubParser.getHubURLs()).get(newGenome.getUCSCId())) != null && !hubUrls.isEmpty()) {
                newGenome.addTrackHubs(HubParser.loadHubs(newGenome.getUCSCId(), hubUrls));
            }
            File genomeCacheDirectory = DirectoryManager.getGenomeCacheDirectory();
            try {
                String aliasPath = new File(genomeCacheDirectory, newGenome.getId() + "_alias.tab").getAbsolutePath();
                if (!new File(aliasPath).exists()) {
                    aliasPath = new File(genomeCacheDirectory, newGenome.getId() + "_alias.tab.txt").getAbsolutePath();
                }
                if (new File(aliasPath).exists()) {
                    newGenome.addChrAliases(ChromAliasParser.loadChrAliases(aliasPath));
                }
            }
            catch (Exception e) {
                log.error("Failed to load user defined alias", e);
            }
            if (IGV.hasInstance()) {
                IGV.getInstance().resetSession(null);
            }
            GenomeListItem genomeListItem = new GenomeListItem(newGenome.getDisplayName(), genomePath, newGenome.getId());
            GenomeListManager.getInstance().addGenomeItem(genomeListItem);
            this.setCurrentGenome(newGenome);
            genome = this.currentGenome;
        }
        catch (SocketException e) {
            try {
                throw new RuntimeException("Server connection error", e);
            }
            catch (Throwable throwable) {
                if (IGV.hasInstance()) {
                    IGV.getInstance().setStatusBarMessage("");
                    WaitCursorManager.removeWaitCursor(cursorToken);
                }
                throw throwable;
            }
        }
        if (IGV.hasInstance()) {
            IGV.getInstance().setStatusBarMessage("");
            WaitCursorManager.removeWaitCursor(cursorToken);
        }
        return genome;
    }

    public void setCurrentGenome(Genome newGenome) {
        this.currentGenome = newGenome;
        if (IGV.hasInstance()) {
            IGV.getInstance().goToLocus(newGenome.getHomeChromosome());
            FrameManager.getDefaultFrame().setChromosomeName(newGenome.getHomeChromosome(), true);
            this.restoreGenomeTracks(newGenome);
            IGV.getInstance().resetFrames();
            IGV.getInstance().getSession().clearHistory();
            if (newGenome != Genome.nullGenome()) {
                PreferencesManager.getPreferences().setLastGenome(newGenome.getId());
            }
            if (PreferencesManager.getPreferences().getAsBoolean("CIRC_VIEW_ENABLED") && CircularViewUtilities.ping()) {
                CircularViewUtilities.changeGenome(newGenome);
            }
            IGVEventBus.getInstance().post(new GenomeChangeEvent(newGenome));
        }
    }

    public void restoreGenomeTracks(Genome genome) {
        IGV.getInstance().setSequenceTrack();
        FeatureTrack geneFeatureTrack = genome.getGeneTrack();
        if (geneFeatureTrack != null) {
            geneFeatureTrack.setAttributeValue("NAME", geneFeatureTrack.getName());
            geneFeatureTrack.setAttributeValue("DATA FILE", "");
            geneFeatureTrack.setAttributeValue("DATA TYPE", geneFeatureTrack.getTrackType().toString());
            IGV.getInstance().addTrack(geneFeatureTrack, PanelName.ANNOTATION_PANEL.getName());
        }
        List<ResourceLocator> resources = genome.getAnnotationResources();
        ArrayList<Track> annotationTracks = new ArrayList<Track>();
        if (resources != null) {
            for (ResourceLocator locator : resources) {
                try {
                    if (locator == null) continue;
                    locator.setPanelName(PanelName.ANNOTATION_PANEL.getName());
                    List<Track> tracks = IGV.getInstance().load(locator);
                    annotationTracks.addAll(tracks);
                }
                catch (DataLoadException e) {
                    log.error("Error loading genome annotations", e);
                }
            }
        }
        if (annotationTracks.size() > 0) {
            IGV.getInstance().addTracks(annotationTracks);
            for (Track track : annotationTracks) {
                ResourceLocator locator = track.getResourceLocator();
                if (locator == null) continue;
                String fn = "";
                if (locator != null) {
                    fn = locator.getPath();
                    int lastSlashIdx = fn.lastIndexOf("/");
                    if (lastSlashIdx < 0) {
                        lastSlashIdx = fn.lastIndexOf("\\");
                    }
                    if (lastSlashIdx > 0) {
                        fn = fn.substring(lastSlashIdx + 1);
                    }
                }
                track.setAttributeValue("NAME", track.getName());
                track.setAttributeValue("DATA FILE", fn);
                track.setAttributeValue("DATA TYPE", track.getTrackType().toString());
            }
        }
        IGV.getInstance().revalidateTrackPanels();
    }

    public File downloadGenome(GenomeListItem item, boolean downloadSequence, boolean downloadAnnotations) {
        try {
            List<TrackConfig> selectedTracks;
            if (item.getPath().endsWith(".genome")) {
                File genomeFile = DotGenomeUtils.getDotGenomeFile(item.getPath());
                return genomeFile;
            }
            JsonGenomeLoader loader = new JsonGenomeLoader(item.getPath());
            GenomeConfig config = loader.loadGenomeConfig();
            if (config.getHubs() != null && config.getHubs().size() > 0 && (selectedTracks = this.selectAnnotationTracks(config, SELECT_ANNOTATIONS_MESSAGE)) != null && selectedTracks.size() > 0) {
                config.setTracks(selectedTracks);
            }
            File downloadedGenome = GenomeDownloadUtils.downloadGenome(config, downloadSequence, downloadAnnotations);
            return downloadedGenome;
        }
        catch (Exception e) {
            MessageUtils.showMessage("Error downloading genome: " + e.getMessage());
            log.error("Error downloading genome " + item.getDisplayableName());
            return null;
        }
    }

    public void updateAnnotations() throws IOException {
        GenomeConfig config;
        if (this.currentGenome != null && (config = this.currentGenome.getConfig()) != null) {
            List<TrackConfig> trackConfigs = config.getTrackConfigs();
            List<String> currentAnnotationPaths = trackConfigs == null ? Collections.EMPTY_LIST : trackConfigs.stream().map(t -> t.url).toList();
            List<TrackConfig> selectedConfigs = this.selectAnnotationTracks(config, UPDATE_ANNOTATIONS_MESSAGE);
            if (selectedConfigs == null) {
                return;
            }
            config.setTracks(selectedConfigs);
            GenomeDownloadUtils.saveLocalGenome(config);
            Set selectedTrackPaths = selectedConfigs.stream().map(t -> t.url).collect(Collectors.toSet());
            HashSet<String> pathsToRemove = new HashSet<String>();
            for (String p : currentAnnotationPaths) {
                if (selectedTrackPaths.contains(p)) continue;
                pathsToRemove.add(p);
            }
            IGV.getInstance().deleteTracksByPath(pathsToRemove);
            Set loadedTrackPaths = IGV.getInstance().getAllTracks().stream().filter(t -> t.getResourceLocator() != null).map(t -> t.getResourceLocator().getPath()).collect(Collectors.toSet());
            List tracksToLoad = selectedConfigs.stream().filter(trackConfig -> !loadedTrackPaths.contains(trackConfig.url)).collect(Collectors.toList());
            List<ResourceLocator> locators = tracksToLoad.stream().map(t -> ResourceLocator.fromTrackConfig(t)).toList();
            for (ResourceLocator locator : locators) {
                locator.setPanelName(PanelName.ANNOTATION_PANEL.getName());
            }
            IGV.getInstance().loadTracks(locators);
        }
    }

    private List<TrackConfig> selectAnnotationTracks(GenomeConfig config, String message) throws IOException {
        String annotationHub = config.getHubs().get(0);
        Hub hub = HubParser.loadHub(annotationHub, config.getUcscID());
        Set<String> currentSelections = config.getTrackConfigs() == null ? Collections.emptySet() : config.getTrackConfigs().stream().map(trackConfig -> trackConfig.url).collect(Collectors.toSet());
        TrackHubSelectionDialog dlg = TrackHubSelectionDialog.getTrackHubSelectionDialog(hub, currentSelections, true, message);
        try {
            UIUtilities.invokeAndWaitOnEventThread(() -> dlg.setVisible(true));
            if (dlg.isCanceled()) {
                return null;
            }
            return dlg.getSelectedConfigs();
        }
        catch (Exception e) {
            log.error("Error opening or using TrackHubSelectionDialog: " + e.getMessage());
            return null;
        }
    }

    public void deleteDownloadedGenomes(List<GenomeListItem> removedValuesList) throws IOException {
        for (GenomeListItem item : removedValuesList) {
            String loc = item.getPath();
            File genomeFile = new File(loc);
            if (!genomeFile.exists() || !DirectoryManager.isChildOf(DirectoryManager.getGenomeCacheDirectory(), genomeFile)) continue;
            genomeFile.delete();
            File dataFileDirectory = new File(DirectoryManager.getGenomeCacheDirectory(), item.getId());
            File localFasta = DotGenomeUtils.getLocalFasta(item.getId());
            if (!dataFileDirectory.isDirectory() && localFasta == null || !MessageUtils.confirm("Delete downloaded data files?")) continue;
            if (dataFileDirectory.isDirectory()) {
                try (Stream<Path> paths = Files.walk(dataFileDirectory.toPath(), new FileVisitOption[0]);){
                    paths.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
                }
                dataFileDirectory.delete();
            }
            if (localFasta == null) continue;
            DotGenomeUtils.removeLocalFasta(item.getId());
            if (!DirectoryManager.isChildOf(DirectoryManager.getGenomeCacheDirectory(), localFasta) || !MessageUtils.confirm("Delete fasta file: " + localFasta.getAbsolutePath() + "?")) continue;
            localFasta.delete();
            File indexFile = new File(localFasta.getAbsolutePath() + ".fai");
            if (!indexFile.exists()) continue;
            indexFile.delete();
        }
    }

    public void setCurrentGenomeForTest(Genome genome) {
        this.currentGenome = genome;
    }
}

