/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.util;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.SAMTextHeaderCodec;
import htsjdk.samtools.util.BufferedLineReader;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalCoordinateComparator;
import htsjdk.samtools.util.IntervalListWriter;
import htsjdk.samtools.util.ListMap;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.OverlapDetector;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import htsjdk.tribble.IntervalList.IntervalListCodec;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

public class IntervalList
implements Iterable<Interval> {
    public static final String INTERVAL_LIST_FILE_EXTENSION = ".interval_list";
    private final SAMFileHeader header;
    private final List<Interval> intervals = new ArrayList<Interval>();
    private static final Log log = Log.getInstance(IntervalList.class);

    public IntervalList(SAMFileHeader header) {
        if (header == null) {
            throw new IllegalArgumentException("SAMFileHeader must be supplied.");
        }
        this.header = header;
    }

    public IntervalList(SAMSequenceDictionary dict) {
        this(new SAMFileHeader(dict));
    }

    public SAMFileHeader getHeader() {
        return this.header;
    }

    @Override
    public Iterator<Interval> iterator() {
        return this.intervals.iterator();
    }

    public void add(Interval interval) {
        if (this.header.getSequence(interval.getContig()) == null) {
            throw new IllegalArgumentException(String.format("Cannot add interval %s, contig not in header", interval.toString()));
        }
        this.intervals.add(interval);
    }

    public void addall(Collection<Interval> intervals) {
        for (Interval interval : intervals) {
            this.add(interval);
        }
    }

    @Deprecated
    public void sort() {
        Collections.sort(this.intervals, new IntervalCoordinateComparator(this.header));
        this.header.setSortOrder(SAMFileHeader.SortOrder.coordinate);
    }

    public IntervalList padded(int before, int after) {
        if (before < 0 || after < 0) {
            throw new IllegalArgumentException("Padding values must be >= 0.");
        }
        IntervalList padded = new IntervalList(this.getHeader().clone());
        SAMSequenceDictionary dict = padded.getHeader().getSequenceDictionary();
        for (Interval i : this) {
            SAMSequenceRecord seq = dict.getSequence(i.getContig());
            int start = Math.max(1, i.getStart() - before);
            int end = Math.min(seq.getSequenceLength(), i.getEnd() + after);
            padded.add(new Interval(i.getContig(), start, end, i.isNegativeStrand(), i.getName()));
        }
        return padded;
    }

    public IntervalList padded(int padding) {
        return this.padded(padding, padding);
    }

    public IntervalList sorted() {
        IntervalList sorted = IntervalList.copyOf(this);
        sorted.intervals.sort(new IntervalCoordinateComparator(sorted.header));
        sorted.header.setSortOrder(SAMFileHeader.SortOrder.coordinate);
        return sorted;
    }

    public IntervalList uniqued() {
        return this.uniqued(true);
    }

    public IntervalList uniqued(boolean concatenateNames) {
        List<Interval> tmp = IntervalList.getUniqueIntervals(this.sorted(), concatenateNames);
        IntervalList value = new IntervalList(this.header.clone());
        value.intervals.addAll(tmp);
        return value;
    }

    @Deprecated
    public void unique() {
        this.unique(true);
    }

    @Deprecated
    public void unique(boolean concatenateNames) {
        this.sort();
        List<Interval> tmp = this.getUniqueIntervals(concatenateNames);
        this.intervals.clear();
        this.intervals.addAll(tmp);
    }

    public List<Interval> getIntervals() {
        return Collections.unmodifiableList(this.intervals);
    }

    @Deprecated
    public List<Interval> getUniqueIntervals() {
        return this.getUniqueIntervals(true);
    }

    public static List<Interval> getUniqueIntervals(IntervalList list, boolean concatenateNames) {
        return IntervalList.getUniqueIntervals(list, true, concatenateNames, false);
    }

    public static List<Interval> getUniqueIntervals(IntervalList list, boolean concatenateNames, boolean enforceSameStrands) {
        return IntervalList.getUniqueIntervals(list, true, concatenateNames, enforceSameStrands);
    }

    public static List<Interval> getUniqueIntervals(IntervalList list, boolean combineAbuttingIntervals, boolean concatenateNames, boolean enforceSameStrands) {
        List<Interval> intervals = list.getHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate ? list.sorted().intervals : list.intervals;
        ArrayList<Interval> unique = new ArrayList<Interval>();
        ArrayList<Interval> toBeMerged = new ArrayList<Interval>();
        Interval current = null;
        for (Interval next : intervals) {
            if (current == null) {
                toBeMerged.add(next);
                current = next;
                continue;
            }
            if (current.intersects(next) || combineAbuttingIntervals && current.abuts(next)) {
                if (enforceSameStrands && current.isNegativeStrand() != next.isNegativeStrand()) {
                    throw new SAMException("Strands were not equal for: " + current.toString() + " and " + next.toString());
                }
                toBeMerged.add(next);
                current = new Interval(current.getContig(), current.getStart(), Math.max(current.getEnd(), next.getEnd()), current.isNegativeStrand(), null);
                continue;
            }
            unique.add(IntervalList.merge(toBeMerged, concatenateNames));
            toBeMerged.clear();
            current = next;
            toBeMerged.add(current);
        }
        if (!toBeMerged.isEmpty()) {
            unique.add(IntervalList.merge(toBeMerged, concatenateNames));
        }
        return unique;
    }

    @Deprecated
    public List<Interval> getUniqueIntervals(boolean concatenateNames) {
        if (this.getHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            this.sort();
        }
        return IntervalList.getUniqueIntervals(this, concatenateNames);
    }

    public static List<Interval> breakIntervalsAtBandMultiples(List<Interval> intervals, int bandMultiple) {
        ArrayList<Interval> brokenUpIntervals = new ArrayList<Interval>();
        for (Interval interval : intervals) {
            if (interval.getEnd() >= interval.getStart()) {
                int endIndex;
                int startIndex = interval.getStart() / bandMultiple;
                if (startIndex == (endIndex = interval.getEnd() / bandMultiple)) {
                    brokenUpIntervals.add(interval);
                    continue;
                }
                brokenUpIntervals.addAll(IntervalList.breakIntervalAtBandMultiples(interval, bandMultiple));
                continue;
            }
            brokenUpIntervals.add(interval);
        }
        return brokenUpIntervals;
    }

    private static List<Interval> breakIntervalAtBandMultiples(Interval interval, int bandMultiple) {
        int startOfIntervalIndex;
        ArrayList<Interval> brokenUpIntervals = new ArrayList<Interval>();
        int startPos = interval.getStart();
        int startIndex = startOfIntervalIndex = startPos / bandMultiple;
        int endIndex = interval.getEnd() / bandMultiple;
        while (startIndex <= endIndex) {
            int endPos = (startIndex + 1) * bandMultiple - 1;
            if (endPos > interval.getEnd()) {
                endPos = interval.getEnd();
            }
            brokenUpIntervals.add(new Interval(interval.getContig(), startPos, endPos, interval.isNegativeStrand(), interval.getName() + "." + (startIndex - startOfIntervalIndex + 1)));
            startPos = ++startIndex * bandMultiple;
        }
        return brokenUpIntervals;
    }

    static Interval merge(Iterable<Interval> intervals, boolean concatenateNames) {
        Interval first = intervals.iterator().next();
        String chrom = first.getContig();
        int start = first.getStart();
        int end = first.getEnd();
        boolean neg = first.isNegativeStrand();
        LinkedHashSet<String> names = new LinkedHashSet<String>();
        for (Interval i : intervals) {
            if (i.getName() != null) {
                names.add(i.getName());
            }
            start = Math.min(start, i.getStart());
            end = Math.max(end, i.getEnd());
        }
        String name = names.isEmpty() ? null : (concatenateNames ? StringUtil.join("|", names) : (String)names.iterator().next());
        return new Interval(chrom, start, end, neg, name);
    }

    public long getBaseCount() {
        return Interval.countBases(this.intervals);
    }

    public long getUniqueBaseCount() {
        return this.uniqued().getBaseCount();
    }

    public int size() {
        return this.intervals.size();
    }

    public static IntervalList copyOf(IntervalList list) {
        IntervalList clone = new IntervalList(list.header.clone());
        clone.intervals.addAll(list.intervals);
        return clone;
    }

    public static IntervalList fromFile(File file) {
        return IntervalList.fromPath(IOUtil.toPath(file));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static IntervalList fromPath(Path path) {
        try (BufferedReader reader = IOUtil.openFileForBufferedReading(path);){
            IntervalList intervalList = IntervalList.fromReader(reader);
            return intervalList;
        }
        catch (IOException e) {
            throw new SAMException(String.format("Failed to close file %s after reading", path.toUri().toString()));
        }
    }

    public static IntervalList fromName(SAMFileHeader header, String sequenceName) {
        IntervalList ref = new IntervalList(header);
        ref.add(new Interval(sequenceName, 1, header.getSequence(sequenceName).getSequenceLength()));
        return ref;
    }

    public static IntervalList fromFiles(Collection<File> intervalListFiles) {
        ArrayList<IntervalList> intervalLists = new ArrayList<IntervalList>();
        for (File file : intervalListFiles) {
            intervalLists.add(IntervalList.fromFile(file));
        }
        return IntervalList.concatenate(intervalLists);
    }

    public static IntervalList fromReader(BufferedReader in) {
        try {
            StringBuilder builder = new StringBuilder(4096);
            String line = null;
            while ((line = in.readLine()) != null && line.startsWith("@")) {
                builder.append(line).append('\n');
            }
            if (builder.length() == 0) {
                throw new IllegalStateException("Interval list file must contain header. ");
            }
            BufferedLineReader headerReader = BufferedLineReader.fromString(builder.toString());
            SAMTextHeaderCodec codec = new SAMTextHeaderCodec();
            IntervalList list = new IntervalList(codec.decode(headerReader, "BufferedReader"));
            SAMSequenceDictionary dict = list.getHeader().getSequenceDictionary();
            if (line == null) {
                return list;
            }
            IntervalListCodec intervalListCodec = new IntervalListCodec(dict);
            do {
                Optional<Interval> maybeInterval = Optional.ofNullable(intervalListCodec.decode(line));
                maybeInterval.ifPresent(list.intervals::add);
            } while ((line = in.readLine()) != null);
            return list;
        }
        catch (IOException ioe) {
            throw new SAMException("Error parsing interval list.", ioe);
        }
    }

    public void write(Path path) {
        try (IntervalListWriter writer = new IntervalListWriter(path, this.header);){
            for (Interval interval : this) {
                writer.write(interval);
            }
        }
        catch (IOException ioe) {
            throw new SAMException("Error writing out interval list to file: " + path.toAbsolutePath(), ioe);
        }
    }

    public void write(File file) {
        this.write(file.toPath());
    }

    public static IntervalList intersection(IntervalList list1, IntervalList list2) {
        SequenceUtil.assertSequenceDictionariesEqual(list1.getHeader().getSequenceDictionary(), list2.getHeader().getSequenceDictionary());
        IntervalList result = new IntervalList(list1.getHeader().clone());
        OverlapDetector<Interval> detector = new OverlapDetector<Interval>(0, 0);
        detector.addAll(list1.getIntervals(), list1.getIntervals());
        for (Interval i : list2.getIntervals()) {
            Set as = detector.getOverlaps(i);
            for (Interval j : as) {
                Interval tmp = i.intersect(j);
                result.add(tmp);
            }
        }
        return result.uniqued();
    }

    public static IntervalList intersection(Collection<IntervalList> lists) {
        IntervalList intersection = null;
        for (IntervalList list : lists) {
            if (intersection == null) {
                intersection = list;
                continue;
            }
            intersection = IntervalList.intersection(intersection, list);
        }
        return intersection;
    }

    public static IntervalList concatenate(Collection<IntervalList> lists) {
        if (lists.isEmpty()) {
            throw new SAMException("Cannot concatenate an empty list of IntervalLists.");
        }
        SAMFileHeader header = lists.iterator().next().getHeader().clone();
        header.setSortOrder(SAMFileHeader.SortOrder.unsorted);
        IntervalList merged = new IntervalList(header);
        for (IntervalList in : lists) {
            SequenceUtil.assertSequenceDictionariesEqual(merged.getHeader().getSequenceDictionary(), in.getHeader().getSequenceDictionary());
            merged.addall(in.intervals);
        }
        return merged;
    }

    public static IntervalList union(Collection<IntervalList> lists) {
        IntervalList merged = IntervalList.concatenate(lists);
        return merged.uniqued();
    }

    public static IntervalList union(IntervalList list1, IntervalList list2) {
        List<IntervalList> duo = CollectionUtil.makeList(list1, list2);
        return IntervalList.union(duo);
    }

    public static IntervalList invert(IntervalList list) {
        IntervalList inverse = new IntervalList(list.header.clone());
        ListMap<Integer, Interval> map = new ListMap<Integer, Interval>();
        for (Interval i : list.uniqued().getIntervals()) {
            map.add(list.getHeader().getSequenceIndex(i.getContig()), i);
        }
        int intervals = 0;
        for (SAMSequenceRecord samSequenceRecord : list.getHeader().getSequenceDictionary().getSequences()) {
            Integer sequenceIndex = samSequenceRecord.getSequenceIndex();
            String sequenceName = samSequenceRecord.getSequenceName();
            int sequenceLength = samSequenceRecord.getSequenceLength();
            int lastCoveredPosition = 0;
            if (map.containsKey(sequenceIndex)) {
                for (Interval i : (List)map.get(sequenceIndex)) {
                    if (i.getStart() > lastCoveredPosition + 1) {
                        inverse.add(new Interval(sequenceName, lastCoveredPosition + 1, i.getStart() - 1, false, "interval-" + ++intervals));
                    }
                    lastCoveredPosition = i.getEnd();
                }
            }
            if (sequenceLength <= lastCoveredPosition) continue;
            inverse.add(new Interval(sequenceName, lastCoveredPosition + 1, sequenceLength, false, "interval-" + ++intervals));
        }
        return inverse;
    }

    public static IntervalList subtract(Collection<IntervalList> lhs, Collection<IntervalList> rhs) {
        return IntervalList.intersection(IntervalList.union(lhs), IntervalList.invert(IntervalList.union(rhs)));
    }

    public static IntervalList subtract(IntervalList lhs, IntervalList rhs) {
        return IntervalList.subtract(Collections.singletonList(lhs), Collections.singletonList(rhs));
    }

    public static IntervalList difference(Collection<IntervalList> lists1, Collection<IntervalList> lists2) {
        return IntervalList.union(IntervalList.subtract(lists1, lists2), IntervalList.subtract(lists2, lists1));
    }

    public static IntervalList overlaps(IntervalList lhs, IntervalList rhs) {
        return IntervalList.overlaps(Collections.singletonList(lhs), Collections.singletonList(rhs));
    }

    public static IntervalList overlaps(Collection<IntervalList> lists1, Collection<IntervalList> lists2) {
        if (lists1.isEmpty()) {
            throw new SAMException("Cannot call overlaps with the first collection having empty list of IntervalLists.");
        }
        SAMFileHeader header = lists1.iterator().next().getHeader().clone();
        header.setSortOrder(SAMFileHeader.SortOrder.unsorted);
        IntervalList overlapIntervals = new IntervalList(header);
        for (IntervalList list : lists2) {
            SequenceUtil.assertSequenceDictionariesEqual(header.getSequenceDictionary(), list.getHeader().getSequenceDictionary());
            overlapIntervals.addall(list.getIntervals());
        }
        OverlapDetector<Integer> detector = new OverlapDetector<Integer>(0, 0);
        int dummy = -1;
        for (Interval interval : overlapIntervals.sorted().uniqued()) {
            detector.addLhs(-1, interval);
        }
        IntervalList merged = new IntervalList(header);
        for (IntervalList list : lists1) {
            SequenceUtil.assertSequenceDictionariesEqual(header.getSequenceDictionary(), list.getHeader().getSequenceDictionary());
            for (Interval interval : list.getIntervals()) {
                if (!detector.overlapsAny(interval)) continue;
                merged.add(interval);
            }
        }
        return merged;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        IntervalList intervals1 = (IntervalList)o;
        return this.header.equals(intervals1.header) && this.intervals.equals(intervals1.intervals);
    }

    public int hashCode() {
        int result = this.header.hashCode();
        result = 31 * result + this.intervals.hashCode();
        return result;
    }
}

