/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools;

import htsjdk.samtools.AbstractBAMFileIndex;
import htsjdk.samtools.BAMIndexContent;
import htsjdk.samtools.BAMIndexMetaData;
import htsjdk.samtools.BAMIndexWriter;
import htsjdk.samtools.Bin;
import htsjdk.samtools.BinaryBAMIndexWriter;
import htsjdk.samtools.Chunk;
import htsjdk.samtools.GenomicIndexUtil;
import htsjdk.samtools.LinearIndex;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.ValidationStringency;
import htsjdk.samtools.cram.build.CramIO;
import htsjdk.samtools.cram.ref.ReferenceContext;
import htsjdk.samtools.cram.structure.AlignmentSpan;
import htsjdk.samtools.cram.structure.Container;
import htsjdk.samtools.cram.structure.ContainerIO;
import htsjdk.samtools.cram.structure.CramHeader;
import htsjdk.samtools.cram.structure.Slice;
import htsjdk.samtools.seekablestream.SeekableStream;
import htsjdk.samtools.util.BlockCompressedFilePointerUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import java.io.File;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

public class CRAMBAIIndexer {
    private final int numReferences;
    private final BAMIndexWriter outputWriter;
    private int currentReference = 0;
    private final CRAMBAIIndexBuilder indexBuilder;

    private CRAMBAIIndexer(File output, SAMFileHeader fileHeader) {
        if (fileHeader.getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new SAMException("CRAM file must be coordinate-sorted for indexing.");
        }
        this.numReferences = fileHeader.getSequenceDictionary().size();
        this.indexBuilder = new CRAMBAIIndexBuilder(fileHeader);
        this.outputWriter = new BinaryBAMIndexWriter(this.numReferences, output);
    }

    public CRAMBAIIndexer(OutputStream output, SAMFileHeader fileHeader) {
        if (fileHeader.getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new SAMException("CRAM file must be coordinate-sorted for indexing.");
        }
        this.numReferences = fileHeader.getSequenceDictionary().size();
        this.indexBuilder = new CRAMBAIIndexBuilder(fileHeader);
        this.outputWriter = new BinaryBAMIndexWriter(this.numReferences, output);
    }

    void processContainer(Container container, ValidationStringency validationStringency) {
        if (container == null || container.isEOF()) {
            return;
        }
        int sliceIndex = 0;
        for (Slice slice : container.getSlices()) {
            slice.index = sliceIndex++;
            if (slice.getReferenceContext().isMultiRef()) {
                Map<ReferenceContext, AlignmentSpan> spanMap = container.getSpans(validationStringency);
                slice.index = sliceIndex++;
                AlignmentSpan unmappedSpan = spanMap.remove(ReferenceContext.UNMAPPED_UNPLACED_CONTEXT);
                for (ReferenceContext refContext : new TreeSet<ReferenceContext>(spanMap.keySet())) {
                    AlignmentSpan span = spanMap.get(refContext);
                    Slice fakeSlice = new Slice(refContext);
                    fakeSlice.containerByteOffset = slice.containerByteOffset;
                    fakeSlice.byteOffsetFromCompressionHeaderStart = slice.byteOffsetFromCompressionHeaderStart;
                    fakeSlice.index = slice.index;
                    fakeSlice.alignmentStart = span.getStart();
                    fakeSlice.alignmentSpan = span.getSpan();
                    fakeSlice.mappedReadsCount = span.getMappedCount();
                    fakeSlice.unmappedReadsCount = span.getUnmappedCount();
                    fakeSlice.unplacedReadsCount = 0;
                    this.processAsSingleReferenceSlice(fakeSlice);
                }
                if (unmappedSpan == null) continue;
                Slice fakeSlice = new Slice(ReferenceContext.UNMAPPED_UNPLACED_CONTEXT);
                fakeSlice.containerByteOffset = slice.containerByteOffset;
                fakeSlice.byteOffsetFromCompressionHeaderStart = slice.byteOffsetFromCompressionHeaderStart;
                fakeSlice.index = slice.index;
                fakeSlice.alignmentStart = 0;
                fakeSlice.alignmentSpan = 0;
                fakeSlice.mappedReadsCount = 0;
                fakeSlice.unmappedReadsCount = 0;
                fakeSlice.unplacedReadsCount = slice.unplacedReadsCount;
                this.processAsSingleReferenceSlice(fakeSlice);
                continue;
            }
            this.processAsSingleReferenceSlice(slice);
        }
    }

    public void processAsSingleReferenceSlice(Slice slice) {
        slice.baiIndexInitializationCheck();
        ReferenceContext sliceContext = slice.getReferenceContext();
        if (sliceContext.isMultiRef()) {
            throw new SAMException("Expecting a single reference or unmapped slice.");
        }
        if (sliceContext.isMappedSingleRef()) {
            int reference = sliceContext.getSequenceId();
            if (reference != this.currentReference) {
                this.advanceToReference(reference);
            }
            if (reference != this.currentReference) {
                throw new SAMException("Unexpected reference " + reference + " when constructing index for " + this.currentReference + " for record " + slice);
            }
        }
        this.indexBuilder.recordSliceIndexMetadata(slice);
        if (sliceContext.isMappedSingleRef()) {
            this.indexBuilder.processSingleReferenceSlice(slice);
        }
    }

    public void finish() {
        this.advanceToReference(this.numReferences);
        this.outputWriter.writeNoCoordinateRecordCount(this.indexBuilder.getNoCoordinateRecordCount());
        this.outputWriter.close();
    }

    private void advanceToReference(int nextReference) {
        while (this.currentReference < nextReference) {
            BAMIndexContent content = this.indexBuilder.processCurrentReference();
            this.outputWriter.writeReference(content);
            ++this.currentReference;
            this.indexBuilder.startNewReference();
        }
    }

    public static void createIndex(SeekableStream stream, File output, Log log, ValidationStringency validationStringency) {
        CramHeader cramHeader = CramIO.readCramHeader(stream);
        if (cramHeader.getSamFileHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new SAMException("Expecting a coordinate sorted file.");
        }
        CRAMBAIIndexer indexer = new CRAMBAIIndexer(output, cramHeader.getSamFileHeader());
        Container container = null;
        ProgressLogger progressLogger = new ProgressLogger(log, 1, "indexed", "slices");
        while ((container = ContainerIO.readContainer(cramHeader.getVersion(), stream)) != null && !container.isEOF()) {
            indexer.processContainer(container, validationStringency);
            if (null != log) {
                String sequenceName;
                ReferenceContext containerContext = container.getReferenceContext();
                switch (containerContext.getType()) {
                    case UNMAPPED_UNPLACED_TYPE: {
                        sequenceName = "?";
                        break;
                    }
                    case MULTIPLE_REFERENCE_TYPE: {
                        sequenceName = "???";
                        break;
                    }
                    default: {
                        sequenceName = cramHeader.getSamFileHeader().getSequence(containerContext.getSequenceId()).getSequenceName();
                    }
                }
                progressLogger.record(sequenceName, container.alignmentStart);
            }
            if (!container.isEOF()) continue;
        }
        indexer.finish();
    }

    private class CRAMBAIIndexBuilder {
        private final SAMFileHeader bamHeader;
        private Bin[] bins;
        private int binsSeen = 0;
        private final long[] index = new long[LinearIndex.MAX_LINEAR_INDEX_SIZE];
        private int largestIndexSeen = -1;
        private final BAMIndexMetaData indexStats = new BAMIndexMetaData();

        private CRAMBAIIndexBuilder(SAMFileHeader header) {
            this.bamHeader = header;
        }

        private SAMFileHeader getBamHeader() {
            return this.bamHeader;
        }

        private void recordSliceIndexMetadata(Slice slice) {
            this.indexStats.recordMetaData(slice);
        }

        private int computeIndexingBin(Slice slice) {
            int alignmentEnd = slice.alignmentStart + slice.alignmentSpan - 1;
            int alignmentStart = slice.alignmentStart - 1;
            if (alignmentEnd <= alignmentStart) {
                alignmentEnd = alignmentStart + 1;
            }
            return GenomicIndexUtil.regionToBin(alignmentStart, alignmentEnd);
        }

        private void processSingleReferenceSlice(Slice slice) {
            Bin bin;
            ReferenceContext sliceContext = slice.getReferenceContext();
            if (!sliceContext.isMappedSingleRef()) {
                return;
            }
            int reference = sliceContext.getSequenceId();
            if (reference != CRAMBAIIndexer.this.currentReference) {
                throw new SAMException("Unexpected reference " + reference + " when constructing index for " + CRAMBAIIndexer.this.currentReference + " for record " + slice);
            }
            int binNum = this.computeIndexingBin(slice);
            if (this.bins == null) {
                SAMSequenceRecord seq = this.bamHeader.getSequence(reference);
                this.bins = seq == null ? new Bin[37451] : new Bin[AbstractBAMFileIndex.getMaxBinNumberForSequenceLength(seq.getSequenceLength()) + 1];
            }
            if (this.bins[binNum] != null) {
                bin = this.bins[binNum];
            } else {
                this.bins[binNum] = bin = new Bin(reference, binNum);
                ++this.binsSeen;
            }
            long chunkStart = slice.containerByteOffset << 16 | (long)slice.index;
            long chunkEnd = (slice.containerByteOffset << 16 | (long)slice.index) + 1L;
            Chunk newChunk = new Chunk(chunkStart, chunkEnd);
            List<Chunk> oldChunks = bin.getChunkList();
            if (!bin.containsChunks()) {
                bin.addInitialChunk(newChunk);
            } else {
                Chunk lastChunk = bin.getLastChunk();
                if (BlockCompressedFilePointerUtil.areInSameOrAdjacentBlocks(lastChunk.getChunkEnd(), chunkStart)) {
                    lastChunk.setChunkEnd(chunkEnd);
                } else {
                    oldChunks.add(newChunk);
                    bin.setLastChunk(newChunk);
                }
            }
            int alignmentStart = slice.alignmentStart;
            int alignmentEnd = slice.alignmentStart + slice.alignmentSpan;
            int startWindow = LinearIndex.convertToLinearIndexOffset(alignmentStart);
            int endWindow = alignmentEnd == 0 ? (startWindow = LinearIndex.convertToLinearIndexOffset(alignmentStart - 1)) : LinearIndex.convertToLinearIndexOffset(alignmentEnd);
            if (endWindow > this.largestIndexSeen) {
                this.largestIndexSeen = endWindow;
            }
            for (int win = startWindow; win <= endWindow; ++win) {
                if (this.index[win] != 0L && chunkStart >= this.index[win]) continue;
                this.index[win] = chunkStart;
            }
        }

        private BAMIndexContent processCurrentReference() {
            if (this.binsSeen == 0) {
                return null;
            }
            long[] newIndex = new long[this.largestIndexSeen + 1];
            long lastNonZeroOffset = 0L;
            for (int i = 0; i <= this.largestIndexSeen; ++i) {
                if (this.index[i] == 0L) {
                    this.index[i] = lastNonZeroOffset;
                } else {
                    lastNonZeroOffset = this.index[i];
                }
                newIndex[i] = this.index[i];
            }
            LinearIndex linearIndex = new LinearIndex(CRAMBAIIndexer.this.currentReference, 0, newIndex);
            return new BAMIndexContent(CRAMBAIIndexer.this.currentReference, this.bins, this.binsSeen, this.indexStats, linearIndex);
        }

        private long getNoCoordinateRecordCount() {
            return this.indexStats.getNoCoordinateRecordCount();
        }

        private void startNewReference() {
            this.bins = null;
            if (this.binsSeen > 0) {
                Arrays.fill(this.index, 0L);
            }
            this.binsSeen = 0;
            this.largestIndexSeen = -1;
            this.indexStats.newReference();
        }
    }
}

