/*
 * Decompiled with CFR 0.152.
 */
package picard.analysis;

import htsjdk.samtools.SAMFileReader;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.filter.DuplicateReadFilter;
import htsjdk.samtools.filter.NotPrimaryAlignmentFilter;
import htsjdk.samtools.filter.SamRecordFilter;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.reference.ReferenceSequenceFileWalker;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.ListMap;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.SamLocusIterator;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.Metrics;
import picard.util.DbSnpBitSetUtil;

@CommandLineProgramProperties(usage="Collects metrics quantifying the CpCG -> CpCA error rate from the provided SAM/BAM", usageShort="Collects metrics quantifying the CpCG -> CpCA error rate from the provided SAM/BAM", programGroup=Metrics.class)
public class CollectOxoGMetrics
extends CommandLineProgram {
    static final String USAGE = "Collects metrics quantifying the CpCG -> CpCA error rate from the provided SAM/BAM";
    @Option(shortName="I", doc="Input BAM file for analysis.")
    public File INPUT;
    @Option(shortName="O", doc="Location of output metrics file to write.")
    public File OUTPUT;
    @Option(shortName="R", doc="Reference sequence to which BAM is aligned.")
    public File REFERENCE_SEQUENCE;
    @Option(doc="An optional list of intervals to restrict analysis to.", optional=true)
    public File INTERVALS;
    @Option(doc="VCF format dbSNP file, used to exclude regions around known polymorphisms from analysis.", optional=true)
    public File DB_SNP;
    @Option(shortName="Q", doc="The minimum base quality score for a base to be included in analysis.")
    public int MINIMUM_QUALITY_SCORE = 20;
    @Option(shortName="MQ", doc="The minimum mapping quality score for a base to be included in analysis.")
    public int MINIMUM_MAPPING_QUALITY = 30;
    @Option(shortName="MIN_INS", doc="The minimum insert size for a read to be included in analysis. Set of 0 to allow unpaired reads.")
    public int MINIMUM_INSERT_SIZE = 60;
    @Option(shortName="MAX_INS", doc="The maximum insert size for a read to be included in analysis. Set of 0 to allow unpaired reads.")
    public int MAXIMUM_INSERT_SIZE = 600;
    @Option(doc="When available, use original quality scores for filtering.")
    public boolean USE_OQ = true;
    @Option(doc="The number of context bases to include on each side of the assayed G/C base.")
    public int CONTEXT_SIZE = 1;
    @Option(doc="The optional set of sequence contexts to restrict analysis to. If not supplied all contexts are analyzed.")
    public Set<String> CONTEXTS = new HashSet<String>();
    @Option(doc="For debugging purposes: stop after visiting this many sites with at least 1X coverage.")
    public int STOP_AFTER = Integer.MAX_VALUE;
    private final Log log = Log.getInstance(CollectOxoGMetrics.class);
    private static final String UNKNOWN_LIBRARY = "UnknownLibrary";
    private static final String UNKNOWN_SAMPLE = "UnknownSample";

    public static void main(String[] args) {
        new CollectOxoGMetrics().instanceMainWithExit(args);
    }

    @Override
    protected String[] customCommandLineValidation() {
        int size = 1 + 2 * this.CONTEXT_SIZE;
        ArrayList<String> messages = new ArrayList<String>();
        for (String ctx : this.CONTEXTS) {
            if (ctx.length() != size) {
                messages.add("Context " + ctx + " is not " + size + " long as implied by CONTEXT_SIZE=" + this.CONTEXT_SIZE);
                continue;
            }
            if (ctx.charAt(ctx.length() / 2) == 'C') continue;
            messages.add("Middle base of context sequence " + ctx + " must be C");
        }
        return messages.isEmpty() ? null : messages.toArray(new String[messages.size()]);
    }

    private final <T> T nvl(T value1, T value2) {
        if (value1 != null) {
            return value1;
        }
        return value2;
    }

    @Override
    protected int doWork() {
        SamLocusIterator iterator;
        IOUtil.assertFileIsReadable(this.INPUT);
        IOUtil.assertFileIsWritable(this.OUTPUT);
        if (this.INTERVALS != null) {
            IOUtil.assertFileIsReadable(this.INTERVALS);
        }
        IOUtil.assertFileIsReadable(this.REFERENCE_SEQUENCE);
        ReferenceSequenceFileWalker refWalker = new ReferenceSequenceFileWalker(this.REFERENCE_SEQUENCE);
        SAMFileReader in = new SAMFileReader(this.INPUT);
        HashSet<String> samples = new HashSet<String>();
        HashSet<String> libraries = new HashSet<String>();
        for (SAMReadGroupRecord rec : in.getFileHeader().getReadGroups()) {
            samples.add(this.nvl(rec.getSample(), UNKNOWN_SAMPLE));
            libraries.add(this.nvl(rec.getLibrary(), UNKNOWN_LIBRARY));
        }
        Set<String> contexts = this.CONTEXTS.isEmpty() ? this.makeContextStrings(this.CONTEXT_SIZE) : this.CONTEXTS;
        ListMap<String, Calculator> calculators = new ListMap<String, Calculator>();
        for (String context : contexts) {
            for (String library : libraries) {
                calculators.add(context, new Calculator(library, context));
            }
        }
        this.log.info("Loading dbSNP File: " + this.DB_SNP);
        DbSnpBitSetUtil dbSnp = this.DB_SNP != null ? new DbSnpBitSetUtil(this.DB_SNP, in.getFileHeader().getSequenceDictionary()) : null;
        if (this.INTERVALS != null) {
            IntervalList intervals = IntervalList.fromFile(this.INTERVALS);
            intervals.unique();
            iterator = new SamLocusIterator(in, intervals, false);
        } else {
            iterator = new SamLocusIterator(in);
        }
        iterator.setEmitUncoveredLoci(false);
        iterator.setMappingQualityScoreCutoff(this.MINIMUM_MAPPING_QUALITY);
        iterator.setSamFilters(Arrays.asList(new NotPrimaryAlignmentFilter(), new DuplicateReadFilter(), new InsertSizeFilter(this.MINIMUM_INSERT_SIZE, this.MAXIMUM_INSERT_SIZE)));
        this.log.info("Starting iteration.");
        long nextLogTime = 0L;
        int sites = 0;
        for (SamLocusIterator.LocusInfo info : iterator) {
            long now;
            byte base;
            String chrom = info.getSequenceName();
            int pos = info.getPosition();
            int index = pos - 1;
            if (dbSnp != null && dbSnp.isDbSnpSite(chrom, pos)) continue;
            byte[] bases = refWalker.get(info.getSequenceIndex()).getBases();
            if (pos < 3 || pos > bases.length - 3 || (base = StringUtil.toUpperCase(bases[index])) != 67 && base != 71) continue;
            String tmp = StringUtil.bytesToString(bases, index - this.CONTEXT_SIZE, 1 + 2 * this.CONTEXT_SIZE).toUpperCase();
            String context = base == 67 ? tmp : SequenceUtil.reverseComplement(tmp);
            List calculatorsForContext = (List)calculators.get(context);
            if (calculatorsForContext == null) continue;
            for (Calculator calc : calculatorsForContext) {
                calc.accept(info, base);
            }
            if (++sites % 100 == 0 && (now = System.currentTimeMillis()) > nextLogTime) {
                this.log.info("Visited " + sites + " sites of interest. Last site: " + chrom + ":" + pos);
                nextLogTime = now + 60000L;
            }
            if (sites < this.STOP_AFTER) continue;
            break;
        }
        MetricsFile file = this.getMetricsFile();
        for (List calcs : calculators.values()) {
            for (Calculator calc : calcs) {
                CpcgMetrics m = calc.finish();
                m.SAMPLE_ALIAS = StringUtil.join(",", new ArrayList(samples));
                file.addMetric(m);
            }
        }
        file.write(this.OUTPUT);
        return 0;
    }

    private Set<String> makeContextStrings(int contextSize) {
        HashSet<String> contexts = new HashSet<String>();
        for (byte[] kmer : this.generateAllKmers(2 * contextSize + 1)) {
            if (kmer[contextSize] != 67) continue;
            contexts.add(StringUtil.bytesToString(kmer));
        }
        this.log.info("Generated " + contexts.size() + " context strings.");
        return contexts;
    }

    private List<byte[]> generateAllKmers(int length) {
        LinkedList<byte[]> sofar = new LinkedList<byte[]>();
        byte[] bases = new byte[]{65, 67, 71, 84};
        if (sofar.size() == 0) {
            sofar.add(new byte[length]);
        }
        block0: while (true) {
            byte[] bs;
            int indexOfNextBase;
            if ((indexOfNextBase = this.findIndexOfNextBase(bs = (byte[])sofar.remove(0))) == -1) {
                sofar.add(bs);
                break;
            }
            byte[] arr$ = bases;
            int len$ = arr$.length;
            int i$ = 0;
            while (true) {
                if (i$ >= len$) continue block0;
                byte b = arr$[i$];
                byte[] next = Arrays.copyOf(bs, bs.length);
                next[indexOfNextBase] = b;
                sofar.add(next);
                ++i$;
            }
            break;
        }
        return sofar;
    }

    private int findIndexOfNextBase(byte[] bs) {
        for (int i = 0; i < bs.length; ++i) {
            if (bs[i] != 0) continue;
            return i;
        }
        return -1;
    }

    private class Calculator {
        private final String library;
        private final String context;
        int sites = 0;
        long refCcontrolA = 0L;
        long refCoxidatedA = 0L;
        long refCcontrolC = 0L;
        long refCoxidatedC = 0L;
        long refGcontrolA = 0L;
        long refGoxidatedA = 0L;
        long refGcontrolC = 0L;
        long refGoxidatedC = 0L;

        Calculator(String library, String context) {
            this.library = library;
            this.context = context;
        }

        void accept(SamLocusIterator.LocusInfo info, byte refBase) {
            Counts counts = this.computeAlleleFraction(info, refBase);
            if (counts.total() > 0) {
                ++this.sites;
                if (refBase == 67) {
                    this.refCcontrolA += (long)counts.controlA;
                    this.refCoxidatedA += (long)counts.oxidatedA;
                    this.refCcontrolC += (long)counts.controlC;
                    this.refCoxidatedC += (long)counts.oxidatedC;
                } else if (refBase == 71) {
                    this.refGcontrolA += (long)counts.controlA;
                    this.refGoxidatedA += (long)counts.oxidatedA;
                    this.refGcontrolC += (long)counts.controlC;
                    this.refGoxidatedC += (long)counts.oxidatedC;
                } else {
                    throw new IllegalStateException("Reference bases other than G and C not supported.");
                }
            }
        }

        CpcgMetrics finish() {
            CpcgMetrics m = new CpcgMetrics();
            m.LIBRARY = this.library;
            m.CONTEXT = this.context;
            m.TOTAL_SITES = this.sites;
            m.TOTAL_BASES = this.refCcontrolC + this.refCoxidatedC + this.refCcontrolA + this.refCoxidatedA + this.refGcontrolC + this.refGoxidatedC + this.refGcontrolA + this.refGoxidatedA;
            m.REF_OXO_BASES = this.refCoxidatedC + this.refGoxidatedC;
            m.REF_NONOXO_BASES = this.refCcontrolC + this.refGcontrolC;
            m.REF_TOTAL_BASES = m.REF_OXO_BASES + m.REF_NONOXO_BASES;
            m.ALT_NONOXO_BASES = this.refCcontrolA + this.refGcontrolA;
            m.ALT_OXO_BASES = this.refCoxidatedA + this.refGoxidatedA;
            m.OXIDATION_ERROR_RATE = (double)Math.max(m.ALT_OXO_BASES - m.ALT_NONOXO_BASES, 1L) / (double)m.TOTAL_BASES;
            m.OXIDATION_Q = -10.0 * Math.log10(m.OXIDATION_ERROR_RATE);
            m.C_REF_REF_BASES = this.refCcontrolC + this.refCoxidatedC;
            m.G_REF_REF_BASES = this.refGcontrolC + this.refGoxidatedC;
            m.C_REF_ALT_BASES = this.refCcontrolA + this.refCoxidatedA;
            m.G_REF_ALT_BASES = this.refGcontrolA + this.refGoxidatedA;
            double cRefErrorRate = (double)m.C_REF_ALT_BASES / (double)(m.C_REF_ALT_BASES + m.C_REF_REF_BASES);
            double gRefErrorRate = (double)m.G_REF_ALT_BASES / (double)(m.G_REF_ALT_BASES + m.G_REF_REF_BASES);
            m.C_REF_OXO_ERROR_RATE = Math.max(cRefErrorRate - gRefErrorRate, 1.0E-10);
            m.G_REF_OXO_ERROR_RATE = Math.max(gRefErrorRate - cRefErrorRate, 1.0E-10);
            m.C_REF_OXO_Q = -10.0 * Math.log10(m.C_REF_OXO_ERROR_RATE);
            m.G_REF_OXO_Q = -10.0 * Math.log10(m.G_REF_OXO_ERROR_RATE);
            return m;
        }

        private Counts computeAlleleFraction(SamLocusIterator.LocusInfo info, byte refBase) {
            Counts counts = new Counts();
            byte altBase = refBase == 67 ? (byte)65 : 84;
            for (SamLocusIterator.RecordAndOffset rec : info.getRecordAndPositions()) {
                int read;
                byte[] oqs;
                SAMRecord samrec = rec.getRecord();
                byte qual = CollectOxoGMetrics.this.USE_OQ ? ((oqs = samrec.getOriginalBaseQualities()) != null ? oqs[rec.getOffset()] : rec.getBaseQuality()) : rec.getBaseQuality();
                if (qual < CollectOxoGMetrics.this.MINIMUM_QUALITY_SCORE || !this.library.equals(CollectOxoGMetrics.this.nvl(samrec.getReadGroup().getLibrary(), CollectOxoGMetrics.UNKNOWN_LIBRARY))) continue;
                byte base = rec.getReadBase();
                byte baseAsRead = samrec.getReadNegativeStrandFlag() ? SequenceUtil.complement(base) : base;
                int n = read = samrec.getReadPairedFlag() && samrec.getSecondOfPairFlag() ? 2 : 1;
                if (base == refBase) {
                    if (baseAsRead == 71 && read == 1) {
                        ++counts.oxidatedC;
                        continue;
                    }
                    if (baseAsRead == 71 && read == 2) {
                        ++counts.controlC;
                        continue;
                    }
                    if (baseAsRead == 67 && read == 1) {
                        ++counts.controlC;
                        continue;
                    }
                    if (baseAsRead != 67 || read != 2) continue;
                    ++counts.oxidatedC;
                    continue;
                }
                if (base != altBase) continue;
                if (baseAsRead == 84 && read == 1) {
                    ++counts.oxidatedA;
                    continue;
                }
                if (baseAsRead == 84 && read == 2) {
                    ++counts.controlA;
                    continue;
                }
                if (baseAsRead == 65 && read == 1) {
                    ++counts.controlA;
                    continue;
                }
                if (baseAsRead != 65 || read != 2) continue;
                ++counts.oxidatedA;
            }
            return counts;
        }
    }

    private static class Counts {
        int controlA;
        int oxidatedA;
        int controlC;
        int oxidatedC;

        private Counts() {
        }

        int total() {
            return this.controlC + this.oxidatedC + this.controlA + this.oxidatedA;
        }
    }

    static class InsertSizeFilter
    implements SamRecordFilter {
        final int minInsertSize;
        final int maxInsertSize;

        InsertSizeFilter(int minInsertSize, int maxInsertSize) {
            this.minInsertSize = minInsertSize;
            this.maxInsertSize = maxInsertSize;
        }

        @Override
        public boolean filterOut(SAMRecord rec) {
            if (this.minInsertSize == 0 && this.maxInsertSize == 0) {
                return false;
            }
            if (rec.getReadPairedFlag()) {
                int ins = Math.abs(rec.getInferredInsertSize());
                return ins < this.minInsertSize || ins > this.maxInsertSize;
            }
            return this.minInsertSize != 0 || this.maxInsertSize != 0;
        }

        @Override
        public boolean filterOut(SAMRecord r1, SAMRecord r2) {
            return this.filterOut(r1) || this.filterOut(r2);
        }
    }

    public static final class CpcgMetrics
    extends MetricBase {
        public String SAMPLE_ALIAS;
        public String LIBRARY;
        public String CONTEXT;
        public int TOTAL_SITES;
        public long TOTAL_BASES;
        public long REF_NONOXO_BASES;
        public long REF_OXO_BASES;
        public long REF_TOTAL_BASES;
        public long ALT_NONOXO_BASES;
        public long ALT_OXO_BASES;
        public double OXIDATION_ERROR_RATE;
        public double OXIDATION_Q;
        public long C_REF_REF_BASES;
        public long G_REF_REF_BASES;
        public long C_REF_ALT_BASES;
        public long G_REF_ALT_BASES;
        public double C_REF_OXO_ERROR_RATE;
        public double C_REF_OXO_Q;
        public double G_REF_OXO_ERROR_RATE;
        public double G_REF_OXO_Q;
    }
}

