/*
 * Decompiled with CFR 0.152.
 */
package picard.vcf;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.util.BlockCompressedInputStream;
import htsjdk.samtools.util.BlockCompressedOutputStream;
import htsjdk.samtools.util.BlockCompressedStreamConstants;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.PeekableIterator;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.RuntimeIOException;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextComparator;
import htsjdk.variant.variantcontext.writer.Options;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterFactory;
import htsjdk.variant.vcf.VCFContigHeaderLine;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFHeader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.VcfOrBcf;

@CommandLineProgramProperties(usage="Gathers multiple VCF files from a scatter operation into a single VCF file. Input files must be supplied in genomic order and must not have events at overlapping positions.", usageShort="Gathers multiple VCF files from a scatter operation into a single VCF file", programGroup=VcfOrBcf.class)
public class GatherVcfs
extends CommandLineProgram {
    @Option(shortName="I", doc="Input VCF file(s).")
    public List<File> INPUT;
    @Option(shortName="O", doc="Output VCF file.")
    public File OUTPUT;
    private static final Log log = Log.getInstance(GatherVcfs.class);

    public static void main(String[] args) {
        new GatherVcfs().instanceMainWithExit(args);
    }

    public GatherVcfs() {
        this.CREATE_INDEX = true;
    }

    @Override
    protected int doWork() {
        log.info("Checking inputs.");
        this.INPUT = IOUtil.unrollFiles(this.INPUT, IOUtil.VCF_EXTENSIONS);
        for (File f2 : this.INPUT) {
            IOUtil.assertFileIsReadable(f2);
        }
        IOUtil.assertFileIsWritable(this.OUTPUT);
        SAMSequenceDictionary sequenceDictionary = VCFFileReader.getSequenceDictionary(this.INPUT.get(0));
        if (this.CREATE_INDEX.booleanValue() && sequenceDictionary == null) {
            throw new PicardException("In order to index the resulting VCF input VCFs must contain ##contig lines.");
        }
        log.info("Checking file headers and first records to ensure compatibility.");
        GatherVcfs.assertSameSamplesAndValidOrdering(this.INPUT);
        if (this.areAllBlockCompressed(this.INPUT) && this.areAllBlockCompressed(CollectionUtil.makeList(this.OUTPUT))) {
            log.info("Gathering by copying gzip blocks. Will not be able to validate position non-overlap of files.");
            if (this.CREATE_INDEX.booleanValue()) {
                log.warn("Index creation not currently supported when gathering block compressed VCFs.");
            }
            GatherVcfs.gatherWithBlockCopying(this.INPUT, this.OUTPUT);
        } else {
            log.info("Gathering by conventional means.");
            GatherVcfs.gatherConventionally(sequenceDictionary, this.CREATE_INDEX, this.INPUT, this.OUTPUT);
        }
        return 0;
    }

    private boolean areAllBlockCompressed(List<File> input) {
        for (File f2 : input) {
            if (!VariantContextWriterFactory.isBCFOutput((File)f2) && VariantContextWriterFactory.isCompressedVcf((File)f2)) continue;
            return false;
        }
        return true;
    }

    private static void assertSameSamplesAndValidOrdering(List<File> inputFiles) {
        VCFHeader header = new VCFFileReader(inputFiles.get(0), false).getFileHeader();
        SAMSequenceDictionary dict = header.getSequenceDictionary();
        VariantContextComparator comparator = new VariantContextComparator(header.getSequenceDictionary());
        List<String> samples = header.getGenotypeSamples();
        File lastFile = null;
        VariantContext lastContext = null;
        for (File f2 : inputFiles) {
            VCFFileReader in = new VCFFileReader(f2, false);
            dict.assertSameDictionary(in.getFileHeader().getSequenceDictionary());
            List<String> theseSamples = in.getFileHeader().getGenotypeSamples();
            if (!((Object)samples).equals(theseSamples)) {
                TreeSet<String> s1 = new TreeSet<String>(samples);
                TreeSet<String> s2 = new TreeSet<String>(theseSamples);
                s1.removeAll(theseSamples);
                s2.removeAll(samples);
                throw new IllegalArgumentException("VCFs do not have identical sample lists. Samples unique to first file: " + s1 + ". Samples unique to " + f2.getAbsolutePath() + ": " + s2 + ".");
            }
            Iterator variantIterator = in.iterator();
            if (variantIterator.hasNext()) {
                VariantContext currentContext = (VariantContext)variantIterator.next();
                if (lastContext != null && comparator.compare(lastContext, currentContext) >= 0) {
                    throw new IllegalArgumentException("First record in file " + f2.getAbsolutePath() + " is not after first record in " + "previous file " + lastFile.getAbsolutePath());
                }
                lastContext = currentContext;
                lastFile = f2;
            }
            CloserUtil.close(in);
        }
    }

    private static void gatherConventionally(SAMSequenceDictionary sequenceDictionary, boolean createIndex, List<File> inputFiles, File outputFile) {
        EnumSet<Options> options = EnumSet.copyOf(VariantContextWriterFactory.DEFAULT_OPTIONS);
        if (createIndex) {
            options.add(Options.INDEX_ON_THE_FLY);
        } else {
            options.remove((Object)Options.INDEX_ON_THE_FLY);
        }
        VariantContextWriter out = VariantContextWriterFactory.create((File)outputFile, (SAMSequenceDictionary)sequenceDictionary, options);
        ProgressLogger progress = new ProgressLogger(log, 10000);
        VariantContext lastContext = null;
        File lastFile = null;
        VCFHeader firstHeader = null;
        VariantContextComparator comparator = null;
        for (File f2 : inputFiles) {
            VariantContext vc;
            log.debug("Gathering from file: ", f2.getAbsolutePath());
            VCFFileReader variantReader = new VCFFileReader(f2, false);
            PeekableIterator variantIterator = new PeekableIterator(variantReader.iterator());
            VCFHeader header = variantReader.getFileHeader();
            if (firstHeader == null) {
                firstHeader = header;
                out.writeHeader(firstHeader);
                comparator = new VariantContextComparator((Collection<VCFContigHeaderLine>)firstHeader.getContigLines());
            }
            if (lastContext != null && variantIterator.hasNext() && comparator.compare(vc = (VariantContext)variantIterator.peek(), lastContext) <= 0) {
                throw new IllegalStateException("First variant in file " + f2.getAbsolutePath() + " is at " + vc.getSource() + " but last variant in earlier file " + lastFile.getAbsolutePath() + " is at " + lastContext.getSource());
            }
            while (variantIterator.hasNext()) {
                lastContext = (VariantContext)variantIterator.next();
                out.add(lastContext);
                progress.record(lastContext.getChr(), lastContext.getStart());
            }
            lastFile = f2;
            CloserUtil.close(variantIterator);
            CloserUtil.close(variantReader);
        }
        out.close();
    }

    private static void gatherWithBlockCopying(List<File> vcfs, File output) {
        try {
            FileOutputStream out = new FileOutputStream(output);
            boolean isFirstFile = true;
            for (File f2 : vcfs) {
                log.info("Gathering " + f2.getAbsolutePath());
                FileInputStream in = new FileInputStream(f2);
                BlockCompressedInputStream.FileTermination term = BlockCompressedInputStream.checkTermination(f2);
                if (term == BlockCompressedInputStream.FileTermination.DEFECTIVE) {
                    throw new PicardException(f2.getAbsolutePath() + " does not have a valid GZIP block at the end of the file.");
                }
                if (!isFirstFile) {
                    BlockCompressedInputStream blockIn = new BlockCompressedInputStream((InputStream)in, false);
                    boolean lastByteNewline = true;
                    while (in.available() > 0) {
                        int blockLength = blockIn.available();
                        byte[] blockContents = new byte[blockLength];
                        int read = blockIn.read(blockContents);
                        if (blockLength == 0 || read != blockLength) {
                            throw new IllegalStateException("Could not read available bytes from BlockCompressedInputStream.");
                        }
                        int firstNonHeaderByteIndex = -1;
                        for (int i2 = 0; i2 < read; ++i2) {
                            boolean thisByteNewline;
                            byte b2 = blockContents[i2];
                            boolean bl = thisByteNewline = b2 == 10 || b2 == 13;
                            if (lastByteNewline && !thisByteNewline && b2 != 35) {
                                firstNonHeaderByteIndex = i2;
                                break;
                            }
                            lastByteNewline = thisByteNewline;
                        }
                        if (firstNonHeaderByteIndex < 0) continue;
                        BlockCompressedOutputStream blockOut = new BlockCompressedOutputStream(out, null);
                        blockOut.write(blockContents, firstNonHeaderByteIndex, blockContents.length - firstNonHeaderByteIndex);
                        blockOut.flush();
                        break;
                    }
                }
                long currentPos = in.getChannel().position();
                long length = f2.length();
                long skipLast = term == BlockCompressedInputStream.FileTermination.HAS_TERMINATOR_BLOCK ? (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length : 0L;
                long bytesToWrite = length - skipLast - currentPos;
                IOUtil.transferByStream(in, out, bytesToWrite);
                in.close();
                isFirstFile = false;
            }
            out.write(BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK);
            out.close();
        }
        catch (IOException ioe) {
            throw new RuntimeIOException(ioe);
        }
    }
}

