/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.utils.variant;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import htsjdk.tribble.TribbleException;
import htsjdk.tribble.util.popgen.HardyWeinbergCalculation;
import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.CommonInfo;
import htsjdk.variant.variantcontext.Genotype;
import htsjdk.variant.variantcontext.GenotypeBuilder;
import htsjdk.variant.variantcontext.GenotypeLikelihoods;
import htsjdk.variant.variantcontext.GenotypesContext;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextBuilder;
import htsjdk.variant.variantcontext.VariantContextUtils;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.lang.ArrayUtils;
import org.apache.log4j.Logger;
import org.broadinstitute.gatk.utils.BaseUtils;
import org.broadinstitute.gatk.utils.GenomeLoc;
import org.broadinstitute.gatk.utils.GenomeLocParser;
import org.broadinstitute.gatk.utils.MathUtils;
import org.broadinstitute.gatk.utils.Utils;
import org.broadinstitute.gatk.utils.collections.Pair;
import org.broadinstitute.gatk.utils.exceptions.UserException;

public class GATKVariantContextUtils {
    private static Logger logger = Logger.getLogger(GATKVariantContextUtils.class);
    public static final int DEFAULT_PLOIDY = 2;
    public static final double SUM_GL_THRESH_NOCALL = -0.1;
    public static final List<Allele> NO_CALL_ALLELES = Arrays.asList(Allele.NO_CALL, Allele.NO_CALL);
    public static final String NON_REF_SYMBOLIC_ALLELE_NAME = "NON_REF";
    public static final Allele NON_REF_SYMBOLIC_ALLELE = Allele.create("<NON_REF>", false);
    public static final String MERGE_FILTER_PREFIX = "filterIn";
    public static final String MERGE_REF_IN_ALL = "ReferenceInAll";
    public static final String MERGE_FILTER_IN_ALL = "FilteredInAll";
    public static final String MERGE_INTERSECTION = "Intersection";

    public static boolean overlapsRegion(VariantContext variantContext, GenomeLoc region) {
        if (region == null) {
            throw new IllegalArgumentException("the active region provided cannot be null");
        }
        if (variantContext == null) {
            throw new IllegalArgumentException("the variant context provided cannot be null");
        }
        if (region.isUnmapped()) {
            return false;
        }
        if (variantContext.getEnd() < region.getStart()) {
            return false;
        }
        if (variantContext.getStart() > region.getStop()) {
            return false;
        }
        return variantContext.getChr().equals(region.getContig());
    }

    public static double getMeanAltAlleleLength(VariantContext vc) {
        double averageLength = 1.0;
        if (!vc.isSNP() && !vc.isSymbolic()) {
            int averageLengthNum = 0;
            int averageLengthDenom = 0;
            int refLength = vc.getReference().length();
            for (Allele a2 : vc.getAlternateAlleles()) {
                int alleleSize;
                int numAllele = vc.getCalledChrCount(a2);
                if (a2.length() == refLength) {
                    byte[] a_bases = a2.getBases();
                    byte[] ref_bases = vc.getReference().getBases();
                    int n_mismatch = 0;
                    for (int idx = 0; idx < a_bases.length; ++idx) {
                        if (a_bases[idx] == ref_bases[idx]) continue;
                        ++n_mismatch;
                    }
                    alleleSize = n_mismatch;
                } else {
                    alleleSize = a2.isSymbolic() ? 1 : Math.abs(refLength - a2.length());
                }
                averageLengthNum += alleleSize * numAllele;
                averageLengthDenom += numAllele;
            }
            averageLength = (double)averageLengthNum / (double)averageLengthDenom;
        }
        return averageLength;
    }

    public static final GenomeLoc getLocation(GenomeLocParser genomeLocParser, VariantContext vc) {
        return genomeLocParser.createGenomeLoc(vc.getChr(), vc.getStart(), vc.getEnd(), true);
    }

    public static BaseUtils.BaseSubstitutionType getSNPSubstitutionType(VariantContext context) {
        if (!context.isSNP() || !context.isBiallelic()) {
            throw new IllegalStateException("Requested SNP substitution type for bialleic non-SNP " + context);
        }
        return BaseUtils.SNPSubstitutionType(context.getReference().getBases()[0], context.getAlternateAllele(0).getBases()[0]);
    }

    public static boolean isTransition(VariantContext context) {
        return GATKVariantContextUtils.getSNPSubstitutionType(context) == BaseUtils.BaseSubstitutionType.TRANSITION;
    }

    public static boolean isTransversion(VariantContext context) {
        return GATKVariantContextUtils.getSNPSubstitutionType(context) == BaseUtils.BaseSubstitutionType.TRANSVERSION;
    }

    public static boolean isTransition(Allele ref, Allele alt) {
        return BaseUtils.SNPSubstitutionType(ref.getBases()[0], alt.getBases()[0]) == BaseUtils.BaseSubstitutionType.TRANSITION;
    }

    public static boolean isTransversion(Allele ref, Allele alt) {
        return BaseUtils.SNPSubstitutionType(ref.getBases()[0], alt.getBases()[0]) == BaseUtils.BaseSubstitutionType.TRANSVERSION;
    }

    public static VariantContext reverseComplement(VariantContext vc) {
        HashMap<Allele, Allele> alleleMap = new HashMap<Allele, Allele>(vc.getAlleles().size());
        for (Allele originalAllele : vc.getAlleles()) {
            Allele newAllele = originalAllele.isNoCall() ? originalAllele : Allele.create(BaseUtils.simpleReverseComplement(originalAllele.getBases()), originalAllele.isReference());
            alleleMap.put(originalAllele, newAllele);
        }
        GenotypesContext newGenotypes = GenotypesContext.create(vc.getNSamples());
        for (Genotype genotype : vc.getGenotypes()) {
            ArrayList<Allele> newAlleles = new ArrayList<Allele>();
            for (Allele allele : genotype.getAlleles()) {
                Allele newAllele = (Allele)alleleMap.get(allele);
                if (newAllele == null) {
                    newAllele = Allele.NO_CALL;
                }
                newAlleles.add(newAllele);
            }
            newGenotypes.add(new GenotypeBuilder(genotype).alleles(newAlleles).make());
        }
        return new VariantContextBuilder(vc).alleles(alleleMap.values()).genotypes(newGenotypes).make();
    }

    @Requires(value={"vc != null", "refBasesStartingAtVCWithPad != null && refBasesStartingAtVCWithPad.length > 0"})
    public static boolean isTandemRepeat(VariantContext vc, byte[] refBasesStartingAtVCWithPad) {
        String refBasesStartingAtVCWithoutPad = new String(refBasesStartingAtVCWithPad).substring(1);
        if (!vc.isIndel()) {
            return false;
        }
        Allele ref = vc.getReference();
        for (Allele allele : vc.getAlternateAlleles()) {
            if (GATKVariantContextUtils.isRepeatAllele(ref, allele, refBasesStartingAtVCWithoutPad)) continue;
            return false;
        }
        return true;
    }

    @Requires(value={"vc != null", "refBasesStartingAtVCWithPad != null && refBasesStartingAtVCWithPad.length > 0"})
    public static Pair<List<Integer>, byte[]> getNumTandemRepeatUnits(VariantContext vc, byte[] refBasesStartingAtVCWithPad) {
        boolean VERBOSE = false;
        String refBasesStartingAtVCWithoutPad = new String(refBasesStartingAtVCWithPad).substring(1);
        if (!vc.isIndel()) {
            return null;
        }
        Allele refAllele = vc.getReference();
        byte[] refAlleleBases = Arrays.copyOfRange(refAllele.getBases(), 1, refAllele.length());
        byte[] repeatUnit = null;
        ArrayList<Integer> lengths = new ArrayList<Integer>();
        for (Allele allele : vc.getAlternateAlleles()) {
            Pair<int[], byte[]> result = GATKVariantContextUtils.getNumTandemRepeatUnits(refAlleleBases, Arrays.copyOfRange(allele.getBases(), 1, allele.length()), refBasesStartingAtVCWithoutPad.getBytes());
            int[] repetitionCount = (int[])result.first;
            if (repetitionCount[0] == 0 || repetitionCount[1] == 0) {
                return null;
            }
            if (lengths.size() == 0) {
                lengths.add(repetitionCount[0]);
            }
            lengths.add(repetitionCount[1]);
            repeatUnit = (byte[])result.second;
        }
        return new Pair(lengths, repeatUnit);
    }

    public static Pair<int[], byte[]> getNumTandemRepeatUnits(byte[] refBases, byte[] altBases, byte[] remainingRefContext) {
        byte[] longB = altBases.length > refBases.length ? altBases : refBases;
        int repeatUnitLength = GATKVariantContextUtils.findRepeatedSubstring(longB);
        byte[] repeatUnit = Arrays.copyOf(longB, repeatUnitLength);
        int[] repetitionCount = new int[2];
        int repetitionsInRef = GATKVariantContextUtils.findNumberofRepetitions(repeatUnit, refBases, true);
        repetitionCount[0] = GATKVariantContextUtils.findNumberofRepetitions(repeatUnit, ArrayUtils.addAll(refBases, remainingRefContext), true) - repetitionsInRef;
        repetitionCount[1] = GATKVariantContextUtils.findNumberofRepetitions(repeatUnit, ArrayUtils.addAll(altBases, remainingRefContext), true) - repetitionsInRef;
        return new Pair<int[], byte[]>(repetitionCount, repeatUnit);
    }

    public static int findRepeatedSubstring(byte[] bases) {
        int repLength;
        for (repLength = 1; repLength <= bases.length; ++repLength) {
            byte[] candidateRepeatUnit = Arrays.copyOf(bases, repLength);
            boolean allBasesMatch = true;
            for (int start = repLength; start < bases.length; start += repLength) {
                byte[] basePiece = Arrays.copyOfRange(bases, start, start + candidateRepeatUnit.length);
                if (Arrays.equals(candidateRepeatUnit, basePiece)) continue;
                allBasesMatch = false;
                break;
            }
            if (!allBasesMatch) continue;
            return repLength;
        }
        return repLength;
    }

    public static int findNumberofRepetitions(byte[] repeatUnit, byte[] testString, boolean lookForward) {
        int end;
        byte[] unit;
        int numRepeats = 0;
        if (lookForward) {
            int end2;
            byte[] unit2;
            for (int start = 0; start < testString.length && Arrays.equals(unit2 = Arrays.copyOfRange(testString, start, end2 = start + repeatUnit.length), repeatUnit); start += repeatUnit.length) {
                ++numRepeats;
            }
            return numRepeats;
        }
        for (int start = testString.length - repeatUnit.length; start >= 0 && Arrays.equals(unit = Arrays.copyOfRange(testString, start, end = start + repeatUnit.length), repeatUnit); start -= repeatUnit.length) {
            ++numRepeats;
        }
        return numRepeats;
    }

    protected static boolean isRepeatAllele(Allele ref, Allele alt, String refBasesStartingAtVCWithoutPad) {
        if (!Allele.oneIsPrefixOfOther(ref, alt)) {
            return false;
        }
        if (ref.length() > alt.length()) {
            return GATKVariantContextUtils.basesAreRepeated(ref.getBaseString(), alt.getBaseString(), refBasesStartingAtVCWithoutPad, 2);
        }
        return GATKVariantContextUtils.basesAreRepeated(alt.getBaseString(), ref.getBaseString(), refBasesStartingAtVCWithoutPad, 1);
    }

    protected static boolean basesAreRepeated(String l2, String s2, String ref, int minNumberOfMatches) {
        String potentialRepeat = l2.substring(s2.length());
        for (int i2 = 0; i2 < minNumberOfMatches; ++i2) {
            int start = i2 * potentialRepeat.length();
            int end = (i2 + 1) * potentialRepeat.length();
            if (ref.length() < end) {
                return false;
            }
            String refSub = ref.substring(start, end);
            if (refSub.equals(potentialRepeat)) continue;
            return false;
        }
        return true;
    }

    public static GenotypesContext subsetDiploidAlleles(VariantContext vc, List<Allele> allelesToUse, GenotypeAssignmentMethod assignGenotypes) {
        if (allelesToUse.get(0).isNonReference()) {
            throw new IllegalArgumentException("First allele must be the reference allele");
        }
        if (allelesToUse.size() == 1) {
            throw new IllegalArgumentException("Cannot subset to only 1 alt allele");
        }
        if (vc.getGenotypes().isEmpty()) {
            return GenotypesContext.create();
        }
        List<Integer> likelihoodIndexesToUse = GATKVariantContextUtils.determineLikelihoodIndexesToUse(vc, allelesToUse);
        return GATKVariantContextUtils.createGenotypesWithSubsettedLikelihoods(vc.getGenotypes(), vc, allelesToUse, likelihoodIndexesToUse, assignGenotypes);
    }

    private static List<Integer> determineLikelihoodIndexesToUse(VariantContext originalVC, List<Allele> allelesToUse) {
        boolean[] alleleIndexesToUse = GATKVariantContextUtils.getAlleleIndexBitset(originalVC, allelesToUse);
        if (MathUtils.countOccurrences(true, alleleIndexesToUse) == alleleIndexesToUse.length) {
            return null;
        }
        return GATKVariantContextUtils.getLikelihoodIndexes(originalVC, alleleIndexesToUse);
    }

    private static List<Integer> getLikelihoodIndexes(VariantContext originalVC, boolean[] alleleIndexesToUse) {
        ArrayList<Integer> result = new ArrayList<Integer>(30);
        int numLikelihoods = GenotypeLikelihoods.numLikelihoods(originalVC.getNAlleles(), 2);
        for (int PLindex = 0; PLindex < numLikelihoods; ++PLindex) {
            GenotypeLikelihoods.GenotypeLikelihoodsAllelePair alleles = GenotypeLikelihoods.getAllelePair(PLindex);
            if (!alleleIndexesToUse[alleles.alleleIndex1] || !alleleIndexesToUse[alleles.alleleIndex2]) continue;
            result.add(PLindex);
        }
        return result;
    }

    private static boolean[] getAlleleIndexBitset(VariantContext originalVC, List<Allele> allelesToKeep) {
        int numOriginalAltAlleles = originalVC.getNAlleles() - 1;
        boolean[] alleleIndexesToKeep = new boolean[numOriginalAltAlleles + 1];
        alleleIndexesToKeep[0] = true;
        for (int i2 = 0; i2 < numOriginalAltAlleles; ++i2) {
            if (!allelesToKeep.contains(originalVC.getAlternateAllele(i2))) continue;
            alleleIndexesToKeep[i2 + 1] = true;
        }
        return alleleIndexesToKeep;
    }

    private static GenotypesContext createGenotypesWithSubsettedLikelihoods(GenotypesContext originalGs, VariantContext vc, List<Allele> allelesToUse, List<Integer> likelihoodIndexesToUse, GenotypeAssignmentMethod assignGenotypes) {
        GenotypesContext newGTs = GenotypesContext.create(originalGs.size());
        int expectedNumLikelihoods = GenotypeLikelihoods.numLikelihoods(vc.getNAlleles(), 2);
        List<String> sampleIndices = originalGs.getSampleNamesOrderedByName();
        for (int k2 = 0; k2 < originalGs.size(); ++k2) {
            double[] newLikelihoods;
            Genotype g2 = originalGs.get(sampleIndices.get(k2));
            GenotypeBuilder gb = new GenotypeBuilder(g2);
            if (!g2.hasLikelihoods()) {
                newLikelihoods = null;
                gb.noPL();
            } else {
                double[] originalLikelihoods = g2.getLikelihoods().getAsVector();
                if (likelihoodIndexesToUse == null) {
                    newLikelihoods = originalLikelihoods;
                } else if (originalLikelihoods.length != expectedNumLikelihoods) {
                    logger.warn("Wrong number of likelihoods in sample " + g2.getSampleName() + " at " + vc + " got " + g2.getLikelihoodsString() + " but expected " + expectedNumLikelihoods);
                    newLikelihoods = null;
                } else {
                    newLikelihoods = new double[likelihoodIndexesToUse.size()];
                    int newIndex = 0;
                    for (int oldIndex : likelihoodIndexesToUse) {
                        newLikelihoods[newIndex++] = originalLikelihoods[oldIndex];
                    }
                    newLikelihoods = MathUtils.normalizeFromLog10(newLikelihoods, false, true);
                }
                if (newLikelihoods == null || GATKVariantContextUtils.likelihoodsAreUninformative(newLikelihoods)) {
                    gb.noPL();
                } else {
                    gb.PL(newLikelihoods);
                }
            }
            GATKVariantContextUtils.updateGenotypeAfterSubsetting(g2.getAlleles(), gb, assignGenotypes, newLikelihoods, allelesToUse);
            newGTs.add(gb.make());
        }
        return GATKVariantContextUtils.fixADFromSubsettedAlleles(newGTs, vc, allelesToUse);
    }

    private static boolean likelihoodsAreUninformative(double[] likelihoods) {
        return MathUtils.sum(likelihoods) > -0.1;
    }

    public static void updateGenotypeAfterSubsetting(List<Allele> originalGT, GenotypeBuilder gb, GenotypeAssignmentMethod assignmentMethod, double[] newLikelihoods, List<Allele> allelesToUse) {
        switch (assignmentMethod) {
            case DO_NOT_ASSIGN_GENOTYPES: {
                break;
            }
            case SET_TO_NO_CALL: {
                gb.alleles(NO_CALL_ALLELES);
                gb.noGQ();
                break;
            }
            case USE_PLS_TO_ASSIGN: {
                if (newLikelihoods == null || GATKVariantContextUtils.likelihoodsAreUninformative(newLikelihoods)) {
                    gb.alleles(NO_CALL_ALLELES);
                    gb.noGQ();
                    break;
                }
                int PLindex = MathUtils.maxElementIndex(newLikelihoods);
                GenotypeLikelihoods.GenotypeLikelihoodsAllelePair alleles = GenotypeLikelihoods.getAllelePair(PLindex);
                gb.alleles(Arrays.asList(allelesToUse.get(alleles.alleleIndex1), allelesToUse.get(alleles.alleleIndex2)));
                gb.log10PError(GenotypeLikelihoods.getGQLog10FromLikelihoods(PLindex, newLikelihoods));
                break;
            }
            case BEST_MATCH_TO_ORIGINAL: {
                LinkedList<Allele> best = new LinkedList<Allele>();
                Allele ref = allelesToUse.get(0);
                for (Allele originalAllele : originalGT) {
                    best.add(allelesToUse.contains(originalAllele) ? originalAllele : ref);
                }
                gb.noGQ();
                gb.noPL();
                gb.alleles(best);
            }
        }
    }

    public static GenotypesContext subsetToRefOnly(VariantContext vc, int ploidy) {
        if (vc == null) {
            throw new IllegalArgumentException("vc cannot be null");
        }
        if (ploidy < 1) {
            throw new IllegalArgumentException("ploidy must be >= 1 but got " + ploidy);
        }
        GenotypesContext oldGTs = vc.getGenotypes();
        if (oldGTs.isEmpty()) {
            return oldGTs;
        }
        GenotypesContext newGTs = GenotypesContext.create(oldGTs.size());
        Allele ref = vc.getReference();
        List<Allele> diploidRefAlleles = Arrays.asList(ref, ref);
        for (Genotype g2 : vc.getGenotypes()) {
            int gPloidy = g2.getPloidy() == 0 ? ploidy : g2.getPloidy();
            List<Allele> refAlleles = gPloidy == 2 ? diploidRefAlleles : Collections.nCopies(gPloidy, ref);
            GenotypeBuilder gb = new GenotypeBuilder(g2.getSampleName(), refAlleles);
            if (g2.hasDP()) {
                gb.DP(g2.getDP());
            }
            if (g2.hasGQ()) {
                gb.GQ(g2.getGQ());
            }
            newGTs.add(gb.make());
        }
        return newGTs;
    }

    public static GenotypesContext assignDiploidGenotypes(VariantContext vc) {
        return GATKVariantContextUtils.subsetDiploidAlleles(vc, vc.getAlleles(), GenotypeAssignmentMethod.USE_PLS_TO_ASSIGN);
    }

    public static List<VariantContext> splitVariantContextToBiallelics(VariantContext vc) {
        return GATKVariantContextUtils.splitVariantContextToBiallelics(vc, false, GenotypeAssignmentMethod.SET_TO_NO_CALL);
    }

    public static List<VariantContext> splitVariantContextToBiallelics(VariantContext vc, boolean trimLeft, GenotypeAssignmentMethod genotypeAssignmentMethod) {
        if (!vc.isVariant() || vc.isBiallelic()) {
            return Collections.singletonList(vc);
        }
        LinkedList<VariantContext> biallelics = new LinkedList<VariantContext>();
        for (Allele alt : vc.getAlternateAlleles()) {
            VariantContextBuilder builder = new VariantContextBuilder(vc);
            List<Allele> alleles = Arrays.asList(vc.getReference(), alt);
            builder.alleles((Collection<Allele>)alleles);
            builder.genotypes(GATKVariantContextUtils.subsetDiploidAlleles(vc, alleles, genotypeAssignmentMethod));
            VariantContextUtils.calculateChromosomeCounts(builder, true);
            VariantContext trimmed = GATKVariantContextUtils.trimAlleles(builder.make(), trimLeft, true);
            biallelics.add(trimmed);
        }
        return biallelics;
    }

    public static Genotype removePLsAndAD(Genotype g2) {
        return g2.hasLikelihoods() || g2.hasAD() ? new GenotypeBuilder(g2).noPL().noAD().make() : g2;
    }

    public static VariantContext simpleMerge(Collection<VariantContext> unsortedVCs, List<String> priorityListOfVCs, FilteredRecordMergeType filteredRecordMergeType, GenotypeMergeType genotypeMergeOptions, boolean annotateOrigin, boolean printMessages, String setKey, boolean filteredAreUncalled, boolean mergeInfoWithMaxAC) {
        int originalNumOfVCs = priorityListOfVCs == null ? 0 : priorityListOfVCs.size();
        return GATKVariantContextUtils.simpleMerge(unsortedVCs, priorityListOfVCs, originalNumOfVCs, filteredRecordMergeType, genotypeMergeOptions, annotateOrigin, printMessages, setKey, filteredAreUncalled, mergeInfoWithMaxAC);
    }

    public static VariantContext simpleMerge(Collection<VariantContext> unsortedVCs, List<String> priorityListOfVCs, int originalNumOfVCs, FilteredRecordMergeType filteredRecordMergeType, GenotypeMergeType genotypeMergeOptions, boolean annotateOrigin, boolean printMessages, String setKey, boolean filteredAreUncalled, boolean mergeInfoWithMaxAC) {
        if (unsortedVCs == null || unsortedVCs.size() == 0) {
            return null;
        }
        if (priorityListOfVCs != null && originalNumOfVCs != priorityListOfVCs.size()) {
            throw new IllegalArgumentException("the number of the original VariantContexts must be the same as the number of VariantContexts in the priority list");
        }
        if (annotateOrigin && priorityListOfVCs == null && originalNumOfVCs == 0) {
            throw new IllegalArgumentException("Cannot merge calls and annotate their origins without a complete priority list of VariantContexts or the number of original VariantContexts");
        }
        List<VariantContext> preFilteredVCs = GATKVariantContextUtils.sortVariantContextsByPriority(unsortedVCs, priorityListOfVCs, genotypeMergeOptions);
        ArrayList<VariantContext> VCs = new ArrayList<VariantContext>();
        for (VariantContext vc : preFilteredVCs) {
            if (filteredAreUncalled && !vc.isNotFiltered()) continue;
            VCs.add(vc);
        }
        if (VCs.size() == 0) {
            return null;
        }
        VariantContext first = (VariantContext)VCs.get(0);
        String name = first.getSource();
        Allele refAllele = GATKVariantContextUtils.determineReferenceAllele(VCs);
        LinkedHashSet<Allele> alleles = new LinkedHashSet<Allele>();
        HashSet<String> filters = new HashSet<String>();
        LinkedHashMap<String, Object> attributes = new LinkedHashMap<String, Object>();
        HashSet<String> inconsistentAttributes = new HashSet<String>();
        HashSet<String> variantSources = new HashSet<String>();
        LinkedHashSet<String> rsIDs = new LinkedHashSet<String>(1);
        VariantContext longestVC = first;
        int depth = 0;
        int maxAC = -1;
        LinkedHashMap<String, Object> attributesWithMaxAC = new LinkedHashMap<String, Object>();
        double log10PError = 1.0;
        boolean anyVCHadFiltersApplied = false;
        VariantContext vcWithMaxAC = null;
        GenotypesContext genotypes = GenotypesContext.create();
        int nFiltered = 0;
        boolean remapped = false;
        for (VariantContext vc : VCs) {
            if (longestVC.getStart() != vc.getStart()) {
                throw new IllegalStateException("BUG: attempting to merge VariantContexts with different start sites: first=" + first.toString() + " second=" + vc.toString());
            }
            if (VariantContextUtils.getSize(vc) > VariantContextUtils.getSize(longestVC)) {
                longestVC = vc;
            }
            nFiltered += vc.isFiltered() ? 1 : 0;
            if (vc.isVariant()) {
                variantSources.add(vc.getSource());
            }
            AlleleMapper alleleMapping = GATKVariantContextUtils.resolveIncompatibleAlleles(refAllele, vc, alleles);
            remapped = remapped || alleleMapping.needsRemapping();
            alleles.addAll(alleleMapping.values());
            GATKVariantContextUtils.mergeGenotypes(genotypes, vc, alleleMapping, genotypeMergeOptions == GenotypeMergeType.UNIQUIFY);
            if (log10PError == 1.0) {
                log10PError = vc.getLog10PError();
            }
            filters.addAll(vc.getFilters());
            anyVCHadFiltersApplied |= vc.filtersWereApplied();
            if (vc.hasAttribute("DP")) {
                depth += vc.getAttributeAsInt("DP", 0);
            }
            if (vc.hasID()) {
                rsIDs.add(vc.getID());
            }
            if (mergeInfoWithMaxAC && vc.hasAttribute("AC")) {
                String rawAlleleCounts = vc.getAttributeAsString("AC", null);
                if (rawAlleleCounts.contains(",")) {
                    List<String> alleleCountArray = Arrays.asList(rawAlleleCounts.substring(1, rawAlleleCounts.length() - 1).split(","));
                    for (String alleleCount : alleleCountArray) {
                        int ac = Integer.valueOf(alleleCount.trim());
                        if (ac <= maxAC) continue;
                        maxAC = ac;
                        vcWithMaxAC = vc;
                    }
                } else {
                    int ac = Integer.valueOf(rawAlleleCounts);
                    if (ac > maxAC) {
                        maxAC = ac;
                        vcWithMaxAC = vc;
                    }
                }
            }
            for (Map.Entry<String, Object> p2 : vc.getAttributes().entrySet()) {
                boolean boundIsMissingValue;
                String key = p2.getKey();
                Object value = p2.getValue();
                if (inconsistentAttributes.contains(key)) continue;
                boolean alreadyFound = attributes.containsKey(key);
                Object boundValue = attributes.get(key);
                boolean bl = boundIsMissingValue = alreadyFound && boundValue.equals(".");
                if (alreadyFound && !boundValue.equals(value) && !boundIsMissingValue) {
                    inconsistentAttributes.add(key);
                    attributes.remove(key);
                    continue;
                }
                if (alreadyFound && !boundIsMissingValue) continue;
                attributes.put(key, value);
            }
        }
        for (VariantContext vc : VCs) {
            if (vc.getAlleles().size() == 1 || !GATKVariantContextUtils.hasPLIncompatibleAlleles(alleles, vc.getAlleles())) continue;
            if (!genotypes.isEmpty()) {
                logger.debug(String.format("Stripping PLs at %s:%d-%d due to incompatible alleles merged=%s vs. single=%s", vc.getChr(), vc.getStart(), vc.getEnd(), alleles, vc.getAlleles()));
            }
            genotypes = GATKVariantContextUtils.stripPLsAndAD(genotypes);
            VariantContextUtils.calculateChromosomeCounts(vc, attributes, true);
            break;
        }
        if (mergeInfoWithMaxAC && vcWithMaxAC != null) {
            attributesWithMaxAC.putAll(vcWithMaxAC.getAttributes());
        }
        if (filteredRecordMergeType == FilteredRecordMergeType.KEEP_IF_ANY_UNFILTERED && nFiltered != VCs.size() || filteredRecordMergeType == FilteredRecordMergeType.KEEP_UNCONDITIONAL) {
            filters.clear();
        }
        if (annotateOrigin) {
            String setValue;
            if (nFiltered == 0 && variantSources.size() == originalNumOfVCs) {
                setValue = MERGE_INTERSECTION;
            } else if (nFiltered == VCs.size()) {
                setValue = MERGE_FILTER_IN_ALL;
            } else if (variantSources.isEmpty()) {
                setValue = MERGE_REF_IN_ALL;
            } else {
                LinkedHashSet<String> s2 = new LinkedHashSet<String>();
                for (VariantContext vc : VCs) {
                    if (!vc.isVariant()) continue;
                    s2.add(vc.isFiltered() ? MERGE_FILTER_PREFIX + vc.getSource() : vc.getSource());
                }
                setValue = Utils.join("-", s2);
            }
            if (setKey != null) {
                attributes.put(setKey, setValue);
                if (mergeInfoWithMaxAC && vcWithMaxAC != null) {
                    attributesWithMaxAC.put(setKey, setValue);
                }
            }
        }
        if (depth > 0) {
            attributes.put("DP", String.valueOf(depth));
        }
        String ID = rsIDs.isEmpty() ? "." : Utils.join(",", rsIDs);
        VariantContextBuilder builder = new VariantContextBuilder().source(name).id(ID);
        builder.loc(longestVC.getChr(), longestVC.getStart(), longestVC.getEnd());
        builder.alleles(alleles);
        builder.genotypes(genotypes);
        builder.log10PError(log10PError);
        if (anyVCHadFiltersApplied) {
            builder.filters(filters.isEmpty() ? filters : new TreeSet(filters));
        }
        builder.attributes(new TreeMap(mergeInfoWithMaxAC ? attributesWithMaxAC : attributes));
        VariantContext merged = builder.make();
        if (printMessages && remapped) {
            System.out.printf("Remapped => %s%n", merged);
        }
        return merged;
    }

    private static Comparable combineAnnotationValues(List<Comparable> array) {
        return MathUtils.median(array);
    }

    public static VariantContext referenceConfidenceMerge(List<VariantContext> VCs, GenomeLoc loc, Byte refBase, boolean removeNonRefSymbolicAllele) {
        if (VCs == null || VCs.size() == 0) {
            return null;
        }
        VariantContext first = VCs.get(0);
        String name = first.getSource();
        Allele refAllele = GATKVariantContextUtils.determineReferenceAlleleGivenReferenceBase(VCs, loc, refBase);
        if (refAllele == null) {
            return null;
        }
        LinkedHashSet<Allele> finalAlleleSet = new LinkedHashSet<Allele>(10);
        finalAlleleSet.add(refAllele);
        LinkedHashMap<String, Object> attributes = new LinkedHashMap<String, Object>();
        LinkedHashSet<String> rsIDs = new LinkedHashSet<String>(1);
        int depth = 0;
        LinkedHashMap<String, List<Comparable>> annotationMap = new LinkedHashMap<String, List<Comparable>>();
        GenotypesContext genotypes = GenotypesContext.create();
        int variantContextCount = VCs.size();
        ArrayList<Pair<VariantContext, List<Allele>>> vcAndNewAllelePairs = new ArrayList<Pair<VariantContext, List<Allele>>>(variantContextCount);
        for (VariantContext vc : VCs) {
            boolean bl = loc.getStart() != vc.getStart();
            vcAndNewAllelePairs.add(new Pair<VariantContext, List<Allele>>(vc, bl ? GATKVariantContextUtils.replaceWithNoCalls(vc.getAlleles()) : GATKVariantContextUtils.remapAlleles(vc.getAlleles(), refAllele, finalAlleleSet)));
        }
        if (!removeNonRefSymbolicAllele) {
            finalAlleleSet.add(NON_REF_SYMBOLIC_ALLELE);
        }
        ArrayList<Allele> allelesList = new ArrayList<Allele>(finalAlleleSet);
        for (Pair pair : vcAndNewAllelePairs) {
            VariantContext vc = (VariantContext)pair.getFirst();
            List remappedAlleles = (List)pair.getSecond();
            GATKVariantContextUtils.mergeRefConfidenceGenotypes(genotypes, vc, remappedAlleles, allelesList);
            if (vc.hasAttribute("DP")) {
                depth += vc.getAttributeAsInt("DP", 0);
            } else {
                for (Genotype gt : vc.getGenotypes()) {
                    depth += gt.hasExtendedAttribute("MIN_DP") ? Integer.parseInt((String)gt.getAnyAttribute("MIN_DP")) : (gt.hasDP() ? gt.getDP() : 0);
                }
            }
            if (loc.getStart() != vc.getStart()) continue;
            if (vc.hasID()) {
                rsIDs.add(vc.getID());
            }
            GATKVariantContextUtils.addReferenceConfidenceAttributes(vc.getAttributes(), annotationMap);
        }
        for (Map.Entry entry : annotationMap.entrySet()) {
            if (((List)entry.getValue()).isEmpty()) continue;
            attributes.put((String)entry.getKey(), GATKVariantContextUtils.combineAnnotationValues((List)entry.getValue()));
        }
        if (depth > 0) {
            attributes.put("DP", String.valueOf(depth));
        }
        GATKVariantContextUtils.removeStaleAttributesAfterMerge(attributes);
        String ID = rsIDs.isEmpty() ? "." : Utils.join(",", rsIDs);
        VariantContextBuilder variantContextBuilder = new VariantContextBuilder().source(name).id(ID).alleles((Collection<Allele>)allelesList).chr(loc.getContig()).start(loc.getStart()).computeEndFromAlleles(allelesList, loc.getStart(), loc.getStart()).genotypes(genotypes).unfiltered().attributes(new TreeMap<String, Object>(attributes)).log10PError(1.0);
        return variantContextBuilder.make();
    }

    private static Allele determineReferenceAlleleGivenReferenceBase(List<VariantContext> VCs, GenomeLoc loc, Byte refBase) {
        Allele refAllele = GATKVariantContextUtils.determineReferenceAllele(VCs, loc);
        if (refAllele == null) {
            return refBase == null ? null : Allele.create(refBase, true);
        }
        return refAllele;
    }

    private static void removeStaleAttributesAfterMerge(Map<String, Object> attributes) {
        attributes.remove("AC");
        attributes.remove("AF");
        attributes.remove("AN");
        attributes.remove("MLEAC");
        attributes.remove("MLEAF");
        attributes.remove("END");
    }

    private static void addReferenceConfidenceAttributes(Map<String, Object> myAttributes, Map<String, List<Comparable>> annotationMap) {
        for (Map.Entry<String, Object> p2 : myAttributes.entrySet()) {
            String key = p2.getKey();
            Object value = p2.getValue();
            List<Comparable> values = annotationMap.get(key);
            if (values == null) {
                values = new ArrayList<Comparable>();
                annotationMap.put(key, values);
            }
            try {
                String stringValue = value.toString();
                if (stringValue.contains(".")) {
                    values.add(Double.valueOf(Double.parseDouble(stringValue)));
                    continue;
                }
                values.add(Integer.valueOf(Integer.parseInt(stringValue)));
            }
            catch (NumberFormatException e2) {}
        }
    }

    private static boolean hasPLIncompatibleAlleles(Collection<Allele> alleleSet1, Collection<Allele> alleleSet2) {
        Iterator<Allele> it1 = alleleSet1.iterator();
        Iterator<Allele> it2 = alleleSet2.iterator();
        while (it1.hasNext() && it2.hasNext()) {
            Allele a2;
            Allele a1 = it1.next();
            if (a1.equals(a2 = it2.next())) continue;
            return true;
        }
        return it1.hasNext() || it2.hasNext();
    }

    private static List<Allele> remapAlleles(List<Allele> vcAlleles, Allele refAllele, LinkedHashSet<Allele> finalAlleles) {
        Allele vcRef = vcAlleles.get(0);
        if (!vcRef.isReference()) {
            throw new IllegalStateException("the first allele of the vc allele list must be reference");
        }
        byte[] refBases = refAllele.getBases();
        int extraBaseCount = refBases.length - vcRef.getBases().length;
        if (extraBaseCount < 0) {
            throw new IllegalStateException("the wrong reference was selected");
        }
        ArrayList<Allele> result = new ArrayList<Allele>(vcAlleles.size());
        for (Allele a2 : vcAlleles) {
            if (a2.isReference()) {
                result.add(refAllele);
                continue;
            }
            if (a2.isSymbolic()) {
                result.add(a2);
                if (a2.equals(NON_REF_SYMBOLIC_ALLELE)) continue;
                finalAlleles.add(a2);
                continue;
            }
            if (a2.isCalled()) {
                Allele newAllele;
                if (extraBaseCount > 0) {
                    byte[] oldBases = a2.getBases();
                    byte[] newBases = Arrays.copyOf(oldBases, oldBases.length + extraBaseCount);
                    System.arraycopy(refBases, refBases.length - extraBaseCount, newBases, oldBases.length, extraBaseCount);
                    newAllele = Allele.create(newBases, false);
                } else {
                    newAllele = a2;
                }
                result.add(newAllele);
                finalAlleles.add(newAllele);
                continue;
            }
            result.add(a2);
        }
        return result;
    }

    public static GenotypesContext stripPLsAndAD(GenotypesContext genotypes) {
        GenotypesContext newGs = GenotypesContext.create(genotypes.size());
        for (Genotype g2 : genotypes) {
            newGs.add(GATKVariantContextUtils.removePLsAndAD(g2));
        }
        return newGs;
    }

    public static GenotypesContext updatePLsAndAD(VariantContext selectedVC, VariantContext originalVC) {
        int numOriginalAlleles;
        int numNewAlleles = selectedVC.getAlleles().size();
        if (numNewAlleles > (numOriginalAlleles = originalVC.getAlleles().size())) {
            throw new IllegalArgumentException("Attempting to fix PLs and AD from what appears to be a *combined* VCF and not a selected one");
        }
        GenotypesContext oldGs = selectedVC.getGenotypes();
        if (numNewAlleles == numOriginalAlleles) {
            return oldGs;
        }
        return GATKVariantContextUtils.fixGenotypesFromSubsettedAlleles(oldGs, originalVC, selectedVC.getAlleles());
    }

    private static GenotypesContext fixGenotypesFromSubsettedAlleles(GenotypesContext originalGs, VariantContext originalVC, List<Allele> allelesToUse) {
        List<Integer> likelihoodIndexesToUse = GATKVariantContextUtils.determineLikelihoodIndexesToUse(originalVC, allelesToUse);
        return GATKVariantContextUtils.createGenotypesWithSubsettedLikelihoods(originalGs, originalVC, allelesToUse, likelihoodIndexesToUse, GenotypeAssignmentMethod.DO_NOT_ASSIGN_GENOTYPES);
    }

    private static GenotypesContext fixADFromSubsettedAlleles(GenotypesContext originalGs, VariantContext originalVC, List<Allele> allelesToUse) {
        boolean[] alleleIndexesToUse = GATKVariantContextUtils.getAlleleIndexBitset(originalVC, allelesToUse);
        GenotypesContext newGTs = GenotypesContext.create(originalGs.size());
        List<String> sampleIndices = originalGs.getSampleNamesOrderedByName();
        for (int k2 = 0; k2 < originalGs.size(); ++k2) {
            Genotype g2 = originalGs.get(sampleIndices.get(k2));
            newGTs.add(GATKVariantContextUtils.fixAD(g2, alleleIndexesToUse, allelesToUse.size()));
        }
        return newGTs;
    }

    private static Genotype fixAD(Genotype genotype, boolean[] alleleIndexesToUse, int nAllelesToUse) {
        if (!genotype.hasAD()) {
            return genotype;
        }
        GenotypeBuilder builder = new GenotypeBuilder(genotype);
        int[] oldAD = genotype.getAD();
        if (oldAD.length != alleleIndexesToUse.length) {
            builder.noAD();
        } else {
            int[] newAD = new int[nAllelesToUse];
            int currentIndex = 0;
            for (int i2 = 0; i2 < oldAD.length; ++i2) {
                if (!alleleIndexesToUse[i2]) continue;
                newAD[currentIndex++] = oldAD[i2];
            }
            builder.AD(newAD);
        }
        return builder.make();
    }

    private static Allele determineReferenceAllele(List<VariantContext> VCs) {
        return GATKVariantContextUtils.determineReferenceAllele(VCs, null);
    }

    private static Allele determineReferenceAllele(List<VariantContext> VCs, GenomeLoc loc) {
        Allele ref = null;
        for (VariantContext vc : VCs) {
            if (!GATKVariantContextUtils.contextMatchesLoc(vc, loc)) continue;
            Allele myRef = vc.getReference();
            if (ref == null || ref.length() < myRef.length()) {
                ref = myRef;
                continue;
            }
            if (ref.length() != myRef.length() || ref.equals(myRef)) continue;
            throw new TribbleException(String.format("The provided variant file(s) have inconsistent references for the same position(s) at %s:%d, %s vs. %s", vc.getChr(), vc.getStart(), ref, myRef));
        }
        return ref;
    }

    public static boolean contextMatchesLoc(VariantContext vc, GenomeLoc loc) {
        return loc == null || loc.getStart() == vc.getStart();
    }

    private static AlleleMapper resolveIncompatibleAlleles(Allele refAllele, VariantContext vc, Set<Allele> allAlleles) {
        if (refAllele.equals(vc.getReference())) {
            return new AlleleMapper(vc);
        }
        Map<Allele, Allele> map = GATKVariantContextUtils.createAlleleMapping(refAllele, vc, allAlleles);
        map.put(vc.getReference(), refAllele);
        return new AlleleMapper(map);
    }

    private static Map<Allele, Allele> createAlleleMapping(Allele refAllele, VariantContext oneVC, Collection<Allele> currentAlleles) {
        Allele myRef = oneVC.getReference();
        if (refAllele.length() <= myRef.length()) {
            throw new IllegalStateException("BUG: myRef=" + myRef + " is longer than refAllele=" + refAllele);
        }
        byte[] extraBases = Arrays.copyOfRange(refAllele.getBases(), myRef.length(), refAllele.length());
        HashMap<Allele, Allele> map = new HashMap<Allele, Allele>();
        for (Allele a2 : oneVC.getAlternateAlleles()) {
            if (!GATKVariantContextUtils.isUsableAlternateAllele(a2)) continue;
            Allele extended = Allele.extend(a2, extraBases);
            for (Allele b2 : currentAlleles) {
                if (!extended.equals(b2)) continue;
                extended = b2;
            }
            map.put(a2, extended);
        }
        return map;
    }

    private static boolean isUsableAlternateAllele(Allele allele) {
        return !allele.isReference() && !allele.isSymbolic();
    }

    public static List<VariantContext> sortVariantContextsByPriority(Collection<VariantContext> unsortedVCs, List<String> priorityListOfVCs, GenotypeMergeType mergeOption) {
        if (mergeOption == GenotypeMergeType.PRIORITIZE && priorityListOfVCs == null) {
            throw new IllegalArgumentException("Cannot merge calls by priority with a null priority list");
        }
        if (priorityListOfVCs == null || mergeOption == GenotypeMergeType.UNSORTED) {
            return new ArrayList<VariantContext>(unsortedVCs);
        }
        ArrayList<VariantContext> sorted = new ArrayList<VariantContext>(unsortedVCs);
        Collections.sort(sorted, new CompareByPriority(priorityListOfVCs));
        return sorted;
    }

    private static void mergeGenotypes(GenotypesContext mergedGenotypes, VariantContext oneVC, AlleleMapper alleleMapping, boolean uniquifySamples) {
        for (Genotype g2 : oneVC.getGenotypes()) {
            String name = GATKVariantContextUtils.mergedSampleName(oneVC.getSource(), g2.getSampleName(), uniquifySamples);
            if (mergedGenotypes.containsSample(name)) continue;
            Genotype newG = g2;
            if (uniquifySamples || alleleMapping.needsRemapping()) {
                List<Allele> alleles = alleleMapping.needsRemapping() ? alleleMapping.remap(g2.getAlleles()) : g2.getAlleles();
                newG = new GenotypeBuilder(g2).name(name).alleles(alleles).make();
            }
            mergedGenotypes.add(newG);
        }
    }

    private static List<Allele> replaceWithNoCalls(List<Allele> alleles) {
        if (alleles == null) {
            throw new IllegalArgumentException("list of alleles cannot be null");
        }
        ArrayList<Allele> result = new ArrayList<Allele>(alleles.size());
        for (Allele allele : alleles) {
            result.add(allele.equals(NON_REF_SYMBOLIC_ALLELE) ? allele : Allele.NO_CALL);
        }
        return result;
    }

    private static void mergeRefConfidenceGenotypes(GenotypesContext mergedGenotypes, VariantContext VC, List<Allele> remappedAlleles, List<Allele> targetAlleles) {
        for (Genotype g2 : VC.getGenotypes()) {
            String name = g2.getSampleName();
            if (mergedGenotypes.containsSample(name)) continue;
            GenotypeBuilder genotypeBuilder = new GenotypeBuilder(g2).alleles(GATKVariantContextUtils.noCallAlleles(g2.getPloidy()));
            if (g2.hasPL()) {
                int[] indexesOfRelevantAlleles = GATKVariantContextUtils.getIndexesOfRelevantAlleles(remappedAlleles, targetAlleles, VC.getStart());
                int[] PLs = GATKVariantContextUtils.generatePLs(g2, indexesOfRelevantAlleles);
                int[] AD = g2.hasAD() ? GATKVariantContextUtils.generateAD(g2.getAD(), indexesOfRelevantAlleles) : null;
                genotypeBuilder.PL(PLs).AD(AD).noGQ();
            }
            mergedGenotypes.add(genotypeBuilder.make());
        }
    }

    public static List<Allele> noCallAlleles(int ploidy) {
        if (ploidy <= 0) {
            return Collections.EMPTY_LIST;
        }
        if (ploidy == 1) {
            return Collections.singletonList(Allele.NO_CALL);
        }
        ArrayList<Allele> result = new ArrayList<Allele>(ploidy);
        for (int i2 = 0; i2 < ploidy; ++i2) {
            result.add(Allele.NO_CALL);
        }
        return result;
    }

    protected static int[] getIndexesOfRelevantAlleles(List<Allele> remappedAlleles, List<Allele> targetAlleles, int position) {
        if (remappedAlleles == null || remappedAlleles.size() == 0) {
            throw new IllegalArgumentException("The list of input alleles must not be null or empty");
        }
        if (targetAlleles == null || targetAlleles.size() == 0) {
            throw new IllegalArgumentException("The list of target alleles must not be null or empty");
        }
        if (!remappedAlleles.contains(NON_REF_SYMBOLIC_ALLELE)) {
            throw new UserException("The list of input alleles must contain " + NON_REF_SYMBOLIC_ALLELE + " as an allele but that is not the case at position " + position + "; please use the Haplotype Caller with gVCF output to generate appropriate records");
        }
        int indexOfGenericAlt = remappedAlleles.indexOf(NON_REF_SYMBOLIC_ALLELE);
        int[] indexMapping = new int[targetAlleles.size()];
        indexMapping[0] = 0;
        for (int i2 = 1; i2 < targetAlleles.size(); ++i2) {
            int indexOfRemappedAllele = remappedAlleles.indexOf(targetAlleles.get(i2));
            indexMapping[i2] = indexOfRemappedAllele == -1 ? indexOfGenericAlt : indexOfRemappedAllele;
        }
        return indexMapping;
    }

    protected static int[] generatePLs(Genotype genotype, int[] indexesOfRelevantAlleles) {
        if (!genotype.hasPL()) {
            throw new IllegalArgumentException("Cannot generate new PLs from a genotype without PLs");
        }
        int[] originalPLs = genotype.getPL();
        int numLikelihoods = GenotypeLikelihoods.numLikelihoods(indexesOfRelevantAlleles.length, 2);
        int[] newPLs = new int[numLikelihoods];
        for (int i2 = 0; i2 < indexesOfRelevantAlleles.length; ++i2) {
            for (int j2 = i2; j2 < indexesOfRelevantAlleles.length; ++j2) {
                int originalPLindex = GATKVariantContextUtils.calculatePLindexFromUnorderedIndexes(indexesOfRelevantAlleles[i2], indexesOfRelevantAlleles[j2]);
                if (originalPLindex >= originalPLs.length) {
                    throw new IllegalStateException("The original PLs do not have enough values; accessing index " + originalPLindex + " but size is " + originalPLs.length);
                }
                int newPLindex = GenotypeLikelihoods.calculatePLindex(i2, j2);
                newPLs[newPLindex] = originalPLs[originalPLindex];
            }
        }
        return newPLs;
    }

    protected static int[] generateAD(int[] originalAD, int[] indexesOfRelevantAlleles) {
        if (originalAD == null || indexesOfRelevantAlleles == null) {
            throw new IllegalArgumentException("The list of input AD values and alleles must not be null");
        }
        int numADs = indexesOfRelevantAlleles.length;
        if (numADs == originalAD.length) {
            return originalAD;
        }
        int[] newAD = new int[numADs];
        for (int i2 = 0; i2 < numADs; ++i2) {
            int oldIndex = indexesOfRelevantAlleles[i2];
            newAD[i2] = oldIndex >= originalAD.length ? 0 : originalAD[oldIndex];
        }
        return newAD;
    }

    protected static int calculatePLindexFromUnorderedIndexes(int originalIndex1, int originalIndex2) {
        return originalIndex2 < originalIndex1 ? GenotypeLikelihoods.calculatePLindex(originalIndex2, originalIndex1) : GenotypeLikelihoods.calculatePLindex(originalIndex1, originalIndex2);
    }

    public static String mergedSampleName(String trackName, String sampleName, boolean uniquify) {
        return uniquify ? sampleName + "." + trackName : sampleName;
    }

    public static VariantContext reverseTrimAlleles(VariantContext inputVC) {
        return GATKVariantContextUtils.trimAlleles(inputVC, false, true);
    }

    public static VariantContext forwardTrimAlleles(VariantContext inputVC) {
        return GATKVariantContextUtils.trimAlleles(inputVC, true, false);
    }

    @Ensures(value={"result != null"})
    public static VariantContext trimAlleles(VariantContext inputVC, boolean trimForward, boolean trimReverse) {
        if (inputVC == null) {
            throw new IllegalArgumentException("inputVC cannot be null");
        }
        if (inputVC.getNAlleles() <= 1 || inputVC.isSNP()) {
            return inputVC;
        }
        int revTrim = trimReverse ? GATKVariantContextUtils.computeReverseClipping(inputVC.getAlleles(), inputVC.getReference().getDisplayString().getBytes()) : 0;
        VariantContext revTrimVC = GATKVariantContextUtils.trimAlleles(inputVC, -1, revTrim);
        int fwdTrim = trimForward ? GATKVariantContextUtils.computeForwardClipping(revTrimVC.getAlleles()) : -1;
        VariantContext vc = GATKVariantContextUtils.trimAlleles(revTrimVC, fwdTrim, 0);
        return vc;
    }

    @Requires(value={"inputVC != null"})
    @Ensures(value={"result != null"})
    protected static VariantContext trimAlleles(VariantContext inputVC, int fwdTrimEnd, int revTrim) {
        if (fwdTrimEnd == -1 && revTrim == 0) {
            return inputVC;
        }
        LinkedList<Allele> alleles = new LinkedList<Allele>();
        HashMap<Allele, Allele> originalToTrimmedAlleleMap = new HashMap<Allele, Allele>();
        for (Allele a2 : inputVC.getAlleles()) {
            if (a2.isSymbolic()) {
                alleles.add(a2);
                originalToTrimmedAlleleMap.put(a2, a2);
                continue;
            }
            byte[] newBases = Arrays.copyOfRange(a2.getBases(), fwdTrimEnd + 1, a2.length() - revTrim);
            Allele trimmedAllele = Allele.create(newBases, a2.isReference());
            alleles.add(trimmedAllele);
            originalToTrimmedAlleleMap.put(a2, trimmedAllele);
        }
        AlleleMapper alleleMapper = new AlleleMapper(originalToTrimmedAlleleMap);
        GenotypesContext genotypes = GATKVariantContextUtils.updateGenotypesWithMappedAlleles(inputVC.getGenotypes(), alleleMapper);
        int start = inputVC.getStart() + (fwdTrimEnd + 1);
        VariantContextBuilder builder = new VariantContextBuilder(inputVC);
        builder.start(start);
        builder.stop(start + ((Allele)alleles.get(0)).length() - 1);
        builder.alleles((Collection<Allele>)alleles);
        builder.genotypes(genotypes);
        return builder.make();
    }

    @Requires(value={"originalGenotypes != null && alleleMapper != null"})
    protected static GenotypesContext updateGenotypesWithMappedAlleles(GenotypesContext originalGenotypes, AlleleMapper alleleMapper) {
        GenotypesContext updatedGenotypes = GenotypesContext.create(originalGenotypes.size());
        for (Genotype genotype : originalGenotypes) {
            List<Allele> updatedAlleles = alleleMapper.remap(genotype.getAlleles());
            updatedGenotypes.add(new GenotypeBuilder(genotype).alleles(updatedAlleles).make());
        }
        return updatedGenotypes;
    }

    public static int computeReverseClipping(List<Allele> unclippedAlleles, byte[] ref) {
        int clipping = 0;
        boolean stillClipping = true;
        while (stillClipping) {
            for (Allele a2 : unclippedAlleles) {
                if (a2.isSymbolic()) continue;
                if (a2.length() - clipping == 0) {
                    return clipping - 1;
                }
                if (a2.length() - clipping <= 0 || a2.length() == 0) {
                    stillClipping = false;
                    continue;
                }
                if (ref.length == clipping) {
                    return -1;
                }
                if (a2.getBases()[a2.length() - clipping - 1] == ref[ref.length - clipping - 1]) continue;
                stillClipping = false;
            }
            if (!stillClipping) continue;
            ++clipping;
        }
        return clipping;
    }

    public static int computeForwardClipping(List<Allele> unclippedAlleles) {
        if (unclippedAlleles.size() <= 1) {
            return -1;
        }
        int minAlleleLength = Integer.MAX_VALUE;
        for (Allele a2 : unclippedAlleles) {
            if (a2.isSymbolic()) {
                return -1;
            }
            minAlleleLength = Math.min(minAlleleLength, a2.length());
        }
        byte[] firstAlleleBases = unclippedAlleles.get(0).getBases();
        int indexOflastSharedBase = -1;
        int i2 = 0;
        while (i2 < minAlleleLength - 1) {
            byte base = firstAlleleBases[i2];
            for (Allele allele : unclippedAlleles) {
                if (allele.getBases()[i2] == base) continue;
                return indexOflastSharedBase;
            }
            indexOflastSharedBase = i2++;
        }
        return indexOflastSharedBase;
    }

    public static double computeHardyWeinbergPvalue(VariantContext vc) {
        if (vc.getCalledChrCount() == 0) {
            return 0.0;
        }
        return HardyWeinbergCalculation.hwCalculate(vc.getHomRefCount(), vc.getHetCount(), vc.getHomVarCount());
    }

    public static boolean requiresPaddingBase(List<String> alleles) {
        for (String allele : alleles) {
            if (!allele.isEmpty()) continue;
            return true;
        }
        int clipping = 0;
        Character currentBase = null;
        while (true) {
            for (String allele : alleles) {
                if (allele.length() - clipping == 0) {
                    return true;
                }
                char myBase = allele.charAt(clipping);
                if (currentBase == null) {
                    currentBase = Character.valueOf(myBase);
                    continue;
                }
                if (currentBase.charValue() == myBase) continue;
                return false;
            }
            ++clipping;
            currentBase = null;
        }
    }

    private static final Map<String, Object> subsetAttributes(CommonInfo igc, Collection<String> keysToPreserve) {
        HashMap<String, Object> attributes = new HashMap<String, Object>(keysToPreserve.size());
        for (String key : keysToPreserve) {
            if (!igc.hasAttribute(key)) continue;
            attributes.put(key, igc.getAttribute(key));
        }
        return attributes;
    }

    @Deprecated
    public static VariantContext pruneVariantContext(VariantContext vc, Collection<String> keysToPreserve) {
        return GATKVariantContextUtils.pruneVariantContext(new VariantContextBuilder(vc), keysToPreserve).make();
    }

    public static VariantContextBuilder pruneVariantContext(VariantContextBuilder builder, Collection<String> keysToPreserve) {
        VariantContext vc = builder.make();
        if (keysToPreserve == null) {
            keysToPreserve = Collections.emptyList();
        }
        Map<String, Object> attributes = GATKVariantContextUtils.subsetAttributes(vc.getCommonInfo(), keysToPreserve);
        GenotypesContext genotypes = GenotypesContext.create(vc.getNSamples());
        for (Genotype g2 : vc.getGenotypes()) {
            GenotypeBuilder gb = new GenotypeBuilder(g2);
            gb.noAD().noDP().noPL().noAttributes();
            genotypes.add(gb.make());
        }
        return builder.genotypes(genotypes).attributes(attributes);
    }

    public static boolean allelesAreSubset(VariantContext vc1, VariantContext vc2) {
        if (!vc1.getReference().equals(vc2.getReference())) {
            return false;
        }
        for (Allele a2 : vc1.getAlternateAlleles()) {
            if (vc2.getAlternateAlleles().contains(a2)) continue;
            return false;
        }
        return true;
    }

    public static Map<VariantContext.Type, List<VariantContext>> separateVariantContextsByType(Collection<VariantContext> VCs) {
        if (VCs == null) {
            throw new IllegalArgumentException("VCs cannot be null.");
        }
        HashMap<VariantContext.Type, List<VariantContext>> mappedVCs = new HashMap<VariantContext.Type, List<VariantContext>>();
        for (VariantContext vc : VCs) {
            VariantContext.Type vcType = vc.getType();
            boolean addtoOwnList = true;
            block1: for (VariantContext.Type type : VariantContext.Type.values()) {
                if (type.equals((Object)vcType) || !mappedVCs.containsKey((Object)type)) continue;
                List<VariantContext> vcList = mappedVCs.get((Object)type);
                for (int k2 = 0; k2 < vcList.size(); ++k2) {
                    VariantContext otherVC = vcList.get(k2);
                    if (GATKVariantContextUtils.allelesAreSubset(otherVC, vc)) {
                        vcList.remove(k2);
                        if (vcList.size() == 0) {
                            mappedVCs.remove((Object)type);
                        }
                        if (!mappedVCs.containsKey((Object)vcType)) {
                            mappedVCs.put(vcType, new ArrayList());
                        }
                        mappedVCs.get((Object)vcType).add(otherVC);
                        continue block1;
                    }
                    if (!GATKVariantContextUtils.allelesAreSubset(vc, otherVC)) continue;
                    mappedVCs.get((Object)type).add(vc);
                    addtoOwnList = false;
                    continue block1;
                }
            }
            if (!addtoOwnList) continue;
            if (!mappedVCs.containsKey((Object)vcType)) {
                mappedVCs.put(vcType, new ArrayList());
            }
            mappedVCs.get((Object)vcType).add(vc);
        }
        return mappedVCs;
    }

    public static VariantContext purgeUnallowedGenotypeAttributes(VariantContext vc, Set<String> allowedAttributes) {
        if (allowedAttributes == null) {
            return vc;
        }
        GenotypesContext newGenotypes = GenotypesContext.create(vc.getNSamples());
        for (Genotype genotype : vc.getGenotypes()) {
            HashMap<String, Object> attrs = new HashMap<String, Object>();
            for (Map.Entry<String, Object> attr : genotype.getExtendedAttributes().entrySet()) {
                if (!allowedAttributes.contains(attr.getKey())) continue;
                attrs.put(attr.getKey(), attr.getValue());
            }
            newGenotypes.add(new GenotypeBuilder(genotype).attributes(attrs).make());
        }
        return new VariantContextBuilder(vc).genotypes(newGenotypes).make();
    }

    public static VariantContext makeFromAlleles(String name, String contig, int start, List<String> alleleStrings) {
        if (alleleStrings == null || alleleStrings.isEmpty()) {
            throw new IllegalArgumentException("alleleStrings must be non-empty, non-null list");
        }
        LinkedList<Allele> alleles = new LinkedList<Allele>();
        int length = alleleStrings.get(0).length();
        boolean first = true;
        for (String alleleString : alleleStrings) {
            alleles.add(Allele.create(alleleString, first));
            first = false;
        }
        return new VariantContextBuilder(name, contig, start, start + length - 1, alleles).make();
    }

    public static List<VariantContext> splitIntoPrimitiveAlleles(VariantContext vc) {
        byte[] alt;
        if (vc == null) {
            throw new IllegalArgumentException("Trying to break a null Variant Context into primitive parts");
        }
        if (!vc.isBiallelic()) {
            throw new IllegalArgumentException("Trying to break a multi-allelic Variant Context into primitive parts");
        }
        if (!vc.isMNP()) {
            return Arrays.asList(vc);
        }
        byte[] ref = vc.getReference().getBases();
        if (ref.length != (alt = vc.getAlternateAllele(0).getBases()).length) {
            throw new IllegalStateException("ref and alt alleles for MNP have different lengths");
        }
        ArrayList<VariantContext> result = new ArrayList<VariantContext>(ref.length);
        for (int i2 = 0; i2 < ref.length; ++i2) {
            if (ref[i2] == alt[i2]) continue;
            Allele newRefAllele = Allele.create(ref[i2], true);
            Allele newAltAllele = Allele.create(alt[i2], false);
            VariantContextBuilder newVC = new VariantContextBuilder(vc).start(vc.getStart() + i2).stop(vc.getStart() + i2).alleles((Collection<Allele>)Arrays.asList(newRefAllele, newAltAllele));
            HashMap<Allele, Allele> alleleMap = new HashMap<Allele, Allele>();
            alleleMap.put(vc.getReference(), newRefAllele);
            alleleMap.put(vc.getAlternateAllele(0), newAltAllele);
            GenotypesContext newGenotypes = GATKVariantContextUtils.updateGenotypesWithMappedAlleles(vc.getGenotypes(), new AlleleMapper(alleleMap));
            result.add(newVC.genotypes(newGenotypes).make());
        }
        if (result.isEmpty()) {
            result.add(vc);
        }
        return result;
    }

    public static boolean equalSites(VariantContext vc1, VariantContext vc2) {
        if (vc1 == null) {
            throw new IllegalArgumentException("vc1 cannot be null");
        }
        if (vc2 == null) {
            throw new IllegalArgumentException("vc2 cannot be null");
        }
        if (vc1.getStart() != vc2.getStart()) {
            return false;
        }
        if (vc1.getEnd() != vc2.getEnd()) {
            return false;
        }
        if (!vc1.getChr().equals(vc2.getChr())) {
            return false;
        }
        return vc1.getAlleles().equals(vc2.getAlleles());
    }

    public static int indexOfAllele(VariantContext vc, Allele allele, boolean ignoreRefState, boolean considerRefAllele, boolean useEquals) {
        if (allele == null) {
            throw new IllegalArgumentException();
        }
        return useEquals ? GATKVariantContextUtils.indexOfEqualAllele(vc, allele, ignoreRefState, considerRefAllele) : GATKVariantContextUtils.indexOfSameAllele(vc, allele, considerRefAllele);
    }

    public static int indexOfAltAllele(VariantContext vc, Allele allele, boolean useEquals) {
        int absoluteIndex = GATKVariantContextUtils.indexOfAllele(vc, allele, true, false, useEquals);
        return absoluteIndex == -1 ? -1 : absoluteIndex - 1;
    }

    private static int indexOfEqualAllele(VariantContext vc, Allele allele, boolean ignoreRefState, boolean considerRefAllele) {
        int i2 = 0;
        for (Allele a2 : vc.getAlleles()) {
            if (a2.equals(allele, ignoreRefState)) {
                return i2 == 0 ? (considerRefAllele ? 0 : -1) : i2;
            }
            ++i2;
        }
        return -1;
    }

    private static int indexOfSameAllele(VariantContext vc, Allele allele, boolean considerRefAllele) {
        int i2 = 0;
        for (Allele a2 : vc.getAlleles()) {
            if (a2 == allele) {
                return i2 == 0 ? (considerRefAllele ? 0 : -1) : i2;
            }
            ++i2;
        }
        return -1;
    }

    private static class CompareByPriority
    implements Comparator<VariantContext>,
    Serializable {
        List<String> priorityListOfVCs;

        public CompareByPriority(List<String> priorityListOfVCs) {
            this.priorityListOfVCs = priorityListOfVCs;
        }

        private int getIndex(VariantContext vc) {
            int i2 = this.priorityListOfVCs.indexOf(vc.getSource());
            if (i2 == -1) {
                throw new IllegalArgumentException("Priority list " + this.priorityListOfVCs + " doesn't contain variant context " + vc.getSource());
            }
            return i2;
        }

        @Override
        public int compare(VariantContext vc1, VariantContext vc2) {
            return Integer.valueOf(this.getIndex(vc1)).compareTo(this.getIndex(vc2));
        }
    }

    protected static class AlleleMapper {
        private VariantContext vc = null;
        private Map<Allele, Allele> map = null;

        public AlleleMapper(VariantContext vc) {
            this.vc = vc;
        }

        public AlleleMapper(Map<Allele, Allele> map) {
            this.map = map;
        }

        public boolean needsRemapping() {
            return this.map != null;
        }

        public Collection<Allele> values() {
            return this.map != null ? this.map.values() : this.vc.getAlleles();
        }

        public Allele remap(Allele a2) {
            return this.map != null && this.map.containsKey(a2) ? this.map.get(a2) : a2;
        }

        public List<Allele> remap(List<Allele> as) {
            ArrayList<Allele> newAs = new ArrayList<Allele>();
            for (Allele a2 : as) {
                newAs.add(this.remap(a2));
            }
            return newAs;
        }

        public List<Allele> getUniqueMappedAlleles() {
            if (this.map == null) {
                return Collections.emptyList();
            }
            return new ArrayList<Allele>(new HashSet<Allele>(this.map.values()));
        }
    }

    public static enum GenotypeAssignmentMethod {
        SET_TO_NO_CALL,
        USE_PLS_TO_ASSIGN,
        BEST_MATCH_TO_ORIGINAL,
        DO_NOT_ASSIGN_GENOTYPES;

    }

    public static enum MultipleAllelesMergeType {
        BY_TYPE,
        MIX_TYPES;

    }

    public static enum FilteredRecordMergeType {
        KEEP_IF_ANY_UNFILTERED,
        KEEP_IF_ALL_UNFILTERED,
        KEEP_UNCONDITIONAL;

    }

    public static enum GenotypeMergeType {
        UNIQUIFY,
        PRIORITIZE,
        UNSORTED,
        REQUIRE_UNIQUE;

    }
}

