/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.tools;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.reference.ReferenceSequenceFileFactory;
import htsjdk.tribble.AbstractFeatureReader;
import htsjdk.tribble.CloseableTribbleIterator;
import htsjdk.tribble.FeatureReader;
import htsjdk.tribble.index.IndexCreator;
import htsjdk.variant.bcf2.BCF2Codec;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.writer.Options;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterFactory;
import htsjdk.variant.vcf.VCFCodec;
import htsjdk.variant.vcf.VCFHeader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import java.util.PriorityQueue;
import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.broadinstitute.gatk.utils.collections.Pair;
import org.broadinstitute.gatk.utils.commandline.Argument;
import org.broadinstitute.gatk.utils.commandline.CommandLineProgram;
import org.broadinstitute.gatk.utils.commandline.Input;
import org.broadinstitute.gatk.utils.commandline.Output;
import org.broadinstitute.gatk.utils.exceptions.UserException;
import org.broadinstitute.gatk.utils.help.DocumentedGATKFeature;
import org.broadinstitute.gatk.utils.text.XReadLines;
import org.broadinstitute.gatk.utils.variant.GATKVCFIndexType;
import org.broadinstitute.gatk.utils.variant.GATKVCFUtils;

@DocumentedGATKFeature(groupName="Variant Evaluation and Manipulation Tools")
public class CatVariants
extends CommandLineProgram {
    private static Logger logger = Logger.getRootLogger();
    @Input(fullName="reference", shortName="R", doc="genome reference file <name>.fasta", required=true)
    private File refFile = null;
    @Input(fullName="variant", shortName="V", doc="Input VCF file/s", required=true)
    private List<File> variant = null;
    @Output(fullName="outputFile", shortName="out", doc="output file", required=true)
    private File outputFile = null;
    @Argument(fullName="assumeSorted", shortName="assumeSorted", doc="assumeSorted should be true if the input files are already sorted (based on the position of the variants)", required=false)
    private Boolean assumeSorted = false;
    @Argument(fullName="variant_index_type", doc="which type of IndexCreator to use for VCF/BCF indices", required=false)
    private GATKVCFIndexType variant_index_type = GATKVCFUtils.DEFAULT_INDEX_TYPE;
    @Argument(fullName="variant_index_parameter", doc="the parameter (bin width or features per bin) to pass to the VCF/BCF IndexCreator", required=false)
    private Integer variant_index_parameter = GATKVCFUtils.DEFAULT_INDEX_PARAMETER;

    private static void printUsage() {
        System.err.println("Usage: java -cp target/GenomeAnalysisTK.jar org.broadinstitute.gatk.tools.CatVariants --reference <reference> --variant <input VCF or BCF file; can specify --variant multiple times> --outputFile <outputFile> [--assumeSorted]");
        System.err.println("    The output file must be of the same type as all input files.");
        System.err.println("    If the input files are already sorted, then indicate that with --assumeSorted to improve performance.");
    }

    private FileType fileExtensionCheck(File inFile, File outFile) {
        String inFileName = inFile.toString().toLowerCase();
        String outFileName = outFile.toString().toLowerCase();
        FileType inFileType = FileType.INVALID;
        if (inFileName.endsWith(".vcf")) {
            inFileType = FileType.VCF;
            if (outFileName.endsWith(".vcf")) {
                return inFileType;
            }
        }
        if (inFileName.endsWith(".bcf")) {
            inFileType = FileType.BCF;
            if (outFileName.endsWith(".bcf")) {
                return inFileType;
            }
        }
        for (String extension : AbstractFeatureReader.BLOCK_COMPRESSED_EXTENSIONS) {
            if (!inFileName.endsWith(".vcf" + extension)) continue;
            inFileType = FileType.BLOCK_COMPRESSED_VCF;
            if (!outFileName.endsWith(".vcf" + extension)) continue;
            return inFileType;
        }
        if (inFileType == FileType.INVALID) {
            System.err.println(String.format("File extension for input file %s is not valid for CatVariants", inFile));
        } else {
            System.err.println(String.format("File extension mismatch between input %s and output %s", inFile, outFile));
        }
        CatVariants.printUsage();
        return FileType.INVALID;
    }

    private FeatureReader<VariantContext> getFeatureReader(FileType fileType, File file) {
        AbstractFeatureReader<VariantContext, Object> reader = null;
        switch (fileType) {
            case VCF: 
            case BLOCK_COMPRESSED_VCF: {
                reader = AbstractFeatureReader.getFeatureReader(file.getAbsolutePath(), new VCFCodec(), false);
                break;
            }
            case BCF: {
                reader = AbstractFeatureReader.getFeatureReader(file.getAbsolutePath(), new BCF2Codec(), false);
            }
        }
        return reader;
    }

    private List<File> parseVariantList(List<File> rawFileList) {
        ArrayList<File> result = new ArrayList<File>(rawFileList.size());
        for (File rawFile : rawFileList) {
            if (rawFile.getName().endsWith(".list")) {
                try {
                    for (String line : new XReadLines(rawFile, true)) {
                        result.add(new File(line));
                    }
                    continue;
                }
                catch (IOException e2) {
                    throw new UserException.CouldNotReadInputFile(rawFile, (Exception)e2);
                }
            }
            result.add(rawFile);
        }
        return result;
    }

    @Override
    protected int execute() throws Exception {
        ReferenceSequenceFile ref;
        BasicConfigurator.configure();
        logger.setLevel(Level.INFO);
        try {
            ref = ReferenceSequenceFileFactory.getReferenceSequenceFile(this.refFile);
        }
        catch (Exception e2) {
            throw new UserException("Couldn't load provided reference sequence file " + this.refFile, e2);
        }
        this.variant = this.parseVariantList(this.variant);
        PositionComparator positionComparator = new PositionComparator();
        AbstractCollection priorityQueue = this.assumeSorted != false ? new LinkedList() : new PriorityQueue<Pair<Integer, File>>(10000, positionComparator);
        FileType fileType = FileType.INVALID;
        for (File file : this.variant) {
            fileType = this.fileExtensionCheck(file, this.outputFile);
            if (fileType == FileType.INVALID) {
                return 1;
            }
            if (this.assumeSorted.booleanValue()) {
                priorityQueue.add(new Pair<Integer, File>(0, file));
                continue;
            }
            if (!file.exists()) {
                throw new UserException(String.format("File %s doesn't exist", file.getAbsolutePath()));
            }
            FeatureReader<VariantContext> reader = this.getFeatureReader(fileType, file);
            CloseableTribbleIterator<VariantContext> it = reader.iterator();
            if (!it.hasNext()) {
                System.err.println(String.format("File %s is empty. This file will be ignored", file.getAbsolutePath()));
                continue;
            }
            VariantContext vc = (VariantContext)it.next();
            int firstPosition = vc.getStart();
            reader.close();
            priorityQueue.add(new Pair<Integer, File>(firstPosition, file));
        }
        FileOutputStream outputStream = new FileOutputStream(this.outputFile);
        EnumSet<Options> options = EnumSet.of(Options.INDEX_ON_THE_FLY);
        IndexCreator idxCreator = GATKVCFUtils.getIndexCreator(this.variant_index_type, this.variant_index_parameter, this.outputFile, ref.getSequenceDictionary());
        VariantContextWriter outputWriter = VariantContextWriterFactory.create((File)this.outputFile, (OutputStream)outputStream, (SAMSequenceDictionary)ref.getSequenceDictionary(), (IndexCreator)idxCreator, options);
        boolean firstFile = true;
        int count = 0;
        while (!priorityQueue.isEmpty()) {
            ++count;
            File file = (File)((Pair)priorityQueue.remove()).getSecond();
            if (!file.exists()) {
                throw new UserException(String.format("File %s doesn't exist", file.getAbsolutePath()));
            }
            FeatureReader<VariantContext> reader = this.getFeatureReader(fileType, file);
            if (count % 10 == 0) {
                System.out.print(count);
            } else {
                System.out.print(".");
            }
            if (firstFile) {
                VCFHeader header = (VCFHeader)reader.getHeader();
                outputWriter.writeHeader(header);
                firstFile = false;
            }
            CloseableTribbleIterator<VariantContext> it = reader.iterator();
            while (it.hasNext()) {
                VariantContext vc = (VariantContext)it.next();
                outputWriter.add(vc);
            }
            reader.close();
        }
        System.out.println();
        outputWriter.close();
        return 0;
    }

    public static void main(String[] args) {
        try {
            CatVariants instance = new CatVariants();
            CatVariants.start(instance, args);
            System.exit(CommandLineProgram.result);
        }
        catch (UserException e2) {
            CatVariants.printUsage();
            CatVariants.exitSystemWithUserError(e2);
        }
        catch (Exception e3) {
            CatVariants.exitSystemWithError(e3);
        }
    }

    private static class PositionComparator
    implements Comparator<Pair<Integer, File>> {
        private PositionComparator() {
        }

        @Override
        public int compare(Pair<Integer, File> p1, Pair<Integer, File> p2) {
            int startPositionP2;
            int startPositionP1 = p1.getFirst();
            if (startPositionP1 == (startPositionP2 = p2.getFirst().intValue())) {
                return 0;
            }
            return startPositionP1 < startPositionP2 ? -1 : 1;
        }
    }

    private static enum FileType {
        VCF,
        BCF,
        BLOCK_COMPRESSED_VCF,
        INVALID;

    }
}

