/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.utils.recalibration;

import com.google.java.contract.Ensures;
import com.google.java.contract.Invariant;
import com.google.java.contract.Requires;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import org.apache.log4j.Logger;
import org.broadinstitute.gatk.engine.report.GATKReport;
import org.broadinstitute.gatk.engine.report.GATKReportTable;
import org.broadinstitute.gatk.utils.QualityUtils;
import org.broadinstitute.gatk.utils.Utils;
import org.broadinstitute.gatk.utils.exceptions.ReviewedGATKException;

public class QualQuantizer {
    private static final Set<QualInterval> MY_EMPTY_SET = Collections.emptySet();
    private static Logger logger = Logger.getLogger(QualQuantizer.class);
    final int nLevels;
    final int minInterestingQual;
    final List<Long> nObservationsPerQual;
    final List<Byte> originalToQuantizedMap;
    final TreeSet<QualInterval> quantizedIntervals;

    protected QualQuantizer(int minInterestingQual) {
        this.nObservationsPerQual = Collections.emptyList();
        this.nLevels = 0;
        this.minInterestingQual = minInterestingQual;
        this.quantizedIntervals = null;
        this.originalToQuantizedMap = null;
    }

    public QualQuantizer(List<Long> nObservationsPerQual, int nLevels, int minInterestingQual) {
        this.nObservationsPerQual = nObservationsPerQual;
        this.nLevels = nLevels;
        this.minInterestingQual = minInterestingQual;
        if (Collections.min(nObservationsPerQual) < 0L) {
            throw new ReviewedGATKException("Quality score histogram has negative values at: " + Utils.join(", ", nObservationsPerQual));
        }
        if (nLevels < 0) {
            throw new ReviewedGATKException("nLevels must be >= 0");
        }
        if (minInterestingQual < 0) {
            throw new ReviewedGATKException("minInterestingQual must be >= 0");
        }
        this.quantizedIntervals = this.quantize();
        this.originalToQuantizedMap = this.intervalsToMap(this.quantizedIntervals);
    }

    @Ensures(value={"! result.isEmpty()", "result.size() == nLevels"})
    private TreeSet<QualInterval> quantize() {
        TreeSet<QualInterval> intervals = new TreeSet<QualInterval>();
        for (int qStart = 0; qStart < this.getNQualsInHistogram(); ++qStart) {
            long nObs = this.nObservationsPerQual.get(qStart);
            double errorRate = QualityUtils.qualToErrorProb((byte)qStart);
            double nErrors = (double)nObs * errorRate;
            QualInterval qi = new QualInterval(qStart, qStart, nObs, (long)((int)Math.floor(nErrors)), 0, (byte)qStart);
            intervals.add(qi);
        }
        while (intervals.size() > this.nLevels) {
            this.mergeLowestPenaltyIntervals(intervals);
        }
        return intervals;
    }

    @Requires(value={"! intervals.isEmpty()"})
    private void mergeLowestPenaltyIntervals(TreeSet<QualInterval> intervals) {
        Iterator<QualInterval> it1 = intervals.iterator();
        Iterator<QualInterval> it1p = intervals.iterator();
        it1p.next();
        QualInterval minMerge = null;
        if (logger.isDebugEnabled()) {
            logger.debug("mergeLowestPenaltyIntervals: " + intervals.size());
        }
        int lastMergeOrder = 0;
        while (it1p.hasNext()) {
            QualInterval left = it1.next();
            QualInterval right = it1p.next();
            QualInterval merged = left.merge(right);
            lastMergeOrder = Math.max(Math.max(lastMergeOrder, left.mergeOrder), right.mergeOrder);
            if (minMerge != null && !(merged.getPenalty() < minMerge.getPenalty())) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("  Updating merge " + minMerge);
            }
            minMerge = merged;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("  => final min merge " + minMerge);
        }
        intervals.removeAll(minMerge.subIntervals);
        intervals.add(minMerge);
        minMerge.mergeOrder = lastMergeOrder + 1;
        if (logger.isDebugEnabled()) {
            logger.debug("updated intervals: " + intervals);
        }
    }

    @Ensures(value={"result.size() == getNQualsInHistogram()"})
    private List<Byte> intervalsToMap(TreeSet<QualInterval> intervals) {
        ArrayList<Byte> map = new ArrayList<Byte>(this.getNQualsInHistogram());
        map.addAll(Collections.nCopies(this.getNQualsInHistogram(), (byte)-128));
        for (QualInterval interval : intervals) {
            for (int q2 = interval.qStart; q2 <= interval.qEnd; ++q2) {
                map.set(q2, interval.getQual());
            }
        }
        if ((Byte)Collections.min(map) == -128) {
            throw new ReviewedGATKException("quantized quality score map contains an un-initialized value");
        }
        return map;
    }

    @Ensures(value={"result > 0"})
    private final int getNQualsInHistogram() {
        return this.nObservationsPerQual.size();
    }

    public void writeReport(PrintStream out) {
        GATKReport report = new GATKReport();
        this.addQualHistogramToReport(report);
        this.addIntervalsToReport(report);
        report.print(out);
    }

    private final void addQualHistogramToReport(GATKReport report) {
        report.addTable("QualHistogram", "Quality score histogram provided to report", 2);
        GATKReportTable table = report.getTable("QualHistogram");
        table.addColumn("qual");
        table.addColumn("count");
        for (int q2 = 0; q2 < this.nObservationsPerQual.size(); ++q2) {
            table.set((Object)q2, "qual", (Object)q2);
            table.set((Object)q2, "count", (Object)this.nObservationsPerQual.get(q2));
        }
    }

    private final void addIntervalsToReport(GATKReport report) {
        report.addTable("QualQuantizerIntervals", "Table of QualQuantizer quantization intervals", 10);
        GATKReportTable table = report.getTable("QualQuantizerIntervals");
        table.addColumn("name");
        table.addColumn("qStart");
        table.addColumn("qEnd");
        table.addColumn("level");
        table.addColumn("merge.order");
        table.addColumn("nErrors");
        table.addColumn("nObservations");
        table.addColumn("qual");
        table.addColumn("penalty");
        table.addColumn("root.node");
        for (QualInterval interval : this.quantizedIntervals) {
            this.addIntervalToReport(table, interval, true);
        }
    }

    private final void addIntervalToReport(GATKReportTable table, QualInterval interval, boolean atRootP) {
        String name = interval.getName();
        table.set((Object)name, "name", (Object)name);
        table.set((Object)name, "qStart", (Object)interval.qStart);
        table.set((Object)name, "qEnd", (Object)interval.qEnd);
        table.set((Object)name, "level", (Object)interval.level);
        table.set((Object)name, "merge.order", (Object)interval.mergeOrder);
        table.set((Object)name, "nErrors", (Object)interval.nErrors);
        table.set((Object)name, "nObservations", (Object)interval.nObservations);
        table.set((Object)name, "qual", (Object)interval.getQual());
        table.set((Object)name, "penalty", (Object)String.format("%.1f", interval.getPenalty()));
        table.set((Object)name, "root.node", (Object)atRootP);
        for (QualInterval sub : interval.subIntervals) {
            this.addIntervalToReport(table, sub, false);
        }
    }

    public List<Byte> getOriginalToQuantizedMap() {
        return this.originalToQuantizedMap;
    }

    @Invariant(value={"qStart <= qEnd", "qStart >= 0", "qEnd <= 1000", "nObservations >= 0", "nErrors >= 0", "nErrors <= nObservations", "fixedQual >= -1 && fixedQual <= QualityUtils.MAX_SAM_QUAL_SCORE", "mergeOrder >= 0"})
    protected final class QualInterval
    implements Comparable<QualInterval> {
        final int qStart;
        final int qEnd;
        final int fixedQual;
        final int level;
        final long nObservations;
        final long nErrors;
        final Set<QualInterval> subIntervals;
        int mergeOrder;

        protected QualInterval(int qStart, int qEnd, long nObservations, long nErrors, int level) {
            this(qStart, qEnd, nObservations, nErrors, level, -1, MY_EMPTY_SET);
        }

        protected QualInterval(int qStart, int qEnd, long nObservations, long nErrors, int level, Set<QualInterval> subIntervals) {
            this(qStart, qEnd, nObservations, nErrors, level, -1, subIntervals);
        }

        protected QualInterval(int qStart, int qEnd, long nObservations, long nErrors, int level, int fixedQual) {
            this(qStart, qEnd, nObservations, nErrors, level, fixedQual, MY_EMPTY_SET);
        }

        @Requires(value={"level >= 0"})
        public QualInterval(int qStart, int qEnd, long nObservations, long nErrors, int level, int fixedQual, Set<QualInterval> subIntervals) {
            this.qStart = qStart;
            this.qEnd = qEnd;
            this.nObservations = nObservations;
            this.nErrors = nErrors;
            this.fixedQual = fixedQual;
            this.level = level;
            this.mergeOrder = 0;
            this.subIntervals = Collections.unmodifiableSet(subIntervals);
        }

        public String getName() {
            return this.qStart + "-" + this.qEnd;
        }

        public String toString() {
            return "QQ:" + this.getName();
        }

        @Ensures(value={"result >= 0.0"})
        public double getErrorRate() {
            if (this.hasFixedQual()) {
                return QualityUtils.qualToErrorProb((byte)this.fixedQual);
            }
            if (this.nObservations == 0L) {
                return 0.0;
            }
            return (double)(this.nErrors + 1L) / (1.0 * (double)(this.nObservations + 1L));
        }

        @Ensures(value={"result >= 0 && result <= QualityUtils.MAX_SAM_QUAL_SCORE"})
        public byte getQual() {
            if (!this.hasFixedQual()) {
                return QualityUtils.errorProbToQual(this.getErrorRate());
            }
            return (byte)this.fixedQual;
        }

        public boolean hasFixedQual() {
            return this.fixedQual != -1;
        }

        @Override
        public int compareTo(QualInterval qualInterval) {
            return Integer.valueOf(this.qStart).compareTo(qualInterval.qStart);
        }

        @Requires(value={"toMerge != null"})
        @Ensures(value={"result != null", "result.nObservations >= this.nObservations", "result.nObservations >= toMerge.nObservations", "result.nErrors >= this.nErrors", "result.nErrors >= toMerge.nErrors", "result.qStart == Math.min(this.qStart, toMerge.qStart)", "result.qEnd == Math.max(this.qEnd, toMerge.qEnd)", "result.level > Math.max(this.level, toMerge.level)", "result.subIntervals.size() == 2"})
        public QualInterval merge(QualInterval toMerge) {
            QualInterval right;
            QualInterval left = this.compareTo(toMerge) < 0 ? this : toMerge;
            QualInterval qualInterval = right = this.compareTo(toMerge) < 0 ? toMerge : this;
            if (left.qEnd + 1 != right.qStart) {
                throw new ReviewedGATKException("Attempting to merge non-contiguous intervals: left = " + left + " right = " + right);
            }
            long nCombinedObs = left.nObservations + right.nObservations;
            long nCombinedErr = left.nErrors + right.nErrors;
            int level = Math.max(left.level, right.level) + 1;
            HashSet<QualInterval> subIntervals = new HashSet<QualInterval>(Arrays.asList(left, right));
            QualInterval merged = new QualInterval(left.qStart, right.qEnd, nCombinedObs, nCombinedErr, level, subIntervals);
            return merged;
        }

        public double getPenalty() {
            return this.calcPenalty(this.getErrorRate());
        }

        @Requires(value={"globalErrorRate >= 0.0"})
        @Ensures(value={"result >= 0.0"})
        private double calcPenalty(double globalErrorRate) {
            if (globalErrorRate == 0.0) {
                return 0.0;
            }
            if (this.subIntervals.isEmpty()) {
                if (this.qEnd <= QualQuantizer.this.minInterestingQual) {
                    return 0.0;
                }
                return Math.abs(Math.log10(this.getErrorRate()) - Math.log10(globalErrorRate)) * (double)this.nObservations;
            }
            double sum = 0.0;
            for (QualInterval interval : this.subIntervals) {
                sum += interval.calcPenalty(globalErrorRate);
            }
            return sum;
        }
    }
}

