/*
 * Decompiled with CFR 0.152.
 */
package picard.util;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMProgramRecord;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.Log;
import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import picard.PicardException;
import picard.cmdline.CommandLineParser;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.Intervals;
import picard.util.IntervalListScatterer;

@CommandLineProgramProperties(usage=" General tool for manipulating interval lists, including sorting, merging, padding, uniqueifying, and other set-theoretic operations. Default operation if given one or more inputs is to merge and sort them.  Other options are controlled by arguments.", usageShort="General tool for manipulating interval lists", programGroup=Intervals.class)
public class IntervalListTools
extends CommandLineProgram {
    @Option(shortName="I", doc="One or more interval lists. If multiple interval lists are provided the output is theresult of merging the inputs.", minElements=1)
    public List<File> INPUT;
    @Option(doc="The output interval list file to write (if SCATTER_COUNT is 1) or the directory into which to write the scattered interval sub-directories (if SCATTER_COUNT > 1)", shortName="O", optional=true)
    public File OUTPUT;
    @Option(doc="The amount to pad each end of the intervals by before other operations are undertaken. Negative numbers are allowed and indicate intervals should be shrunk. Resulting intervals < 0 bases long will be removed. Padding is applied to the interval lists <b> before </b> the ACTION is performed.", optional=true)
    public int PADDING = 0;
    @Option(doc="If true, merge overlapping and adjacent intervals to create a list of unique intervals. Implies SORT=true")
    public boolean UNIQUE = false;
    @Option(doc="If true, sort the resulting interval list by coordinate.")
    public boolean SORT = true;
    @Option(doc="Action to take on inputs.")
    public Action ACTION = Action.CONCAT;
    @Option(shortName="SI", doc="Second set of intervals for SUBTRACT and DIFFERENCE operations.", optional=true)
    public List<File> SECOND_INPUT;
    @Option(doc="One or more lines of comment to add to the header of the output file.", optional=true)
    public List<String> COMMENT = null;
    @Option(doc="The number of files into which to scatter the resulting list by locus; in some situations, fewer intervals may be emitted.")
    public int SCATTER_COUNT = 1;
    @Option(shortName="M", doc="Do not subdivide ")
    public IntervalListScatterer.Mode SUBDIVISION_MODE = IntervalListScatterer.Mode.INTERVAL_SUBDIVISION;
    @Option(doc="Produce the inverse list", optional=true)
    public boolean INVERT = false;
    private static final Log LOG = Log.getInstance(IntervalListTools.class);

    public static void main(String[] args) {
        new IntervalListTools().instanceMainWithExit(args);
    }

    @Override
    protected int doWork() {
        List<IntervalList> resultIntervals;
        for (File f2 : this.INPUT) {
            IOUtil.assertFileIsReadable(f2);
        }
        for (File f2 : this.SECOND_INPUT) {
            IOUtil.assertFileIsReadable(f2);
        }
        if (this.OUTPUT != null) {
            if (this.SCATTER_COUNT == 1) {
                IOUtil.assertFileIsWritable(this.OUTPUT);
            } else {
                IOUtil.assertDirectoryIsWritable(this.OUTPUT);
            }
        }
        ArrayList<IntervalList> lists = new ArrayList<IntervalList>();
        for (File f3 : this.INPUT) {
            IntervalList list = IntervalList.fromFile(f3);
            if (this.PADDING != 0) {
                IntervalList out = new IntervalList(list.getHeader());
                for (Interval i2 : list) {
                    int end;
                    int start = i2.getStart() - this.PADDING;
                    if (start > (end = i2.getEnd() + this.PADDING)) continue;
                    Interval i22 = new Interval(i2.getSequence(), start, end, i2.isNegativeStrand(), i2.getName());
                    out.add(i22);
                }
                lists.add(out);
                continue;
            }
            lists.add(list);
        }
        ArrayList<IntervalList> secondLists = new ArrayList<IntervalList>();
        for (File f4 : this.SECOND_INPUT) {
            IntervalList list = IntervalList.fromFile(f4);
            if (this.PADDING != 0) {
                IntervalList out = new IntervalList(list.getHeader());
                for (Interval i3 : list) {
                    int end;
                    int start = i3.getStart() - this.PADDING;
                    if (start > (end = i3.getEnd() + this.PADDING)) continue;
                    Interval i2 = new Interval(i3.getSequence(), start, end, i3.isNegativeStrand(), i3.getName());
                    out.add(i2);
                }
                secondLists.add(out);
                continue;
            }
            secondLists.add(list);
        }
        if (this.UNIQUE && !this.SORT) {
            LOG.warn("UNIQUE=true requires sorting but SORT=false was specified.  Results will be sorted!");
        }
        IntervalList result = this.ACTION.act(lists, secondLists);
        if (this.INVERT) {
            this.SORT = false;
            this.UNIQUE = false;
        }
        IntervalList possiblySortedResult = this.SORT ? result.sorted() : result;
        IntervalList possiblyInvertedResult = this.INVERT ? IntervalList.invert(possiblySortedResult) : possiblySortedResult;
        List<Interval> finalIntervals = this.UNIQUE ? possiblyInvertedResult.uniqued().getIntervals() : possiblyInvertedResult.getIntervals();
        SAMFileHeader header = result.getHeader();
        HashSet<String> pgs = new HashSet<String>();
        for (SAMProgramRecord pg : header.getProgramRecords()) {
            pgs.add(pg.getId());
        }
        for (int i4 = 1; i4 < Integer.MAX_VALUE; ++i4) {
            if (pgs.contains(String.valueOf(i4))) continue;
            SAMProgramRecord pg = new SAMProgramRecord(String.valueOf(i4));
            pg.setCommandLine(this.getCommandLine());
            pg.setProgramName(this.getClass().getSimpleName());
            header.addProgramRecord(pg);
            break;
        }
        if (this.COMMENT != null) {
            for (String comment : this.COMMENT) {
                header.addComment(comment);
            }
        }
        IntervalList output = new IntervalList(header);
        for (Interval i5 : finalIntervals) {
            output.add(i5);
        }
        if (this.OUTPUT != null) {
            if (this.SCATTER_COUNT == 1) {
                output.write(this.OUTPUT);
                resultIntervals = Arrays.asList(output);
            } else {
                List<IntervalList> scattered = this.writeScatterIntervals(output);
                LOG.info(String.format("Wrote %s scatter subdirectories to %s.", scattered.size(), this.OUTPUT));
                if (scattered.size() != this.SCATTER_COUNT) {
                    LOG.warn(String.format("Requested scatter width of %s, but only emitted %s.  (This may be an expected consequence of running in %s mode.)", new Object[]{this.SCATTER_COUNT, scattered.size(), IntervalListScatterer.Mode.BALANCING_WITHOUT_INTERVAL_SUBDIVISION}));
                }
                resultIntervals = scattered;
            }
        } else {
            resultIntervals = Arrays.asList(output);
        }
        long totalUniqueBaseCount = 0L;
        long intervalCount = 0L;
        for (IntervalList finalInterval : resultIntervals) {
            totalUniqueBaseCount = finalInterval.getUniqueBaseCount();
            intervalCount += (long)finalInterval.size();
        }
        LOG.info("Produced " + intervalCount + " intervals totalling " + totalUniqueBaseCount + " unique bases.");
        return 0;
    }

    @Override
    protected String[] customCommandLineValidation() {
        if (this.SCATTER_COUNT < 1) {
            return new String[]{"SCATTER_COUNT must be greater than 0."};
        }
        return null;
    }

    private List<IntervalList> writeScatterIntervals(IntervalList list) {
        IntervalListScatterer scatterer = new IntervalListScatterer(this.SUBDIVISION_MODE);
        List<IntervalList> scattered = scatterer.scatter(list, this.SCATTER_COUNT);
        DecimalFormat fileNameFormatter = new DecimalFormat("0000");
        int fileIndex = 1;
        for (IntervalList intervals : scattered) {
            intervals.write(IntervalListTools.createDirectoryAndGetScatterFile(this.OUTPUT, scattered.size(), fileNameFormatter.format(fileIndex++)));
        }
        return scattered;
    }

    public static File getScatteredFileName(File scatterDirectory, long scatterTotal, String formattedIndex) {
        return new File(scatterDirectory.getAbsolutePath() + "/temp_" + formattedIndex + "_of_" + scatterTotal + "/scattered.intervals");
    }

    private static File createDirectoryAndGetScatterFile(File outputDirectory, long scatterCount, String formattedIndex) {
        IntervalListTools.createDirectoryOrFail(outputDirectory);
        File result = IntervalListTools.getScatteredFileName(outputDirectory, scatterCount, formattedIndex);
        IntervalListTools.createDirectoryOrFail(result.getParentFile());
        return result;
    }

    private static void createDirectoryOrFail(File directory) {
        if (!directory.exists() && !directory.mkdir()) {
            throw new PicardException("Unable to create directory: " + directory.getAbsolutePath());
        }
    }

    public static enum Action implements CommandLineParser.ClpEnum
    {
        CONCAT("The concatenation of all the INPUTs, no sorting or merging of overlapping/abutting intervals implied. Will result in an unsorted list unless requested otherwise."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.concatenate(list);
            }
        }
        ,
        UNION("Like CONCATENATE but with UNIQUE and SORT implied, the result being the set-wise union of all INPUTS."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.union(list);
            }
        }
        ,
        INTERSECT("The sorted, uniqued set of all loci that are contained in all of the INPUTs."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.intersection(list);
            }
        }
        ,
        SUBTRACT("Subtracts SECOND_INPUT from INPUT. The resulting loci are there in INPUT that are not in SECOND_INPUT"){

            @Override
            IntervalList act(List<IntervalList> list1, List<IntervalList> list2) {
                return IntervalList.subtract(list1, list2);
            }
        }
        ,
        SYMDIFF("Find loci that are in INPUT or SECOND_INPUT but are not in both."){

            @Override
            IntervalList act(List<IntervalList> list1, List<IntervalList> list2) {
                return IntervalList.difference(list1, list2);
            }
        };

        String helpdoc;

        private Action(String helpdoc) {
            this.helpdoc = helpdoc;
        }

        @Override
        public String getHelpDoc() {
            return this.helpdoc;
        }

        abstract IntervalList act(List<IntervalList> var1, List<IntervalList> var2);
    }
}

