/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.linear;

import org.apache.commons.math.MathRuntimeException;
import org.apache.commons.math.exception.util.LocalizedFormats;
import org.apache.commons.math.linear.Array2DRowRealMatrix;
import org.apache.commons.math.linear.ArrayRealVector;
import org.apache.commons.math.linear.DecompositionSolver;
import org.apache.commons.math.linear.InvalidMatrixException;
import org.apache.commons.math.linear.LUDecomposition;
import org.apache.commons.math.linear.MatrixUtils;
import org.apache.commons.math.linear.NonSquareMatrixException;
import org.apache.commons.math.linear.RealMatrix;
import org.apache.commons.math.linear.RealVector;
import org.apache.commons.math.linear.SingularMatrixException;
import org.apache.commons.math.util.FastMath;

public class LUDecompositionImpl
implements LUDecomposition {
    private static final double DEFAULT_TOO_SMALL = 1.0E-11;
    private double[][] lu;
    private int[] pivot;
    private boolean even;
    private boolean singular;
    private RealMatrix cachedL;
    private RealMatrix cachedU;
    private RealMatrix cachedP;

    public LUDecompositionImpl(RealMatrix matrix) throws InvalidMatrixException {
        this(matrix, 1.0E-11);
    }

    public LUDecompositionImpl(RealMatrix matrix, double singularityThreshold) throws NonSquareMatrixException {
        if (!matrix.isSquare()) {
            throw new NonSquareMatrixException(matrix.getRowDimension(), matrix.getColumnDimension());
        }
        int m2 = matrix.getColumnDimension();
        this.lu = matrix.getData();
        this.pivot = new int[m2];
        this.cachedL = null;
        this.cachedU = null;
        this.cachedP = null;
        for (int row = 0; row < m2; ++row) {
            this.pivot[row] = row;
        }
        this.even = true;
        this.singular = false;
        for (int col = 0; col < m2; ++col) {
            double sum = 0.0;
            for (int row = 0; row < col; ++row) {
                double[] luRow = this.lu[row];
                sum = luRow[col];
                for (int i2 = 0; i2 < row; ++i2) {
                    sum -= luRow[i2] * this.lu[i2][col];
                }
                luRow[col] = sum;
            }
            int max = col;
            double largest = Double.NEGATIVE_INFINITY;
            for (int row = col; row < m2; ++row) {
                double[] luRow = this.lu[row];
                sum = luRow[col];
                for (int i3 = 0; i3 < col; ++i3) {
                    sum -= luRow[i3] * this.lu[i3][col];
                }
                luRow[col] = sum;
                if (!(FastMath.abs(sum) > largest)) continue;
                largest = FastMath.abs(sum);
                max = row;
            }
            if (FastMath.abs(this.lu[max][col]) < singularityThreshold) {
                this.singular = true;
                return;
            }
            if (max != col) {
                double tmp = 0.0;
                double[] luMax = this.lu[max];
                double[] luCol = this.lu[col];
                for (int i4 = 0; i4 < m2; ++i4) {
                    tmp = luMax[i4];
                    luMax[i4] = luCol[i4];
                    luCol[i4] = tmp;
                }
                int temp = this.pivot[max];
                this.pivot[max] = this.pivot[col];
                this.pivot[col] = temp;
                this.even = !this.even;
            }
            double luDiag = this.lu[col][col];
            for (int row = col + 1; row < m2; ++row) {
                double[] dArray = this.lu[row];
                int n2 = col;
                dArray[n2] = dArray[n2] / luDiag;
            }
        }
    }

    public RealMatrix getL() {
        if (this.cachedL == null && !this.singular) {
            int m2 = this.pivot.length;
            this.cachedL = MatrixUtils.createRealMatrix(m2, m2);
            for (int i2 = 0; i2 < m2; ++i2) {
                double[] luI = this.lu[i2];
                for (int j2 = 0; j2 < i2; ++j2) {
                    this.cachedL.setEntry(i2, j2, luI[j2]);
                }
                this.cachedL.setEntry(i2, i2, 1.0);
            }
        }
        return this.cachedL;
    }

    public RealMatrix getU() {
        if (this.cachedU == null && !this.singular) {
            int m2 = this.pivot.length;
            this.cachedU = MatrixUtils.createRealMatrix(m2, m2);
            for (int i2 = 0; i2 < m2; ++i2) {
                double[] luI = this.lu[i2];
                for (int j2 = i2; j2 < m2; ++j2) {
                    this.cachedU.setEntry(i2, j2, luI[j2]);
                }
            }
        }
        return this.cachedU;
    }

    public RealMatrix getP() {
        if (this.cachedP == null && !this.singular) {
            int m2 = this.pivot.length;
            this.cachedP = MatrixUtils.createRealMatrix(m2, m2);
            for (int i2 = 0; i2 < m2; ++i2) {
                this.cachedP.setEntry(i2, this.pivot[i2], 1.0);
            }
        }
        return this.cachedP;
    }

    public int[] getPivot() {
        return (int[])this.pivot.clone();
    }

    public double getDeterminant() {
        if (this.singular) {
            return 0.0;
        }
        int m2 = this.pivot.length;
        double determinant = this.even ? 1.0 : -1.0;
        for (int i2 = 0; i2 < m2; ++i2) {
            determinant *= this.lu[i2][i2];
        }
        return determinant;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.lu, this.pivot, this.singular);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] lu;
        private final int[] pivot;
        private final boolean singular;

        private Solver(double[][] lu, int[] pivot, boolean singular) {
            this.lu = lu;
            this.pivot = pivot;
            this.singular = singular;
        }

        public boolean isNonSingular() {
            return !this.singular;
        }

        public double[] solve(double[] b2) throws IllegalArgumentException, InvalidMatrixException {
            int i2;
            double bpCol;
            int col;
            int m2 = this.pivot.length;
            if (b2.length != m2) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.VECTOR_LENGTH_MISMATCH, b2.length, m2);
            }
            if (this.singular) {
                throw new SingularMatrixException();
            }
            double[] bp = new double[m2];
            for (int row = 0; row < m2; ++row) {
                bp[row] = b2[this.pivot[row]];
            }
            for (col = 0; col < m2; ++col) {
                bpCol = bp[col];
                for (i2 = col + 1; i2 < m2; ++i2) {
                    int n2 = i2;
                    bp[n2] = bp[n2] - bpCol * this.lu[i2][col];
                }
            }
            for (col = m2 - 1; col >= 0; --col) {
                int n3 = col;
                bp[n3] = bp[n3] / this.lu[col][col];
                bpCol = bp[col];
                for (i2 = 0; i2 < col; ++i2) {
                    int n4 = i2;
                    bp[n4] = bp[n4] - bpCol * this.lu[i2][col];
                }
            }
            return bp;
        }

        public RealVector solve(RealVector b2) throws IllegalArgumentException, InvalidMatrixException {
            try {
                return this.solve((ArrayRealVector)b2);
            }
            catch (ClassCastException cce) {
                int i2;
                double bpCol;
                int col;
                int m2 = this.pivot.length;
                if (b2.getDimension() != m2) {
                    throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.VECTOR_LENGTH_MISMATCH, b2.getDimension(), m2);
                }
                if (this.singular) {
                    throw new SingularMatrixException();
                }
                double[] bp = new double[m2];
                for (int row = 0; row < m2; ++row) {
                    bp[row] = b2.getEntry(this.pivot[row]);
                }
                for (col = 0; col < m2; ++col) {
                    bpCol = bp[col];
                    for (i2 = col + 1; i2 < m2; ++i2) {
                        int n2 = i2;
                        bp[n2] = bp[n2] - bpCol * this.lu[i2][col];
                    }
                }
                for (col = m2 - 1; col >= 0; --col) {
                    int n3 = col;
                    bp[n3] = bp[n3] / this.lu[col][col];
                    bpCol = bp[col];
                    for (i2 = 0; i2 < col; ++i2) {
                        int n4 = i2;
                        bp[n4] = bp[n4] - bpCol * this.lu[i2][col];
                    }
                }
                return new ArrayRealVector(bp, false);
            }
        }

        public ArrayRealVector solve(ArrayRealVector b2) throws IllegalArgumentException, InvalidMatrixException {
            return new ArrayRealVector(this.solve(b2.getDataRef()), false);
        }

        public RealMatrix solve(RealMatrix b2) throws IllegalArgumentException, InvalidMatrixException {
            double[] bpCol;
            int col;
            int m2 = this.pivot.length;
            if (b2.getRowDimension() != m2) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.DIMENSIONS_MISMATCH_2x2, b2.getRowDimension(), b2.getColumnDimension(), m2, "n");
            }
            if (this.singular) {
                throw new SingularMatrixException();
            }
            int nColB = b2.getColumnDimension();
            double[][] bp = new double[m2][nColB];
            for (int row = 0; row < m2; ++row) {
                double[] bpRow = bp[row];
                int pRow = this.pivot[row];
                for (int col2 = 0; col2 < nColB; ++col2) {
                    bpRow[col2] = b2.getEntry(pRow, col2);
                }
            }
            for (col = 0; col < m2; ++col) {
                bpCol = bp[col];
                for (int i2 = col + 1; i2 < m2; ++i2) {
                    double[] bpI = bp[i2];
                    double luICol = this.lu[i2][col];
                    for (int j2 = 0; j2 < nColB; ++j2) {
                        int n2 = j2;
                        bpI[n2] = bpI[n2] - bpCol[j2] * luICol;
                    }
                }
            }
            for (col = m2 - 1; col >= 0; --col) {
                bpCol = bp[col];
                double luDiag = this.lu[col][col];
                int j3 = 0;
                while (j3 < nColB) {
                    int n3 = j3++;
                    bpCol[n3] = bpCol[n3] / luDiag;
                }
                for (int i3 = 0; i3 < col; ++i3) {
                    double[] bpI = bp[i3];
                    double luICol = this.lu[i3][col];
                    for (int j4 = 0; j4 < nColB; ++j4) {
                        int n4 = j4;
                        bpI[n4] = bpI[n4] - bpCol[j4] * luICol;
                    }
                }
            }
            return new Array2DRowRealMatrix(bp, false);
        }

        public RealMatrix getInverse() throws InvalidMatrixException {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.pivot.length));
        }
    }
}

