/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.sam;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import net.sf.samtools.Cigar;
import net.sf.samtools.CigarElement;
import net.sf.samtools.CigarOperator;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileWriter;
import net.sf.samtools.SAMFileWriterFactory;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMRecordCoordinateComparator;
import net.sf.samtools.SAMSequenceDictionary;
import net.sf.samtools.SAMSequenceRecord;
import org.apache.log4j.Logger;
import org.broadinstitute.sting.gatk.GenomeAnalysisEngine;
import org.broadinstitute.sting.utils.BaseUtils;
import org.broadinstitute.sting.utils.GenomeLoc;
import org.broadinstitute.sting.utils.MathUtils;
import org.broadinstitute.sting.utils.NGSPlatform;
import org.broadinstitute.sting.utils.collections.Pair;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.exceptions.UserException;
import org.broadinstitute.sting.utils.sam.AlignmentStartComparator;
import org.broadinstitute.sting.utils.sam.ArtificialSAMUtils;
import org.broadinstitute.sting.utils.sam.GATKSAMReadGroupRecord;
import org.broadinstitute.sting.utils.sam.GATKSAMRecord;

public class ReadUtils {
    private static final Logger logger = Logger.getLogger(ReadUtils.class);
    private static final String OFFSET_OUT_OF_BOUNDS_EXCEPTION = "Offset cannot be greater than read length %d : %d";
    private static final String OFFSET_NOT_ZERO_EXCEPTION = "We ran past the end of the read and never found the offset, something went wrong!";
    private static final int DEFAULT_ADAPTOR_SIZE = 100;
    public static final int CLIPPING_GOAL_NOT_REACHED = -1;
    private static final Map<Integer, String> readFlagNames = new HashMap<Integer, String>();
    public static int CANNOT_COMPUTE_ADAPTOR_BOUNDARY;

    private ReadUtils() {
    }

    public static int getMeanRepresentativeReadCount(GATKSAMRecord read) {
        if (!read.isReducedRead()) {
            return 1;
        }
        int[] counts = read.getReducedReadCounts();
        return (int)Math.round((double)MathUtils.sum(counts) / (double)counts.length);
    }

    public static SAMFileWriter createSAMFileWriterWithCompression(SAMFileHeader header, boolean presorted, String file, int compression) {
        ReadUtils.validateCompressionLevel(compression);
        if (file.endsWith(".bam")) {
            return new SAMFileWriterFactory().makeBAMWriter(header, presorted, new File(file), compression);
        }
        return new SAMFileWriterFactory().makeSAMOrBAMWriter(header, presorted, new File(file));
    }

    public static int validateCompressionLevel(int requestedCompressionLevel) {
        if (requestedCompressionLevel < 0 || requestedCompressionLevel > 9) {
            throw new UserException.BadArgumentValue("compress", "Compression level must be 0-9 but got " + requestedCompressionLevel);
        }
        return requestedCompressionLevel;
    }

    public static boolean isBaseInsideAdaptor(GATKSAMRecord read, long basePos) {
        int adaptorBoundary = read.getAdaptorBoundary();
        if (adaptorBoundary == CANNOT_COMPUTE_ADAPTOR_BOUNDARY || read.getInferredInsertSize() > 100) {
            return false;
        }
        return read.getReadNegativeStrandFlag() ? basePos <= (long)adaptorBoundary : basePos >= (long)adaptorBoundary;
    }

    public static int getAdaptorBoundary(SAMRecord read) {
        int MAXIMUM_ADAPTOR_LENGTH = 8;
        int insertSize = Math.abs(read.getInferredInsertSize());
        if (insertSize == 0 || read.getReadUnmappedFlag()) {
            return CANNOT_COMPUTE_ADAPTOR_BOUNDARY;
        }
        if (read.getReadPairedFlag() && read.getReadNegativeStrandFlag() == read.getMateNegativeStrandFlag()) {
            return CANNOT_COMPUTE_ADAPTOR_BOUNDARY;
        }
        int adaptorBoundary = read.getReadNegativeStrandFlag() ? read.getMateAlignmentStart() - 1 : read.getAlignmentStart() + insertSize + 1;
        if (adaptorBoundary < read.getAlignmentStart() - 8 || adaptorBoundary > read.getAlignmentEnd() + 8) {
            adaptorBoundary = CANNOT_COMPUTE_ADAPTOR_BOUNDARY;
        }
        return adaptorBoundary;
    }

    public static boolean is454Read(GATKSAMRecord read) {
        return NGSPlatform.fromRead(read) == NGSPlatform.LS454;
    }

    public static boolean isIonRead(GATKSAMRecord read) {
        return NGSPlatform.fromRead(read) == NGSPlatform.ION_TORRENT;
    }

    public static boolean isSOLiDRead(GATKSAMRecord read) {
        return NGSPlatform.fromRead(read) == NGSPlatform.SOLID;
    }

    public static boolean isIlluminaRead(GATKSAMRecord read) {
        return NGSPlatform.fromRead(read) == NGSPlatform.ILLUMINA;
    }

    public static boolean isPlatformRead(GATKSAMRecord read, String name) {
        String readPlatformAttr;
        GATKSAMReadGroupRecord readGroup = read.getReadGroup();
        if (readGroup != null && (readPlatformAttr = readGroup.getAttribute("PL")) != null) {
            return readPlatformAttr.toString().toUpperCase().contains(name);
        }
        return false;
    }

    public static final List<GATKSAMRecord> sortReadsByCoordinate(List<GATKSAMRecord> reads) {
        SAMRecordCoordinateComparator comparer = new SAMRecordCoordinateComparator();
        Collections.sort(reads, comparer);
        return reads;
    }

    public static final int getFirstInsertionOffset(SAMRecord read) {
        CigarElement e2 = read.getCigar().getCigarElement(0);
        if (e2.getOperator() == CigarOperator.I) {
            return e2.getLength();
        }
        return 0;
    }

    public static final int getLastInsertionOffset(SAMRecord read) {
        CigarElement e2 = read.getCigar().getCigarElement(read.getCigarLength() - 1);
        if (e2.getOperator() == CigarOperator.I) {
            return e2.getLength();
        }
        return 0;
    }

    public static ReadAndIntervalOverlap getReadAndIntervalOverlapType(GATKSAMRecord read, GenomeLoc interval) {
        int sStart = read.getSoftStart();
        int sStop = read.getSoftEnd();
        int uStart = read.getUnclippedStart();
        int uStop = read.getUnclippedEnd();
        if (!read.getReferenceName().equals(interval.getContig())) {
            return ReadAndIntervalOverlap.NO_OVERLAP_CONTIG;
        }
        if (uStop < interval.getStart()) {
            return ReadAndIntervalOverlap.NO_OVERLAP_LEFT;
        }
        if (uStart > interval.getStop()) {
            return ReadAndIntervalOverlap.NO_OVERLAP_RIGHT;
        }
        if (sStop < interval.getStart()) {
            return ReadAndIntervalOverlap.NO_OVERLAP_HARDCLIPPED_LEFT;
        }
        if (sStart > interval.getStop()) {
            return ReadAndIntervalOverlap.NO_OVERLAP_HARDCLIPPED_RIGHT;
        }
        if (sStart >= interval.getStart() && sStop <= interval.getStop()) {
            return ReadAndIntervalOverlap.OVERLAP_CONTAINED;
        }
        if (sStart < interval.getStart() && sStop > interval.getStop()) {
            return ReadAndIntervalOverlap.OVERLAP_LEFT_AND_RIGHT;
        }
        if (sStart < interval.getStart()) {
            return ReadAndIntervalOverlap.OVERLAP_LEFT;
        }
        return ReadAndIntervalOverlap.OVERLAP_RIGHT;
    }

    @Requires(value={"refCoord >= read.getUnclippedStart()", "refCoord <= read.getUnclippedEnd() || (read.getUnclippedEnd() < read.getUnclippedStart())"})
    @Ensures(value={"result >= 0", "result < read.getReadLength()"})
    public static int getReadCoordinateForReferenceCoordinate(GATKSAMRecord read, int refCoord, ClippingTail tail) {
        return ReadUtils.getReadCoordinateForReferenceCoordinate(read.getSoftStart(), read.getCigar(), refCoord, tail, false);
    }

    public static int getReadCoordinateForReferenceCoordinateUpToEndOfRead(GATKSAMRecord read, int refCoord, ClippingTail tail) {
        int leftmostSafeVariantPosition = Math.max(read.getSoftStart(), refCoord);
        return ReadUtils.getReadCoordinateForReferenceCoordinate(read.getSoftStart(), read.getCigar(), leftmostSafeVariantPosition, tail, false);
    }

    public static int getReadCoordinateForReferenceCoordinate(int alignmentStart, Cigar cigar, int refCoord, ClippingTail tail, boolean allowGoalNotReached) {
        Pair<Integer, Boolean> result = ReadUtils.getReadCoordinateForReferenceCoordinate(alignmentStart, cigar, refCoord, allowGoalNotReached);
        int readCoord = result.getFirst();
        if (result.getSecond().booleanValue() && tail == ClippingTail.RIGHT_TAIL) {
            ++readCoord;
        }
        CigarElement firstElementIsInsertion = ReadUtils.readStartsWithInsertion(cigar);
        if (readCoord == 0 && tail == ClippingTail.LEFT_TAIL && firstElementIsInsertion != null) {
            readCoord = Math.min(firstElementIsInsertion.getLength(), cigar.getReadLength() - 1);
        }
        return readCoord;
    }

    @Requires(value={"refCoord >= read.getSoftStart()", "refCoord <= read.getSoftEnd()"})
    @Ensures(value={"result.getFirst() >= 0", "result.getFirst() < read.getReadLength()"})
    public static Pair<Integer, Boolean> getReadCoordinateForReferenceCoordinate(GATKSAMRecord read, int refCoord) {
        return ReadUtils.getReadCoordinateForReferenceCoordinate(read.getSoftStart(), read.getCigar(), refCoord, false);
    }

    public static Pair<Integer, Boolean> getReadCoordinateForReferenceCoordinate(int alignmentStart, Cigar cigar, int refCoord, boolean allowGoalNotReached) {
        int readBases = 0;
        int refBases = 0;
        boolean fallsInsideDeletion = false;
        int goal = refCoord - alignmentStart;
        if (goal < 0) {
            if (allowGoalNotReached) {
                return new Pair<Integer, Boolean>(-1, false);
            }
            throw new ReviewedStingException("Somehow the requested coordinate is not covered by the read. Too many deletions?");
        }
        boolean goalReached = refBases == goal;
        Iterator<CigarElement> cigarElementIterator = cigar.getCigarElements().iterator();
        while (!goalReached && cigarElementIterator.hasNext()) {
            boolean endsWithinCigar;
            CigarElement cigarElement = cigarElementIterator.next();
            int shift = 0;
            if (cigarElement.getOperator().consumesReferenceBases() || cigarElement.getOperator() == CigarOperator.SOFT_CLIP) {
                shift = refBases + cigarElement.getLength() < goal ? cigarElement.getLength() : goal - refBases;
                refBases += shift;
            }
            boolean bl = goalReached = refBases == goal;
            if (!goalReached && cigarElement.getOperator().consumesReadBases()) {
                readBases += cigarElement.getLength();
            }
            if (!goalReached) continue;
            boolean bl2 = endsWithinCigar = shift < cigarElement.getLength();
            if (!endsWithinCigar && !cigarElementIterator.hasNext()) {
                if (allowGoalNotReached) {
                    return new Pair<Integer, Boolean>(-1, false);
                }
                throw new ReviewedStingException(String.format("Reference coordinate corresponds to a non-existent base in the read. This should never happen -- check read with alignment start: %s  and cigar: %s", alignmentStart, cigar));
            }
            if (endsWithinCigar) {
                fallsInsideDeletion = cigarElement.getOperator() == CigarOperator.DELETION;
            } else {
                CigarElement nextCigarElement = cigarElementIterator.next();
                if (nextCigarElement.getOperator() == CigarOperator.INSERTION) {
                    readBases += nextCigarElement.getLength();
                    if (!cigarElementIterator.hasNext()) {
                        if (allowGoalNotReached) {
                            return new Pair<Integer, Boolean>(-1, false);
                        }
                        throw new ReviewedStingException(String.format("Reference coordinate corresponds to a non-existent base in the read. This should never happen -- check read with alignment start: %s  and cigar: %s", alignmentStart, cigar));
                    }
                    nextCigarElement = cigarElementIterator.next();
                }
                boolean bl3 = fallsInsideDeletion = nextCigarElement.getOperator() == CigarOperator.DELETION;
            }
            if (!fallsInsideDeletion && cigarElement.getOperator().consumesReadBases()) {
                readBases += shift;
                continue;
            }
            if (fallsInsideDeletion && !endsWithinCigar && cigarElement.getOperator().consumesReadBases()) {
                readBases += shift - 1;
                continue;
            }
            if (!fallsInsideDeletion || !endsWithinCigar) continue;
            --readBases;
        }
        if (!goalReached) {
            if (allowGoalNotReached) {
                return new Pair<Integer, Boolean>(-1, false);
            }
            throw new ReviewedStingException("Somehow the requested coordinate is not covered by the read. Alignment " + alignmentStart + " | " + cigar);
        }
        return new Pair<Integer, Boolean>(readBases, fallsInsideDeletion);
    }

    @Requires(value={"read1 != null", "read2 != null"})
    public static int compareSAMRecords(GATKSAMRecord read1, GATKSAMRecord read2) {
        AlignmentStartComparator comp = new AlignmentStartComparator();
        return comp.compare(read1, read2);
    }

    public static boolean isInsideRead(GATKSAMRecord read, int referenceCoordinate) {
        return referenceCoordinate >= read.getAlignmentStart() && referenceCoordinate <= read.getAlignmentEnd();
    }

    public static boolean readIsEntirelyInsertion(GATKSAMRecord read) {
        for (CigarElement cigarElement : read.getCigar().getCigarElements()) {
            if (cigarElement.getOperator() == CigarOperator.INSERTION) continue;
            return false;
        }
        return true;
    }

    public static CigarElement readStartsWithInsertion(Cigar cigarForRead) {
        return ReadUtils.readStartsWithInsertion(cigarForRead, true);
    }

    public static CigarElement readStartsWithInsertion(Cigar cigarForRead, boolean ignoreClipOps) {
        for (CigarElement cigarElement : cigarForRead.getCigarElements()) {
            if (cigarElement.getOperator() == CigarOperator.INSERTION) {
                return cigarElement;
            }
            if (ignoreClipOps && (cigarElement.getOperator() == CigarOperator.HARD_CLIP || cigarElement.getOperator() == CigarOperator.SOFT_CLIP)) continue;
            break;
        }
        return null;
    }

    public static int[] getCoverageDistributionOfReads(List<GATKSAMRecord> list, int startLocation, int stopLocation) {
        int[] totalCoverage = new int[stopLocation - startLocation + 1];
        for (GATKSAMRecord read : list) {
            int[] readCoverage = ReadUtils.getCoverageDistributionOfRead(read, startLocation, stopLocation);
            totalCoverage = MathUtils.addArrays(totalCoverage, readCoverage);
        }
        return totalCoverage;
    }

    public static int[] getCoverageDistributionOfRead(GATKSAMRecord read, int startLocation, int stopLocation) {
        int[] coverage = new int[stopLocation - startLocation + 1];
        int refLocation = read.getSoftStart();
        for (CigarElement cigarElement : read.getCigar().getCigarElements()) {
            switch (cigarElement.getOperator()) {
                case S: 
                case M: 
                case EQ: 
                case N: 
                case X: 
                case D: {
                    for (int i2 = 0; i2 < cigarElement.getLength(); ++i2) {
                        if (refLocation >= startLocation && refLocation <= stopLocation) {
                            int baseCount = read.isReducedRead() ? read.getReducedCount(refLocation - read.getSoftStart()) : 1;
                            int n2 = refLocation - startLocation;
                            coverage[n2] = coverage[n2] + baseCount;
                        }
                        ++refLocation;
                    }
                    break;
                }
            }
            if (refLocation <= stopLocation) continue;
            break;
        }
        return coverage;
    }

    public static Pair<HashMap<Integer, HashSet<GATKSAMRecord>>, HashMap<GATKSAMRecord, Boolean[]>> getBothReadToLociMappings(List<GATKSAMRecord> readList, int startLocation, int stopLocation) {
        int arraySize = stopLocation - startLocation + 1;
        HashMap locusToReadMap = new HashMap(2 * (stopLocation - startLocation + 1), 0.5f);
        HashMap<GATKSAMRecord, Boolean[]> readToLocusMap = new HashMap<GATKSAMRecord, Boolean[]>(2 * readList.size(), 0.5f);
        for (int i2 = startLocation; i2 <= stopLocation; ++i2) {
            locusToReadMap.put(i2, new HashSet());
        }
        for (GATKSAMRecord read : readList) {
            readToLocusMap.put(read, new Boolean[arraySize]);
            int[] readCoverage = ReadUtils.getCoverageDistributionOfRead(read, startLocation, stopLocation);
            for (int i3 = 0; i3 < readCoverage.length; ++i3) {
                int refLocation = i3 + startLocation;
                if (readCoverage[i3] > 0) {
                    HashSet readSet = (HashSet)locusToReadMap.get(refLocation);
                    readSet.add(read);
                    ((Boolean[])readToLocusMap.get((Object)read))[refLocation - startLocation] = true;
                    continue;
                }
                ((Boolean[])readToLocusMap.get((Object)read))[refLocation - startLocation] = false;
            }
        }
        return new Pair<HashMap<Integer, HashSet<GATKSAMRecord>>, HashMap<GATKSAMRecord, Boolean[]>>(locusToReadMap, readToLocusMap);
    }

    public static byte[] createRandomReadQuals(int length) {
        Random random = GenomeAnalysisEngine.getRandomGenerator();
        byte[] quals = new byte[length];
        for (int i2 = 0; i2 < length; ++i2) {
            quals[i2] = (byte)random.nextInt(50);
        }
        return quals;
    }

    public static byte[] createRandomReadBases(int length, boolean allowNs) {
        Random random = GenomeAnalysisEngine.getRandomGenerator();
        int numberOfBases = allowNs ? 5 : 4;
        byte[] bases = new byte[length];
        block7: for (int i2 = 0; i2 < length; ++i2) {
            switch (random.nextInt(numberOfBases)) {
                case 0: {
                    bases[i2] = 65;
                    continue block7;
                }
                case 1: {
                    bases[i2] = 67;
                    continue block7;
                }
                case 2: {
                    bases[i2] = 71;
                    continue block7;
                }
                case 3: {
                    bases[i2] = 84;
                    continue block7;
                }
                case 4: {
                    bases[i2] = 78;
                    continue block7;
                }
                default: {
                    throw new ReviewedStingException("Something went wrong, this is just impossible");
                }
            }
        }
        return bases;
    }

    public static GATKSAMRecord createRandomRead(int length) {
        return ReadUtils.createRandomRead(length, true);
    }

    public static GATKSAMRecord createRandomRead(int length, boolean allowNs) {
        byte[] quals = ReadUtils.createRandomReadQuals(length);
        byte[] bbases = ReadUtils.createRandomReadBases(length, allowNs);
        return ArtificialSAMUtils.createArtificialRead(bbases, quals, bbases.length + "M");
    }

    public static String prettyPrintSequenceRecords(SAMSequenceDictionary sequenceDictionary) {
        Object[] sequenceRecordNames = new String[sequenceDictionary.size()];
        int sequenceRecordIndex = 0;
        for (SAMSequenceRecord sequenceRecord : sequenceDictionary.getSequences()) {
            sequenceRecordNames[sequenceRecordIndex++] = sequenceRecord.getSequenceName();
        }
        return Arrays.deepToString(sequenceRecordNames);
    }

    public static long getReferenceCoordinateForReadCoordinate(GATKSAMRecord read, int offset) {
        if (offset > read.getReadLength()) {
            throw new ReviewedStingException(String.format(OFFSET_OUT_OF_BOUNDS_EXCEPTION, offset, read.getReadLength()));
        }
        long location = read.getAlignmentStart();
        Iterator<CigarElement> cigarElementIterator = read.getCigar().getCigarElements().iterator();
        while (offset > 0 && cigarElementIterator.hasNext()) {
            CigarElement cigarElement = cigarElementIterator.next();
            long move = 0L;
            if (cigarElement.getOperator().consumesReferenceBases()) {
                move = Math.min(cigarElement.getLength(), offset);
            }
            location += move;
            offset = (int)((long)offset - move);
        }
        if (offset > 0 && !cigarElementIterator.hasNext()) {
            throw new ReviewedStingException(OFFSET_NOT_ZERO_EXCEPTION);
        }
        return location;
    }

    public static Map<CigarOperator, ArrayList<Integer>> getCigarOperatorForAllBases(GATKSAMRecord read) {
        HashMap<CigarOperator, ArrayList<Integer>> events = new HashMap<CigarOperator, ArrayList<Integer>>();
        int position = 0;
        for (CigarElement cigarElement : read.getCigar().getCigarElements()) {
            ArrayList list;
            CigarOperator op = cigarElement.getOperator();
            if (op.consumesReadBases()) {
                list = (ArrayList)events.get((Object)op);
                if (list == null) {
                    list = new ArrayList();
                    events.put(op, list);
                }
                for (int i2 = position; i2 < cigarElement.getLength(); ++i2) {
                    list.add(position++);
                }
                continue;
            }
            list = (ArrayList)events.get((Object)op);
            if (list == null) {
                list = new ArrayList();
                events.put(op, list);
            }
            list.add(position);
        }
        return events;
    }

    public static String convertReadBasesToString(GATKSAMRecord read) {
        String bases = "";
        for (byte b2 : read.getReadBases()) {
            bases = bases + (char)b2;
        }
        return bases.toUpperCase();
    }

    public static String convertReadQualToString(byte[] quals) {
        String result = "";
        for (byte b2 : quals) {
            result = result + (char)(33 + b2);
        }
        return result;
    }

    public static String convertReadQualToString(GATKSAMRecord read) {
        return ReadUtils.convertReadQualToString(read.getBaseQualities());
    }

    public static String getBasesReverseComplement(byte[] bases) {
        String reverse = "";
        for (int i2 = bases.length - 1; i2 >= 0; --i2) {
            reverse = reverse + (char)BaseUtils.getComplement(bases[i2]);
        }
        return reverse;
    }

    public static String getBasesReverseComplement(GATKSAMRecord read) {
        return ReadUtils.getBasesReverseComplement(read.getReadBases());
    }

    @Ensures(value={"result >= 0"})
    public static int getMaxReadLength(List<GATKSAMRecord> reads) {
        if (reads == null) {
            throw new IllegalArgumentException("Attempting to check a null list of reads.");
        }
        int maxReadLength = 0;
        for (GATKSAMRecord read : reads) {
            maxReadLength = Math.max(maxReadLength, read.getReadLength());
        }
        return maxReadLength;
    }

    static {
        readFlagNames.put(1, "Paired");
        readFlagNames.put(2, "Proper");
        readFlagNames.put(4, "Unmapped");
        readFlagNames.put(8, "MateUnmapped");
        readFlagNames.put(16, "Forward");
        readFlagNames.put(64, "FirstOfPair");
        readFlagNames.put(128, "SecondOfPair");
        readFlagNames.put(256, "NotPrimary");
        readFlagNames.put(512, "NON-PF");
        readFlagNames.put(1024, "Duplicate");
        CANNOT_COMPUTE_ADAPTOR_BOUNDARY = Integer.MIN_VALUE;
    }

    public static enum ReadAndIntervalOverlap {
        NO_OVERLAP_CONTIG,
        NO_OVERLAP_LEFT,
        NO_OVERLAP_RIGHT,
        NO_OVERLAP_HARDCLIPPED_LEFT,
        NO_OVERLAP_HARDCLIPPED_RIGHT,
        OVERLAP_LEFT,
        OVERLAP_RIGHT,
        OVERLAP_LEFT_AND_RIGHT,
        OVERLAP_CONTAINED;

    }

    public static enum ClippingTail {
        LEFT_TAIL,
        RIGHT_TAIL;

    }
}

