/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.gatk.utils.recalibration;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import org.apache.commons.math.MathException;
import org.apache.commons.math.stat.inference.ChiSquareTestImpl;
import org.apache.log4j.Logger;
import org.broadinstitute.gatk.utils.collections.Pair;
import org.broadinstitute.gatk.utils.exceptions.ReviewedGATKException;
import org.broadinstitute.gatk.utils.recalibration.RecalDatum;

public class RecalDatumNode<T extends RecalDatum> {
    private static final double SMALLEST_CHI2_PVALUE = 1.0E-300;
    protected static final Logger logger = Logger.getLogger(RecalDatumNode.class);
    private static final double UNINITIALIZED = Double.NEGATIVE_INFINITY;
    private final T recalDatum;
    private double fixedPenalty = Double.NEGATIVE_INFINITY;
    private final Set<RecalDatumNode<T>> subnodes;

    @Requires(value={"recalDatum != null"})
    public RecalDatumNode(T recalDatum) {
        this(recalDatum, new HashSet<RecalDatumNode<T>>());
    }

    public String toString() {
        return ((RecalDatum)this.recalDatum).toString();
    }

    @Requires(value={"recalDatum != null", "subnodes != null"})
    public RecalDatumNode(T recalDatum, Set<RecalDatumNode<T>> subnodes) {
        this(recalDatum, Double.NEGATIVE_INFINITY, subnodes);
    }

    @Requires(value={"recalDatum != null"})
    protected RecalDatumNode(T recalDatum, double fixedPenalty) {
        this(recalDatum, fixedPenalty, new HashSet<RecalDatumNode<T>>());
    }

    @Requires(value={"recalDatum != null", "subnodes != null"})
    protected RecalDatumNode(T recalDatum, double fixedPenalty, Set<RecalDatumNode<T>> subnodes) {
        this.recalDatum = recalDatum;
        this.fixedPenalty = fixedPenalty;
        this.subnodes = new HashSet<RecalDatumNode<T>>(subnodes);
    }

    @Ensures(value={"result != null"})
    public T getRecalDatum() {
        return this.recalDatum;
    }

    @Ensures(value={"result != null"})
    public Set<RecalDatumNode<T>> getSubnodes() {
        return this.subnodes;
    }

    public double getPenalty() {
        if (this.fixedPenalty != Double.NEGATIVE_INFINITY) {
            return this.fixedPenalty;
        }
        return this.calcPenalty();
    }

    public double calcAndSetFixedPenalty(boolean doEntireTree) {
        this.fixedPenalty = this.calcPenalty();
        if (doEntireTree) {
            for (RecalDatumNode<T> sub : this.subnodes) {
                sub.calcAndSetFixedPenalty(doEntireTree);
            }
        }
        return this.fixedPenalty;
    }

    @Requires(value={"sub != null"})
    public void addSubnode(RecalDatumNode<T> sub) {
        this.subnodes.add(sub);
    }

    public boolean isLeaf() {
        return this.subnodes.isEmpty();
    }

    public boolean isAboveOnlyLeaves() {
        for (RecalDatumNode<T> sub : this.subnodes) {
            if (sub.isLeaf()) continue;
            return false;
        }
        return true;
    }

    @Ensures(value={"result >= 0"})
    public int getNumSubnodes() {
        return this.subnodes.size();
    }

    public double totalPenalty() {
        if (this.isLeaf()) {
            return this.getPenalty();
        }
        double sum = 0.0;
        for (RecalDatumNode<T> sub : this.subnodes) {
            sum += sub.totalPenalty();
        }
        return sum;
    }

    public double maxPenalty(boolean leafOnly) {
        double max = !leafOnly || this.isLeaf() ? this.getPenalty() : Double.MIN_VALUE;
        for (RecalDatumNode<T> sub : this.subnodes) {
            max = Math.max(max, sub.maxPenalty(leafOnly));
        }
        return max;
    }

    public double minPenalty(boolean leafOnly) {
        double min = !leafOnly || this.isLeaf() ? this.getPenalty() : Double.MAX_VALUE;
        for (RecalDatumNode<T> sub : this.subnodes) {
            min = Math.min(min, sub.minPenalty(leafOnly));
        }
        return min;
    }

    public int maxDepth() {
        int subMax = 0;
        for (RecalDatumNode<T> sub : this.subnodes) {
            subMax = Math.max(subMax, sub.maxDepth());
        }
        return subMax + 1;
    }

    @Ensures(value={"result > 0"})
    public int minDepth() {
        if (this.isLeaf()) {
            return 1;
        }
        int subMin = Integer.MAX_VALUE;
        for (RecalDatumNode<T> sub : this.subnodes) {
            subMin = Math.min(subMin, sub.minDepth());
        }
        return subMin + 1;
    }

    @Ensures(value={"result > 0"})
    public int size() {
        int size = 1;
        for (RecalDatumNode<T> sub : this.subnodes) {
            size += sub.size();
        }
        return size;
    }

    @Ensures(value={"result >= 0"})
    public int numLeaves() {
        if (this.isLeaf()) {
            return 1;
        }
        int size = 0;
        for (RecalDatumNode<T> sub : this.subnodes) {
            size += sub.numLeaves();
        }
        return size;
    }

    private double calcPenalty() {
        if (this.isLeaf() || this.freeToMerge()) {
            return 0.0;
        }
        if (this.subnodes.size() == 1) {
            return 0.0;
        }
        long[][] counts = new long[this.subnodes.size()][2];
        int i2 = 0;
        for (RecalDatumNode<T> subnode : this.subnodes) {
            counts[i2][0] = Math.round(((RecalDatum)subnode.getRecalDatum()).getNumMismatches()) + 1L;
            counts[i2][1] = ((RecalDatum)subnode.getRecalDatum()).getNumObservations() + 2L;
            ++i2;
        }
        try {
            double chi2PValue = new ChiSquareTestImpl().chiSquareTest(counts);
            double penalty = -10.0 * Math.log10(Math.max(chi2PValue, 1.0E-300));
            if (Double.isInfinite(penalty) || Double.isNaN(penalty)) {
                throw new ReviewedGATKException("chi2 value is " + chi2PValue + " at " + this.getRecalDatum());
            }
            return penalty;
        }
        catch (MathException e2) {
            throw new ReviewedGATKException("Failed in calculating chi2 value", e2);
        }
    }

    private boolean freeToMerge() {
        if (this.isLeaf()) {
            return true;
        }
        byte myQual = ((RecalDatum)this.getRecalDatum()).getEmpiricalQualityAsByte();
        for (RecalDatumNode<T> sub : this.subnodes) {
            if (((RecalDatum)sub.getRecalDatum()).getEmpiricalQualityAsByte() == myQual) continue;
            return false;
        }
        return true;
    }

    @Requires(value={"globalErrorRate >= 0.0"})
    @Ensures(value={"result >= 0.0"})
    private double calcPenaltyLog10(double globalErrorRate) {
        if (globalErrorRate == 0.0) {
            return 0.0;
        }
        if (this.isLeaf()) {
            return Math.abs(Math.log10(((RecalDatum)this.recalDatum).getEmpiricalErrorRate()) - Math.log10(globalErrorRate)) * (double)((RecalDatum)this.recalDatum).getNumObservations();
        }
        double sum = 0.0;
        for (RecalDatumNode<T> hrd : this.subnodes) {
            sum += super.calcPenaltyLog10(globalErrorRate);
        }
        return sum;
    }

    public RecalDatumNode<T> pruneToDepth(int maxDepth) {
        if (maxDepth < 1) {
            throw new IllegalArgumentException("maxDepth < 1");
        }
        HashSet<RecalDatumNode<T>> subPruned = new HashSet<RecalDatumNode<T>>(this.getNumSubnodes());
        if (maxDepth > 1) {
            for (RecalDatumNode<T> sub : this.subnodes) {
                subPruned.add(sub.pruneToDepth(maxDepth - 1));
            }
        }
        return new RecalDatumNode<T>(this.getRecalDatum(), this.fixedPenalty, subPruned);
    }

    public RecalDatumNode<T> pruneByPenalty(int maxElements) {
        RecalDatumNode<T> root = this;
        while (root.size() > maxElements) {
            root = root.removeLowestPenaltyNode();
        }
        return root;
    }

    public RecalDatumNode<T> pruneToNoMoreThanPenalty(double maxPenaltyIn, boolean applyBonferroniCorrection) {
        double maxPenalty;
        RecalDatumNode<T> root = this;
        double bonferroniCorrection = 10.0 * Math.log10(this.size());
        double d2 = maxPenalty = applyBonferroniCorrection ? maxPenaltyIn + bonferroniCorrection : maxPenaltyIn;
        if (applyBonferroniCorrection) {
            logger.info(String.format("Applying Bonferroni correction for %d nodes = %.2f to initial penalty %.2f for total corrected max penalty of %.2f", this.size(), bonferroniCorrection, maxPenaltyIn, maxPenalty));
        }
        while (true) {
            Pair<RecalDatumNode<T>, Double> minPenaltyNode;
            if ((minPenaltyNode = root.getMinPenaltyAboveLeafNode()) == null || minPenaltyNode.getSecond() > maxPenalty) {
                if (minPenaltyNode == null) {
                    if (!logger.isDebugEnabled()) break;
                    logger.debug("Stopping because no candidates could be found");
                    break;
                }
                if (!logger.isDebugEnabled()) break;
                logger.debug("Stopping because node " + minPenaltyNode.getFirst() + " has penalty " + minPenaltyNode.getSecond() + " > max " + maxPenalty);
                break;
            }
            if (logger.isDebugEnabled()) {
                logger.debug("Removing node " + minPenaltyNode.getFirst() + " with penalty " + minPenaltyNode.getSecond());
            }
            root = root.removeLowestPenaltyNode();
        }
        return root;
    }

    private RecalDatumNode<T> removeLowestPenaltyNode() {
        Pair<RecalDatumNode<T>, Boolean> result;
        Pair<RecalDatumNode<T>, Double> nodeToRemove = this.getMinPenaltyAboveLeafNode();
        if (logger.isDebugEnabled()) {
            logger.debug("Removing " + nodeToRemove.getFirst() + " with penalty " + nodeToRemove.getSecond());
        }
        if (!(result = this.removeNode(nodeToRemove.getFirst())).getSecond().booleanValue()) {
            throw new IllegalStateException("Never removed any node!");
        }
        RecalDatumNode<T> oneRemoved = result.getFirst();
        if (oneRemoved == null) {
            throw new IllegalStateException("Removed our root node, wow, didn't expect that");
        }
        return oneRemoved;
    }

    private Pair<RecalDatumNode<T>, Double> getMinPenaltyAboveLeafNode() {
        if (this.isLeaf()) {
            return null;
        }
        if (this.isAboveOnlyLeaves()) {
            return new Pair<RecalDatumNode<T>, Double>(this, this.getPenalty());
        }
        Pair<RecalDatumNode<T>, Double> minNode = null;
        for (RecalDatumNode<T> sub : this.subnodes) {
            Pair<RecalDatumNode<T>, Double> subFind = super.getMinPenaltyAboveLeafNode();
            if (subFind == null || minNode != null && !(subFind.getSecond() < minNode.getSecond())) continue;
            minNode = subFind;
        }
        return minNode;
    }

    private Pair<RecalDatumNode<T>, Boolean> removeNode(RecalDatumNode<T> nodeToRemove) {
        if (this == nodeToRemove) {
            if (this.isLeaf()) {
                throw new IllegalStateException("Trying to remove a leaf node from the tree! " + this + " " + nodeToRemove);
            }
            RecalDatumNode<T> node = new RecalDatumNode<T>(this.getRecalDatum(), this.fixedPenalty);
            return new Pair<RecalDatumNode<T>, Boolean>(node, true);
        }
        boolean removedSomething = false;
        HashSet<RecalDatumNode<T>> sub = new HashSet<RecalDatumNode<T>>(this.getNumSubnodes());
        for (RecalDatumNode<T> sub1 : this.subnodes) {
            if (removedSomething) {
                sub.add(sub1);
                continue;
            }
            Pair<RecalDatumNode<T>, Boolean> maybeRemoved = super.removeNode(nodeToRemove);
            removedSomething = maybeRemoved.getSecond();
            sub.add(maybeRemoved.getFirst());
        }
        RecalDatumNode<T> node = new RecalDatumNode<T>(this.getRecalDatum(), this.fixedPenalty, sub);
        return new Pair<RecalDatumNode<T>, Boolean>(node, removedSomething);
    }

    public Collection<T> getAllLeaves() {
        LinkedList list = new LinkedList();
        this.getAllLeavesRec(list);
        return list;
    }

    private void getAllLeavesRec(LinkedList<T> list) {
        if (this.isLeaf()) {
            list.add(this.getRecalDatum());
        } else {
            for (RecalDatumNode<T> sub : this.subnodes) {
                super.getAllLeavesRec(list);
            }
        }
    }
}

