/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.progressmeter;

import com.google.java.contract.Ensures;
import com.google.java.contract.Invariant;
import com.google.java.contract.Requires;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.List;
import org.apache.log4j.Logger;
import org.broadinstitute.sting.utils.AutoFormattingTime;
import org.broadinstitute.sting.utils.GenomeLoc;
import org.broadinstitute.sting.utils.GenomeLocSortedSet;
import org.broadinstitute.sting.utils.SimpleTimer;
import org.broadinstitute.sting.utils.Utils;
import org.broadinstitute.sting.utils.exceptions.UserException;
import org.broadinstitute.sting.utils.progressmeter.ProgressMeterDaemon;
import org.broadinstitute.sting.utils.progressmeter.ProgressMeterData;

@Invariant(value={"targetSizeInBP >= 0", "progressPrintFrequency > 0"})
public class ProgressMeter {
    protected static final Logger logger = Logger.getLogger(ProgressMeter.class);
    private static final long MIN_ELAPSED_TIME_BEFORE_FIRST_PROGRESS = 30000L;
    private static final long PERFORMANCE_LOG_PRINT_FREQUENCY = 10000L;
    private static final double TWO_HOURS_IN_SECONDS = 7200.0;
    private static final double TWELVE_HOURS_IN_SECONDS = 43200.0;
    private long lastProgressPrintTime = -1L;
    private long progressPrintFrequency = 10000L;
    private long lastPerformanceLogPrintTime = -1L;
    private final GenomeLocSortedSet regionsBeingProcessed;
    private final long targetSizeInBP;
    private final String processingUnitName;
    private final PrintStream performanceLog;
    private final SimpleTimer timer = new SimpleTimer();
    private GenomeLoc maxGenomeLoc = null;
    private Position position = new Position(PositionStatus.STARTING);
    private long nTotalRecordsProcessed = 0L;
    private long elapsedTimeInNanosecondUpdatedByDaemon = 0L;
    final ProgressMeterDaemon progressMeterDaemon;

    public ProgressMeter(File performanceLogFile, String processingUnitName, GenomeLocSortedSet processingIntervals) {
        this(performanceLogFile, processingUnitName, processingIntervals, 10000L);
    }

    protected ProgressMeter(File performanceLogFile, String processingUnitName, GenomeLocSortedSet processingIntervals, long pollingFrequency) {
        if (processingUnitName == null) {
            throw new IllegalArgumentException("processingUnitName cannot be null");
        }
        if (processingIntervals == null) {
            throw new IllegalArgumentException("Target intervals cannot be null");
        }
        this.processingUnitName = processingUnitName;
        this.regionsBeingProcessed = processingIntervals;
        if (performanceLogFile != null) {
            try {
                this.performanceLog = new PrintStream(new FileOutputStream(performanceLogFile));
                List<String> pLogHeader = Arrays.asList("elapsed.time", "units.processed", "processing.speed", "bp.processed", "bp.speed", "genome.fraction.complete", "est.total.runtime", "est.time.remaining");
                this.performanceLog.println(Utils.join("\t", pLogHeader));
            }
            catch (FileNotFoundException e) {
                throw new UserException.CouldNotCreateOutputFile(performanceLogFile, (Exception)e);
            }
        } else {
            this.performanceLog = null;
        }
        this.targetSizeInBP = processingIntervals.coveredSize();
        this.progressMeterDaemon = new ProgressMeterDaemon(this, pollingFrequency);
    }

    public ProgressMeterDaemon getProgressMeterDaemon() {
        return this.progressMeterDaemon;
    }

    @Requires(value={"progressMeterDaemon != null"})
    public synchronized void start() {
        this.timer.start();
        this.lastProgressPrintTime = this.timer.currentTime();
        logger.info((Object)"[INITIALIZATION COMPLETE; STARTING PROCESSING]");
        logger.info((Object)String.format("%15s processed.%s  runtime per.1M.%s completed total.runtime remaining", "Location", this.processingUnitName, this.processingUnitName));
        this.progressMeterDaemon.start();
    }

    @Ensures(value={"result >= 0"})
    public long getRuntimeInNanoseconds() {
        return this.timer.getElapsedTimeNano();
    }

    @Ensures(value={"result >= 0"})
    public long getRuntimeInNanosecondsUpdatedPeriodically() {
        return this.elapsedTimeInNanosecondUpdatedByDaemon;
    }

    protected void updateElapsedTimeInNanoseconds() {
        this.elapsedTimeInNanosecondUpdatedByDaemon = this.getRuntimeInNanoseconds();
    }

    public synchronized void notifyOfProgress(GenomeLoc loc, long nTotalRecordsProcessed) {
        if (nTotalRecordsProcessed < 0L) {
            throw new IllegalArgumentException("nTotalRecordsProcessed must be >= 0");
        }
        if (loc.size() != 1) {
            throw new IllegalArgumentException("GenomeLoc must have size == 1 but got " + loc);
        }
        this.maxGenomeLoc = loc == null ? loc : (this.maxGenomeLoc == null ? loc : loc.max(this.maxGenomeLoc));
        this.nTotalRecordsProcessed = Math.max(this.nTotalRecordsProcessed, nTotalRecordsProcessed);
        this.position = this.maxGenomeLoc == null ? new Position(PositionStatus.IN_UNMAPPED_READS) : new Position(this.maxGenomeLoc);
    }

    protected synchronized void printProgress(boolean mustPrint) {
        boolean printLog;
        long curTime = this.timer.currentTime();
        boolean printProgress = mustPrint || this.maxElapsedIntervalForPrinting(curTime, this.lastProgressPrintTime, this.progressPrintFrequency);
        boolean bl = printLog = this.performanceLog != null && this.maxElapsedIntervalForPrinting(curTime, this.lastPerformanceLogPrintTime, 10000L);
        if (printProgress || printLog) {
            ProgressMeterData progressData = this.takeProgressSnapshot(this.maxGenomeLoc, this.nTotalRecordsProcessed);
            AutoFormattingTime elapsed = new AutoFormattingTime(progressData.getElapsedSeconds());
            AutoFormattingTime bpRate = new AutoFormattingTime(progressData.secondsPerMillionBP());
            AutoFormattingTime unitRate = new AutoFormattingTime(progressData.secondsPerMillionElements());
            double fractionGenomeTargetCompleted = progressData.calculateFractionGenomeTargetCompleted(this.targetSizeInBP);
            AutoFormattingTime estTotalRuntime = new AutoFormattingTime(elapsed.getTimeInSeconds() / fractionGenomeTargetCompleted);
            AutoFormattingTime timeToCompletion = new AutoFormattingTime(estTotalRuntime.getTimeInSeconds() - elapsed.getTimeInSeconds());
            if (printProgress) {
                this.lastProgressPrintTime = curTime;
                this.updateLoggerPrintFrequency(estTotalRuntime.getTimeInSeconds());
                logger.info((Object)String.format("%15s        %5.2e %s     %s    %5.1f%%      %s  %s", this.position.getMessage(), (double)progressData.getUnitsProcessed() * 1.0, elapsed, unitRate, 100.0 * fractionGenomeTargetCompleted, estTotalRuntime, timeToCompletion));
            }
            if (printLog) {
                this.lastPerformanceLogPrintTime = curTime;
                this.performanceLog.printf("%.2f\t%d\t%.2e\t%d\t%.2e\t%.2e\t%.2f\t%.2f%n", elapsed.getTimeInSeconds(), progressData.getUnitsProcessed(), unitRate.getTimeInSeconds(), progressData.getBpProcessed(), bpRate.getTimeInSeconds(), fractionGenomeTargetCompleted, estTotalRuntime.getTimeInSeconds(), timeToCompletion.getTimeInSeconds());
            }
        }
    }

    private void updateLoggerPrintFrequency(double totalRuntimeSeconds) {
        this.progressPrintFrequency = totalRuntimeSeconds > 43200.0 ? 60000L : (totalRuntimeSeconds > 7200.0 ? 30000L : 10000L);
    }

    private ProgressMeterData takeProgressSnapshot(GenomeLoc loc, long nTotalRecordsProcessed) {
        long bpProcessed = loc == null ? this.targetSizeInBP : this.regionsBeingProcessed.sizeBeforeLoc(loc);
        return new ProgressMeterData(this.timer.getElapsedTime(), nTotalRecordsProcessed, bpProcessed);
    }

    public void notifyDone(long nTotalRecordsProcessed) {
        this.nTotalRecordsProcessed = nTotalRecordsProcessed;
        this.position = new Position(PositionStatus.DONE);
        this.printProgress(true);
        logger.info((Object)String.format("Total runtime %.2f secs, %.2f min, %.2f hours", this.timer.getElapsedTime(), this.timer.getElapsedTime() / 60.0, this.timer.getElapsedTime() / 3600.0));
        if (this.performanceLog != null) {
            this.performanceLog.close();
        }
        this.progressMeterDaemon.done();
    }

    private boolean maxElapsedIntervalForPrinting(long curTime, long lastPrintTime, long printFreq) {
        long elapsed = curTime - lastPrintTime;
        return elapsed > printFreq && elapsed > 30000L;
    }

    private class Position {
        final PositionStatus type;
        final GenomeLoc maybeLoc;

        @Requires(value={"type != null", "type != PositionStatus.ON_GENOME"})
        private Position(PositionStatus type) {
            this.type = type;
            this.maybeLoc = null;
        }

        @Requires(value={"loc != null"})
        private Position(GenomeLoc loc) {
            this.type = PositionStatus.ON_GENOME;
            this.maybeLoc = loc;
        }

        private String getMessage() {
            if (this.type == PositionStatus.ON_GENOME) {
                return ProgressMeter.this.maxGenomeLoc.getContig() + ":" + ProgressMeter.this.maxGenomeLoc.getStart();
            }
            return this.type.message;
        }
    }

    private static enum PositionStatus {
        STARTING("Starting"),
        DONE("done"),
        IN_UNMAPPED_READS("unmapped reads"),
        ON_GENOME(null);

        public final String message;

        private PositionStatus(String message) {
            this.message = message;
        }
    }
}

