/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.nanoScheduler;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.log4j.Logger;
import org.broadinstitute.sting.utils.MultiThreadedErrorTracker;
import org.broadinstitute.sting.utils.nanoScheduler.InputProducer;
import org.broadinstitute.sting.utils.nanoScheduler.MapResult;
import org.broadinstitute.sting.utils.nanoScheduler.NSMapFunction;
import org.broadinstitute.sting.utils.nanoScheduler.NSProgressFunction;
import org.broadinstitute.sting.utils.nanoScheduler.NSReduceFunction;
import org.broadinstitute.sting.utils.nanoScheduler.NSRuntimeProfile;
import org.broadinstitute.sting.utils.nanoScheduler.Reducer;
import org.broadinstitute.sting.utils.threading.NamedThreadFactory;

public class NanoScheduler<InputType, MapType, ReduceType> {
    private static final Logger logger = Logger.getLogger(NanoScheduler.class);
    private static final boolean ALLOW_SINGLE_THREAD_FASTPATH = true;
    private static final boolean LOG_MAP_TIMES = false;
    final int bufferSize;
    final int nThreads;
    final ExecutorService inputExecutor;
    final ExecutorService masterExecutor;
    final ExecutorService mapExecutor;
    final Semaphore runningMapJobSlots;
    final MultiThreadedErrorTracker errorTracker = new MultiThreadedErrorTracker();
    boolean shutdown = false;
    boolean debug = false;
    private NSProgressFunction<InputType> progressFunction = null;
    private static final NSRuntimeProfile combinedNSRuntimeProfiler = new NSRuntimeProfile();
    private final NSRuntimeProfile myNSRuntimeProfile = new NSRuntimeProfile();

    public NanoScheduler(int nThreads) {
        this(nThreads * 100, nThreads);
    }

    protected NanoScheduler(int bufferSize, int nThreads) {
        if (bufferSize < 1) {
            throw new IllegalArgumentException("bufferSize must be >= 1, got " + bufferSize);
        }
        if (nThreads < 1) {
            throw new IllegalArgumentException("nThreads must be >= 1, got " + nThreads);
        }
        this.bufferSize = bufferSize;
        this.nThreads = nThreads;
        if (nThreads == 1) {
            this.masterExecutor = null;
            this.inputExecutor = null;
            this.mapExecutor = null;
            this.runningMapJobSlots = null;
        } else {
            this.mapExecutor = Executors.newFixedThreadPool(nThreads - 1, new NamedThreadFactory("NS-map-thread-%d"));
            this.runningMapJobSlots = new Semaphore(this.bufferSize);
            this.inputExecutor = Executors.newSingleThreadExecutor(new NamedThreadFactory("NS-input-thread-%d"));
            this.masterExecutor = Executors.newSingleThreadExecutor(new NamedThreadFactory("NS-master-thread-%d"));
        }
        this.myNSRuntimeProfile.outsideSchedulerTimer.start();
    }

    @Ensures(value={"result > 0"})
    public int getnThreads() {
        return this.nThreads;
    }

    @Ensures(value={"result > 0"})
    public int getBufferSize() {
        return this.bufferSize;
    }

    public void shutdown() {
        this.myNSRuntimeProfile.outsideSchedulerTimer.stop();
        combinedNSRuntimeProfiler.combine(this.myNSRuntimeProfile);
        if (this.nThreads > 1) {
            this.shutdownExecutor("inputExecutor", this.inputExecutor);
            this.shutdownExecutor("mapExecutor", this.mapExecutor);
            this.shutdownExecutor("masterExecutor", this.masterExecutor);
        }
        this.shutdown = true;
    }

    public void printRuntimeProfile() {
        this.myNSRuntimeProfile.log(logger);
    }

    public static void printCombinedRuntimeProfile() {
        if (combinedNSRuntimeProfiler.totalRuntimeInSeconds() > 0.1) {
            combinedNSRuntimeProfiler.log(logger);
        }
    }

    protected double getTotalRuntime() {
        return this.myNSRuntimeProfile.totalRuntimeInSeconds();
    }

    @Requires(value={"name != null", "executorService != null"})
    @Ensures(value={"executorService.isShutdown()"})
    private void shutdownExecutor(String name, ExecutorService executorService) {
        if (executorService.isShutdown() || executorService.isTerminated()) {
            throw new IllegalStateException("Executor service " + name + " is already shut down!");
        }
        List<Runnable> remaining = executorService.shutdownNow();
        if (!remaining.isEmpty()) {
            throw new IllegalStateException(remaining.size() + " remaining tasks found in an executor " + name + ", unexpected behavior!");
        }
    }

    public boolean isShutdown() {
        return this.shutdown;
    }

    public boolean isDebug() {
        return this.debug;
    }

    @Requires(value={"format != null"})
    protected void debugPrint(String format, Object ... args) {
        if (this.isDebug()) {
            logger.warn("Thread " + Thread.currentThread().getId() + ":" + String.format(format, args));
        }
    }

    public void setDebug(boolean debug) {
        this.debug = debug;
    }

    public void setProgressFunction(NSProgressFunction<InputType> progressFunction) {
        this.progressFunction = progressFunction;
    }

    public ReduceType execute(Iterator<InputType> inputReader, NSMapFunction<InputType, MapType> map, ReduceType initialValue, NSReduceFunction<MapType, ReduceType> reduce) {
        if (this.isShutdown()) {
            throw new IllegalStateException("execute called on already shutdown NanoScheduler");
        }
        if (inputReader == null) {
            throw new IllegalArgumentException("inputReader cannot be null");
        }
        if (map == null) {
            throw new IllegalArgumentException("map function cannot be null");
        }
        if (reduce == null) {
            throw new IllegalArgumentException("reduce function cannot be null");
        }
        this.myNSRuntimeProfile.outsideSchedulerTimer.stop();
        ReduceType result = this.getnThreads() == 1 ? this.executeSingleThreaded(inputReader, map, initialValue, reduce) : this.executeMultiThreaded(inputReader, map, initialValue, reduce);
        this.myNSRuntimeProfile.outsideSchedulerTimer.restart();
        return result;
    }

    @Requires(value={"inputReader != null", "map != null", "reduce != null"})
    private ReduceType executeSingleThreaded(Iterator<InputType> inputReader, NSMapFunction<InputType, MapType> map, ReduceType initialValue, NSReduceFunction<MapType, ReduceType> reduce) {
        ReduceType sum = initialValue;
        int i = 0;
        while (true) {
            this.myNSRuntimeProfile.inputTimer.restart();
            if (!inputReader.hasNext()) break;
            InputType input = inputReader.next();
            this.myNSRuntimeProfile.inputTimer.stop();
            this.myNSRuntimeProfile.mapTimer.restart();
            long preMapTime = this.myNSRuntimeProfile.mapTimer.currentTimeNano();
            MapType mapValue = map.apply(input);
            this.myNSRuntimeProfile.mapTimer.stop();
            if (i++ % this.bufferSize == 0 && this.progressFunction != null) {
                this.progressFunction.progress(input);
            }
            this.myNSRuntimeProfile.reduceTimer.restart();
            sum = reduce.apply(mapValue, sum);
            this.myNSRuntimeProfile.reduceTimer.stop();
        }
        this.myNSRuntimeProfile.inputTimer.stop();
        return sum;
    }

    @Requires(value={"inputReader != null", "map != null", "reduce != null"})
    private ReduceType executeMultiThreaded(Iterator<InputType> inputReader, NSMapFunction<InputType, MapType> map, ReduceType initialValue, NSReduceFunction<MapType, ReduceType> reduce) {
        this.debugPrint("Executing nanoScheduler", new Object[0]);
        MasterJob masterJob = new MasterJob(inputReader, map, initialValue, reduce);
        Future reduceResult = this.masterExecutor.submit(masterJob);
        while (true) {
            this.handleErrors();
            try {
                Object result = reduceResult.get(100L, TimeUnit.MILLISECONDS);
                this.handleErrors();
                return (ReduceType)result;
            }
            catch (TimeoutException ex) {
                continue;
            }
            catch (InterruptedException ex) {
                this.errorTracker.notifyOfError(ex);
                continue;
            }
            catch (ExecutionException ex) {
                this.errorTracker.notifyOfError(ex);
                continue;
            }
            break;
        }
    }

    private void handleErrors() {
        if (this.errorTracker.hasAnErrorOccurred()) {
            this.masterExecutor.shutdownNow();
            this.mapExecutor.shutdownNow();
            this.inputExecutor.shutdownNow();
            this.errorTracker.throwErrorIfPending();
        }
    }

    private class MapReduceJob
    implements Runnable {
        final BlockingQueue<InputProducer.InputValue> inputQueue;
        final PriorityBlockingQueue<MapResult<MapType>> mapResultQueue;
        final NSMapFunction<InputType, MapType> map;
        final Reducer<MapType, ReduceType> reducer;

        private MapReduceJob(BlockingQueue<InputProducer.InputValue> inputQueue, PriorityBlockingQueue<MapResult<MapType>> mapResultQueue, NSMapFunction<InputType, MapType> map, Reducer<MapType, ReduceType> reducer) {
            this.inputQueue = inputQueue;
            this.mapResultQueue = mapResultQueue;
            this.map = map;
            this.reducer = reducer;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                MapResult result;
                InputProducer.InputValue inputWrapper = this.inputQueue.take();
                int jobID = inputWrapper.getId();
                if (!inputWrapper.isEOFMarker()) {
                    Object input = inputWrapper.getValue();
                    ((NanoScheduler)NanoScheduler.this).myNSRuntimeProfile.mapTimer.restart();
                    long preMapTime = ((NanoScheduler)NanoScheduler.this).myNSRuntimeProfile.mapTimer.currentTimeNano();
                    Object mapValue = this.map.apply(input);
                    ((NanoScheduler)NanoScheduler.this).myNSRuntimeProfile.mapTimer.stop();
                    result = new MapResult(mapValue, jobID);
                    if (jobID % NanoScheduler.this.bufferSize == 0 && NanoScheduler.this.progressFunction != null) {
                        NanoScheduler.this.progressFunction.progress(input);
                    }
                } else {
                    this.inputQueue.put(inputWrapper.nextEOF());
                    result = new MapResult(jobID);
                }
                this.mapResultQueue.put(result);
                int nReduced = this.reducer.reduceAsMuchAsPossible(this.mapResultQueue);
            }
            catch (Exception ex) {
                NanoScheduler.this.errorTracker.notifyOfError(ex);
            }
            finally {
                NanoScheduler.this.runningMapJobSlots.release();
            }
        }
    }

    private class MasterJob
    implements Callable<ReduceType> {
        final Iterator<InputType> inputReader;
        final NSMapFunction<InputType, MapType> map;
        final ReduceType initialValue;
        final NSReduceFunction<MapType, ReduceType> reduce;

        private MasterJob(Iterator<InputType> inputReader, NSMapFunction<InputType, MapType> map, ReduceType initialValue, NSReduceFunction<MapType, ReduceType> reduce) {
            this.inputReader = inputReader;
            this.map = map;
            this.initialValue = initialValue;
            this.reduce = reduce;
        }

        @Override
        public ReduceType call() {
            LinkedBlockingDeque<InputProducer.InputValue> inputQueue = new LinkedBlockingDeque<InputProducer.InputValue>(NanoScheduler.this.bufferSize + 1);
            InputProducer inputProducer = new InputProducer(this.inputReader, NanoScheduler.this.errorTracker, ((NanoScheduler)NanoScheduler.this).myNSRuntimeProfile.inputTimer, inputQueue);
            NanoScheduler.this.inputExecutor.submit(inputProducer);
            PriorityBlockingQueue mapResultQueue = new PriorityBlockingQueue();
            Reducer reducer = new Reducer(this.reduce, NanoScheduler.this.errorTracker, ((NanoScheduler)NanoScheduler.this).myNSRuntimeProfile.reduceTimer, this.initialValue);
            try {
                int nSubmittedJobs = 0;
                while (this.continueToSubmitJobs(nSubmittedJobs, inputProducer)) {
                    NanoScheduler.this.runningMapJobSlots.acquire();
                    NanoScheduler.this.mapExecutor.submit(new MapReduceJob(inputQueue, mapResultQueue, this.map, reducer));
                    ++nSubmittedJobs;
                }
                reducer.setTotalJobCount(nSubmittedJobs);
                return this.waitForCompletion(inputProducer, reducer);
            }
            catch (Exception ex) {
                NanoScheduler.this.errorTracker.notifyOfError(ex);
                return this.initialValue;
            }
        }

        private ReduceType waitForCompletion(InputProducer<InputType> inputProducer, Reducer<MapType, ReduceType> reducer) throws InterruptedException {
            Object finalSum = reducer.waitForFinalReduce();
            inputProducer.waitForDone();
            NanoScheduler.this.runningMapJobSlots.acquire(NanoScheduler.this.bufferSize);
            NanoScheduler.this.runningMapJobSlots.release(NanoScheduler.this.bufferSize);
            return finalSum;
        }

        private boolean continueToSubmitJobs(int nJobsSubmitted, InputProducer<InputType> inputProducer) {
            int nReadItems = inputProducer.getNumInputValues();
            return nReadItems == -1 || nJobsSubmitted < nReadItems;
        }
    }
}

